//******************************************************************************
// file         : DRV_SQ7705_UART.c
// version      : V1.3 (2025/03/13)
// description  : UART related functions
// note         : UART related functions are gathered in this subroutine
//******************************************************************************
#include <stdlib.h>
#include "DRV_SQ7705_UART.h"

static UART_CALLBACK UART0_TX_CallBackFunc;
static UART_CALLBACK UART0_RX_CallBackFunc;

static UART_CALLBACK UART1_TX_CallBackFunc;
static UART_CALLBACK UART1_RX_CallBackFunc;

static UART_CALLBACK UART2_TX_CallBackFunc;
static UART_CALLBACK UART2_RX_CallBackFunc;

//******************************************************************************
// name         : DRV_UART0_Init()
// description  : UART0 initialization
// input param  : rxPin : assign UART RX pin 
//                        UART_RX_PIN_NONE    // not assign UART0_RX. 
//                        UART0_RX_PIN_P00    // P00 = UART0_RX.
//                        UART0_RX_PIN_P60    // P60 = UART0_RX.
//                        UART0_RX_PIN_P65    // P65 = UART0_RX.
//                txPin : assign UART TX pin 
//                        UART_TX_PIN_NONE    // not assign UART0_TX. 
//                        UART0_TX_PIN_P01    // P01 = UART0_TX.
//                        UART0_TX_PIN_P61    // P61 = UART0_TX.
//                        UART0_TX_PIN_P66    // P66 = UART0_TX.
//                parityCheck : assign parity check mode
//                        PARITY_NONE         // no parity.
//                        PARITY_ODD          // odd-numbered parity.
//                        PARITY_EVEN         // even-numbered parity.
//                stopBits : assign stop bit 
//                        STOP_1BIT           // Stop 1 bit.
//                        STOP_2BIT           // Stop 2 bits.
// retval       : Error code 
// note         : 
//******************************************************************************
int8_t DRV_UART0_Init(uint8_t rxPin, uint8_t txPin, uint8_t parityCheck, uint8_t stopBits)
{
    UART0_TX_CallBackFunc = NULL;
    UART0_RX_CallBackFunc = NULL;
    
    //====== UART0_RX pin initialization ======
    switch (rxPin) {
        case UART0_RX_PIN_P00:
            P0CFGCR = 0x82;
            break;
        case UART0_RX_PIN_P60:
            P6CFGCR = 0x82;
            break;
        case UART0_RX_PIN_P65:
            P6CFGCR = 0xD2;
            break;
        case UART_RX_PIN_NONE:
            break;
        default:
            return DRV_UART_RET_INVALID_PARAM;
    }

    //====== UART0_TX pin initialization ======
    switch (txPin) {
        case UART0_TX_PIN_P01:
            P0CFGCR = 0x92;
            break;
        case UART0_TX_PIN_P61:
            P6CFGCR = 0x92;
            break;
        case UART0_TX_PIN_P66:
            P6CFGCR = 0xE2;
            break;
        case UART_RX_PIN_NONE:
            break;
        default:
            return DRV_UART_RET_INVALID_PARAM;
    }

    //====== function enable ======
    PCKEN1_UART0 = 1;                     // UART0 Enable-- this must be enabled first, 
                                          // the relevant staging settings later to be usefu

    //====== parameter setting ======
    UART0CR1  = (UART0CR1 & ~PARITY_MASK) | parityCheck;  // set parity.

    UART0CR1_STOPBT = stopBits;           // set the bit length of the TX0 stop bit
    UART0CR2_STOPBR = stopBits;           // set the bit length of the RX0 stop bit.

    // TX pin output moed selection.
    UART0CR1_IRDASEL = TX_OUTPUT_UART;    // UART output.
    //UART0CR1_IRDASEL = TX_OUTPUT_IRDA;  // IrDA output.
    
    UART0CR1_BRG = 0;                     // use system clock as UART transfer base clock
    UART0CR2_DV = DIV_BY1;                // UART dividing frequency selection = fsysclk/1.

    return DRV_UART_RET_SUCCESS;
}

//******************************************************************************
// name         : DRV_UART0_DeInit
// description  : Deinitialize UART0.
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
int8_t DRV_UART0_DeInit(void)
{
    __asm("DI");                       // disable all maskable interrupts.

    DRV_UART_Stop(UART0);
    DRV_UART0_INT(DISABLE, UART_INT_EN_TX_RX, NULL, NULL);
    
    //--------------------------------------------------------------------------
    // Reset UART0 setting as defualt.
    //--------------------------------------------------------------------------
    UART0CR1 = 0x00;
    UART0CR2 = 0x00;
    UART0DR = 0x00;

    PCKEN1_UART0 = DISABLE;            // Disable UART0 peripheral clock.

    UART0_TX_CallBackFunc = NULL;
    UART0_RX_CallBackFunc = NULL;
    __asm("EI");                       // enable all maskable interrupts.

    return DRV_UART_RET_SUCCESS;
}

//******************************************************************************
// name         : DRV_UART0_INT
// description  : enable/disable UART0 interrupt
// input param  : enFlag : enable/disable interrupt
//                        ENABLE              // Enable operation.
//                        DISABLE             // Disable operation.
//                selection : setting target is Tx/Rx or both 
//                        UART_INT_EN_TX      // Enable/Disable UARTx TX interrupt only.
//                        UART_INT_EN_RX      // Enable/Disable UARTx RX interrupt only.
//                        UART_INT_EN_TX_RX   // Enable/Disable UARTx TX & RX interrupt.
//
// retval       : DRV_RET_SUCCESS             // Execution success. 
//                DRV_RET_INVALID_PARAM       // Invalid input parameters.
// note         : 
// example      : 
//   1. if only enable RX0 interrupt DRV_UART0_INT(ENABLE, UART_INT_EN_RX);
//   2. if only disable RX0 interrupt DRV_UART0_INT(DISABLE, UART_INT_EN_RX);
//   3. if both enable TX0 andf RX0 interrupts DRV_UART0_INT(ENABLE, UART_INT_EN_TX_RX);
//******************************************************************************
int8_t DRV_UART0_INT(uint8_t enFlag, uint8_t selection, UART_CALLBACK rxCB, UART_CALLBACK txCB)
{
    if ((enFlag != ENABLE) && (enFlag != DISABLE)) {
        return DRV_UART_RET_INVALID_PARAM;
    }

    __asm("DI");                       // disable all maskable interrupts.

    switch (selection) {
        case UART_INT_EN_TX:
            IER_UART0_TX = enFlag;
            if ((enFlag == ENABLE) && (txCB != NULL)) {
                UART0_TX_CallBackFunc = txCB;
            } else {
                UART0_TX_CallBackFunc = NULL;
            }
            break;
        case UART_INT_EN_RX:
            IER_UART0_RX = enFlag;
            if ((enFlag == ENABLE) && (rxCB != NULL)) {
                UART0_RX_CallBackFunc = rxCB;
            } else {
                UART0_RX_CallBackFunc = NULL;
            }            
            break;         
        case UART_INT_EN_TX_RX:
            IER_UART0_TX = enFlag;
            IER_UART0_RX = enFlag;
            if (enFlag == ENABLE) {
                if (txCB != NULL) {
                    UART0_TX_CallBackFunc = txCB;
                } else {
                    UART0_TX_CallBackFunc = NULL;
                }

                if (rxCB != NULL) {
                    UART0_RX_CallBackFunc = rxCB;
                } else {
                    UART0_RX_CallBackFunc = NULL;
                }
            }
            break;
        default:
            return DRV_UART_RET_INVALID_PARAM;
    }

    __asm("EI");                       // enable all maskable interrupts.

    return DRV_UART_RET_SUCCESS;
}

//******************************************************************************
// name         : DRV_UART1_Init()
// description  : UART1 initialization
// input param  : rxPin : assign UART1 RX pin 
//                        UART_RX_PIN_NONE    // not assign UART1_RX. 
//                        UART1_RX_PIN_P07    // P07 = UART1_RX.
//                        UART1_RX_PIN_P10    // P10 = UART1_RX.
//                        UART1_RX_PIN_P50    // P50 = UART1_RX.
//                txPin : assign UART1 TX pin 
//                        UART_TX_PIN_NONE    // not assign UART1_TX. 
//                        UART1_TX_PIN_P11    // P11 = UART1_TX.
//                        UART1_TX_PIN_P34    // P34 = UART1_TX.
//                        UART1_TX_PIN_P51    // P51 = UART1_TX.
//                parityCheck : assign parity check mode
//                        PARITY_NONE         // no parity.
//                        PARITY_ODD          // odd-numbered parity.
//                        PARITY_EVEN         // even-numbered parity.
//                stopBits : assign stop bit 
//                        STOP_1BIT           // Stop 1 bit.
//                        STOP_2BIT           // Stop 2 bits.
// retval       : Error code 
// note         : 
//******************************************************************************
int8_t DRV_UART1_Init(uint8_t rxPin, uint8_t txPin, uint8_t parityCheck, uint8_t stopBits)
{
    UART1_TX_CallBackFunc = NULL;
    UART1_RX_CallBackFunc = NULL;

    //====== UART1_RX pin initialization ======
    switch (rxPin) {
        case UART1_RX_PIN_P07:
            P0CFGCR = 0x80 | 0x70 | 0x02;
            break;
        case UART1_RX_PIN_P10:
            P1CFGCR = 0x80 | 0x00 | 0x02;
            break;
        case UART1_RX_PIN_P50:
            P5CFGCR = 0x80 | 0x00 | 0x02;
            break;    
        case UART_RX_PIN_NONE:
            break;
        default:
            return DRV_UART_RET_INVALID_PARAM;
    }

    //====== UART1_TX pin initialization ======
    switch (txPin) {
        case UART1_TX_PIN_P11:
            P1CFGCR = 0x80 | 0x10 | 0x02;
            break;
        case UART1_TX_PIN_P34:
            P3CFGCR = 0x80 | 0x40 | 0x02;
            break;
        case UART1_TX_PIN_P51:
            P5CFGCR = 0x80 | 0x10 | 0x02;
            break;
        case UART_TX_PIN_NONE:
            break;
        default:
            return DRV_UART_RET_INVALID_PARAM;
    }

    //====== function enable ======
    PCKEN1_UART1 = 1;                     // UART1 Enable-- this must be enabled first, 
                                          // the relevant staging settings later to be useful.

    //====== parameter setting ======
    UART1CR1  = (UART1CR1 & ~PARITY_MASK) | parityCheck;    // set parity.

    UART1CR1_STOPBT = stopBits;           // set the bit length of the TX1 stop bit
    UART1CR2_STOPBR = stopBits;           // set the bit length of the RX1 stop bit.

    // TX pin output mode selection.
    UART1CR1_IRDASEL = TX_OUTPUT_UART;    // UART output.
    //UART0CR1_IRDASEL = TX_OUTPUT_IRDA;  // IrDA output.

    UART1CR1_BRG = 0;                     // use system clock as UART transfer base clock
    UART1CR2_DV = DIV_BY1;                // UART dividing frequency selection = fsysclk/1.

    return DRV_UART_RET_SUCCESS;
}

//******************************************************************************
// name         : DRV_UART1_DeInit
// description  : Deinitialize UART1.
// param        : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_UART1_DeInit(void)
{
    __asm("DI");                    // disable all maskable interrupts.

    DRV_UART_Stop(UART1);
    DRV_UART1_INT(DISABLE, UART_INT_EN_TX_RX, NULL, NULL);
    
    //--------------------------------------------------------------------------
    // Reset UART1 setting as defualt.
    //--------------------------------------------------------------------------
    UART1CR1 = 0x00;
    UART1CR2 = 0x00;
    UART1DR  = 0x00;

    PCKEN1_UART1 = DISABLE;         // Disable UART1 peripheral clock.

    UART1_TX_CallBackFunc = NULL;
    UART1_RX_CallBackFunc = NULL;
    __asm("EI");                    // enable all maskable interrupts.
}

//******************************************************************************
// name         : DRV_UART1_INT
// description  : enable/disable UART1 interrupt
// input param  : enFlag : enable/disable interrupt
//                        ENABLE              // Enable operation.
//                        DISABLE             // Disable operation.
//                selection : setting target is Tx/Rx or both 
//                        UART_INT_EN_TX      // Enable/Disable UARTx TX interrupt only.
//                        UART_INT_EN_RX      // Enable/Disable UARTx RX interrupt only.
//                        UART_INT_EN_TX_RX   // Enable/Disable TX & RX interrupt.
//
// retval       : DRV_RET_SUCCESS             // Execution success. 
//                DRV_RET_INVALID_PARAM       // Invalid input parameters.
// note         : 
// example      : 
//   1. if only enable RX1 interrupt DRV_UART1_INT(ENABLE, UART_INT_EN_RX);
//   2. if only disable RX1 interrupt DRV_UART1_INT(DISABLE, UART_INT_EN_RX);
//   3. if both enable TX1 andf RX0 interrupts DRV_UART1_INT(ENABLE, UART_INT_EN_TX_RX);
//******************************************************************************
int8_t DRV_UART1_INT(uint8_t enFlag, uint8_t selection, UART_CALLBACK rxCB, UART_CALLBACK txCB)
{
    if ((enFlag != ENABLE) && (enFlag != DISABLE)) {
        return DRV_UART_RET_INVALID_PARAM;
    }

    __asm("DI");                        // disable all maskable interrupts.

    switch (selection) {        
        case UART_INT_EN_TX:
            IER_UART1_TX = enFlag;
            if ((enFlag == ENABLE) && (txCB != NULL)) {
                UART1_TX_CallBackFunc = txCB;
            } else {
                UART1_TX_CallBackFunc = NULL;
            }
            break;
        case UART_INT_EN_RX:
            IER_UART1_RX = enFlag;
            if ((enFlag == ENABLE) && (rxCB != NULL)) {
                UART1_RX_CallBackFunc = rxCB;
            } else {
                UART1_RX_CallBackFunc = NULL;
            } 
            break;         
        case UART_INT_EN_TX_RX:
            IER_UART1_TX = enFlag;
            IER_UART1_RX = enFlag;
            if (enFlag == ENABLE) {
                if (txCB != NULL) {
                    UART1_TX_CallBackFunc = txCB;
                } else {
                    UART1_TX_CallBackFunc = NULL;
                }

                if (rxCB != NULL) {
                    UART1_RX_CallBackFunc = rxCB;
                } else {
                    UART1_RX_CallBackFunc = NULL;
                }
            }
            break;

        default:
            return DRV_UART_RET_INVALID_PARAM;
    }

    __asm("EI");                        // enable all maskable interrupts.

     return DRV_UART_RET_SUCCESS;
}

//******************************************************************************
// name         : DRV_UART2_Init()
// description  : UART2 initialization
// input param  : rxPin : assign UART2 RX pin 
//                        UART_RX_PIN_NONE    // not assign UART2_RX. 
//                        UART2_RX_PIN_P36    // P36 = UART2_RX.
//                        UART2_RX_PIN_P74    // P74 = UART2_RX.
//                txPin : assign UART2 TX pin 
//                        UART_TX_PIN_NONE    // not assign UART2_TX. 
//                        UART2_TX_PIN_P37    // P37 = UART2_TX.
//                        UART2_TX_PIN_P75    // P75 = UART2_TX.
//                parityCheck : assign parity check mode
//                        PARITY_NONE         // no parity.
//                        PARITY_ODD          // odd-numbered parity.
//                        PARITY_EVEN         // even-numbered parity.
//                stopBits : assign stop bit 
//                        STOP_1BIT           // Stop 1 bit.
//                        STOP_2BIT           // Stop 2 bits.
// retval       : DRV_RET_SUCCESS             // Execution success. 
//                DRV_RET_INVALID_PARAM       // Invalid input parameters.
// note         : 
//******************************************************************************
int8_t DRV_UART2_Init(uint8_t rxPin, uint8_t txPin, uint8_t parityCheck, uint8_t stopBits)
{
    UART2_TX_CallBackFunc = NULL;
    UART2_RX_CallBackFunc = NULL;

    //====== UART2_RX pin initialization ======
    switch (rxPin) {
        case UART2_RX_PIN_P36:
            P3CFGCR = 0x80 | 0x60 | 0x02;
            break;
        case UART2_RX_PIN_P74:
            P7CFGCR = 0x80 | 0x40 | 0x02;
            break;
        case UART_RX_PIN_NONE:
            break;
        default:
            return DRV_UART_RET_INVALID_PARAM;
    }

    //====== UART1_TX pin initialization ======
    switch (txPin) {
        case UART2_TX_PIN_P37:
            P3CFGCR = 0x80 | 0x70 | 0x02;
            break;
        case UART2_TX_PIN_P75:
            P7CFGCR = 0x80 | 0x50 | 0x02;
            break;
        case UART_TX_PIN_NONE:
            break;
        default:
            return DRV_UART_RET_INVALID_PARAM;
    }

    //====== function enable ======
    PCKEN1_UART2 = 1;                     // UART2 Enable-- this must be enabled first, 
                                          // the relevant staging settings later to be useful.

    //====== parameter setting ======
    UART2CR1  = (UART2CR1 & ~PARITY_MASK) | parityCheck;  // set parity.

    UART2CR1_STOPBT = stopBits;           // set the bit length of the TX2 stop bit
    UART2CR2_STOPBR = stopBits;           // set the bit length of the RX2 stop bit.

    // TX pin output mode selection.
    UART2CR1_IRDASEL = TX_OUTPUT_UART;    // UART output.
    //UART2CR1_IRDASEL = TX_OUTPUT_IRDA;  // IrDA output.

    UART2CR1_BRG = 0;                     // use system clock as UART transfer base clock
    UART2CR2_DV = DIV_BY1;                // UART dividing frequency selection = fsysclk/1.

    return DRV_UART_RET_SUCCESS;
}

//******************************************************************************
// name         : DRV_UART2_DeInit
// description  : Deinitialize UART2.
// param        : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_UART2_DeInit(void)
{
    __asm("DI");                    // disable all maskable interrupts.

    DRV_UART_Stop(UART2);
    DRV_UART2_INT(DISABLE, UART_INT_EN_TX_RX, NULL, NULL);
    
    //--------------------------------------------------------------------------
    // Reset UART1 setting as defualt.
    //--------------------------------------------------------------------------
    UART2CR1 = 0x00;
    UART2CR2 = 0x00;
    UART2DR  = 0x00;

    PCKEN1_UART2 = DISABLE;         // Disable UART2 peripheral clock.

    UART2_TX_CallBackFunc = NULL;
    UART2_RX_CallBackFunc = NULL;
    __asm("EI");                    // enable all maskable interrupts.
}

//******************************************************************************
// name         : DRV_UART2_INT
// description  : enable/disable UART2 interrupt
// input param  : enFlag : enable/disable interrupt
//                        ENABLE              // Enable operation.
//                        DISABLE             // Disable operation.
//                selection : setting target is Tx/Rx or both 
//                        UART_INT_EN_TX      // Enable/Disable UARTx TX interrupt only.
//                        UART_INT_EN_RX      // Enable/Disable UARTx RX interrupt only.
//                        UART_INT_EN_TX_RX   // Enable/Disable TX & RX interrupt.
//
// retval       : DRV_RET_SUCCESS             // Execution success. 
//                DRV_RET_INVALID_PARAM       // Invalid input parameters.
// note         : 
// example      : 
//   1. if only enable RX2 interrupt DRV_UART2_INT(ENABLE, UART_INT_EN_RX);
//   2. if only disable RX2 interrupt DRV_UART2_INT(DISABLE, UART_INT_EN_RX);
//   3. if both enable TX1 andf RX0 interrupts DRV_UART2_INT(ENABLE, UART_INT_EN_TX_RX);
//******************************************************************************
int8_t DRV_UART2_INT(uint8_t enFlag, uint8_t selection, UART_CALLBACK rxCB, UART_CALLBACK txCB)
{
    if ((enFlag != ENABLE) && (enFlag != DISABLE)) {
        return DRV_UART_RET_INVALID_PARAM;
    }

    __asm("DI");                        // disable all maskable interrupts.

    switch (selection) {        
        case UART_INT_EN_TX:
            IER_UART2_TX = enFlag;
            if ((enFlag == ENABLE) && (txCB != NULL)) {
                UART2_TX_CallBackFunc = txCB;
            } else {
                UART2_TX_CallBackFunc = NULL;
            }
            break;
        case UART_INT_EN_RX:
            IER_UART2_RX = enFlag;
            if ((enFlag == ENABLE) && (rxCB != NULL)) {
                UART2_RX_CallBackFunc = rxCB;
            } else {
                UART2_RX_CallBackFunc = NULL;
            } 
            break;         
        case UART_INT_EN_TX_RX:
            IER_UART2_TX = enFlag;
            IER_UART2_RX = enFlag;
            if (enFlag == ENABLE) {
                if (txCB != NULL) {
                    UART2_TX_CallBackFunc = txCB;
                } else {
                    UART2_TX_CallBackFunc = NULL;
                }

                if (rxCB != NULL) {
                    UART2_RX_CallBackFunc = rxCB;
                } else {
                    UART2_RX_CallBackFunc = NULL;
                }
            }
            break;

        default:
            return DRV_UART_RET_INVALID_PARAM;
    }

    __asm("EI");                        // enable all maskable interrupts.

     return DRV_UART_RET_SUCCESS;
}

//******************************************************************************
// name         : DRV_UART_SetBaudRate()
// description  : set UART/0/1/2 port Baud rate
// input param  : uartSel : select UART port 
//                        UART0, UART1, UART2
//                baudSel : select baud-rate 
//                        Operat24M_Baud128000
//                        Operat24M_Baud115200
//                        Operat24M_Baud76800
//                        Operat24M_Baud62500
//                        Operat24M_Baud57600
//                        Operat24M_Baud38400
//                        Operat24M_Baud19200
//                        Operat24M_Baud9600
// retval       : none
// note         : the above baudSel is in the case of UART0/1/2 operating at 24MHz 
//                for other frequencies, please refer to the DRV_SQ7705_UART.h for the corresponding baudSel settings
//******************************************************************************
void DRV_UART_SetBaudRate(uint8_t uartSel, uint16_t baudSel)
{
    if(uartSel == UART0){
        UART0CR2_RTSEL = LOW_BYTE(baudSel); // select the number of RT clocks
        
        // select Baud rate
        DRV_UART_Stop(uartSel);             // before changing UART0DR, set UART0CR1_RXE and UART0CR1_TXE to "0"
        UART0DR = HIGH_BYTE(baudSel);       // Set baud-rate
    } else if(uartSel == UART1){
        UART1CR2_RTSEL = LOW_BYTE(baudSel); // select the number of RT clocks
        
        // select Baud rate
        DRV_UART_Stop(uartSel);             // before changing UART0DR, set UART0CR1_RXE and UART0CR1_TXE to "0"
        UART1DR = HIGH_BYTE(baudSel);       // Set baud-rate
    } else if(uartSel == UART2){
        UART2CR2_RTSEL = LOW_BYTE(baudSel); // select the number of RT clocks
        
        // select Baud rate
        DRV_UART_Stop(uartSel);             // before changing UART0DR, set UART0CR1_RXE and UART0CR1_TXE to "0"
        UART2DR = HIGH_BYTE(baudSel);       // Set baud-rate
    }
}

//******************************************************************************
// name         : DRV_UART_Start
// description  : Start UART0/1/2.
// input param  : uartSel : select UART port 
//                          UART0, UART1, UART2
// retval       : 
// note         : Enable UART0/1/2  TX and RX.
//******************************************************************************
void DRV_UART_Start(uint8_t uartSel)
{
    if(uartSel == UART0){
        //UART0CR1_TXE = ENABLE;            // enable TX0.
        //UART0CR1_RXE = ENABLE;            // enable RX0.
        UART0CR1 |=  0xC0;                  // enable TX0 and RX0.
    } else if(uartSel == UART1){
        //UART1CR1_TXE = ENABLE;            // enable TX1
        //UART1CR1_RXE = ENABLE;            // enable RX1
        UART1CR1 |= 0xC0;                   // enable TX1, RX1
    } else if(uartSel == UART2){
        //UART2CR1_TXE = ENABLE;            // enable TX2.
        //UART2CR1_RXE = ENABLE;            // enable RX2.
        UART2CR1 |=  0xC0;                  // enable TX2 and RX2.
    }
}

//******************************************************************************
// name         : DRV_UART_Stop
// description  : stop(disable) UART0/1/2 UART0/1/2  TX and RX.
// input param  : uartSel : select UART port 
//                          UART0, UART1, UART2
// retval       : 
// note         : 
//******************************************************************************
void DRV_UART_Stop(uint8_t uartSel)
{
    if(uartSel == UART0){
        //UART0CR1_TXE = DISABLE;            // disable TX0.
        //UART0CR1_RXE = DISABLE;            // disable RX0.
        UART0CR1 &= ~0xC0;                   // disable TX0 and RX0.
    } else if(uartSel == UART1){
        //UART1CR1_TXE = DISABLE;            // disable TX1
        //UART1CR1_RXE = DISABLE;            // disable RX1
        UART1CR1 &= ~0xC0;                   // disable TX1 and RX1.
    } else if(uartSel == UART2){
        //UART2CR1_TXE = DISABLE;            // disable TX2.
        //UART2CR1_RXE = DISABLE;            // disable RX2.
        UART2CR1 &= ~0xC0;                   // disable TX2 and RX2.
    }
}

//******************************************************************************
// name         : DRV_UART_Send_OneByte()
// description  : TX transmits a single byte
// input param  : uartSel : select output UART port 
//                          UART0, UART1, UART2
//                data    : data to be transmitted by TX
// retval       : none
// note         : 
//******************************************************************************
void DRV_UART_Send_OneByte(uint8_t uartSel, uint8_t data)
{
    if(uartSel == UART0){
        while(UART0SR_TBFL){   // wait for the send buffer to be empty before writing the data to be sent again
            CLR_WDT;           // clear the watchdog timer 
        }
        TD0BUF = data;         // write the data to be sent
        while(UART0SR_TBSY){
            CLR_WDT;           // clear the watchdog timer 
        }
    } else if(uartSel == UART1){
        while(UART1SR_TBFL){   // wait for the send buffer to be empty before writing the data to be sent again
            CLR_WDT;           // clear the watchdog timer 
        }
        TD1BUF = data;         // write the data to be sent
        while(UART1SR_TBSY){
            CLR_WDT;           // clear the watchdog timer 
        }
    } else if(uartSel == UART2){
        while(UART2SR_TBFL){   // wait for the send buffer to be empty before writing the data to be sent again
            CLR_WDT;           // clear the watchdog timer 
        }
        TD2BUF = data;         // write the data to be sent
        while(UART2SR_TBSY){
            CLR_WDT;           // clear the watchdog timer 
        }
    }
}

//******************************************************************************
// name         : DRV_UART_Send_Bytes()
// description  : TX transmits multiple bytes
// input param  : uartSel : Output UARTT port 
//                          UART0, UART1, UART2
//                sentBuf : the bytes to be transferred are first placed in the sentBuf[]
//                len     : the length of output data
// retval       : 
// note         : when the data to be transferred is more than 1 Byte, the user can set the length of data
// example      : To transfer 12 bytes, the bytes to be transferred are first placed in sentBuf[] 
//                and transfer in the order sentBuf[0]~sentBuf[11]
//                DRV_UART_Send_Bytes(UART1,sentBuf, 12);          
//******************************************************************************
void DRV_UART_Send_Bytes(uint8_t uartSel, uint8_t* sentBuf, uint8_t len)
{
     uint8_t i;

    for(i=0; i<len; i++){
        if(uartSel == UART0){
            while(UART0SR_TBFL){
                CLR_WDT;                  // clear the watchdog timer 
            }
            TD0BUF = sentBuf[i];          // write the data to be sent
            while(UART0SR_TBSY){
                CLR_WDT;                  // clear the watchdog timer 
            }    
        }        
        else if(uartSel == UART1){
            while(UART1SR_TBFL){
                CLR_WDT;                  // clear the watchdog timer 
            }
            TD1BUF = sentBuf[i];          // write the data to be sent
            while(UART1SR_TBSY){
                CLR_WDT;                  // clear the watchdog timer 
            }                        
        }        
        else if(uartSel == UART2){
            while(UART2SR_TBFL){
                CLR_WDT;                  // clear the watchdog timer 
            }
            TD2BUF = sentBuf[i];          // write the data to be sent
            while(UART2SR_TBSY){
                CLR_WDT;                  // clear the watchdog timer 
            }
        }
    }
}

//******************************************************************************
// name         : DRV_UART_Recv_OneByte()
// description  : TX receive a single byte
// input param  : uartSel : select input UART port 
//                          UART0, UART1, UART2
// retval       : UART received single byte data
// note         : 
//******************************************************************************
uint8_t DRV_UART_Recv_OneByte(uint8_t uartSel)
{
    if(uartSel == UART0)
        return RD0BUF;
    else if(uartSel == UART1)
        return RD1BUF;
    else if(uartSel == UART2)
        return RD2BUF;
    return 0;
}

//******************************************************************************
// name         : DRV_RX0_IRQ()
// description  : RX0 interrupt service routine
// note         :
//******************************************************************************
void __interrupt DRV_RX0_IRQ(void)
{
    uint8_t dummyReadByte;

    if (UART0_RX_CallBackFunc != NULL){
        UART0_RX_CallBackFunc();
    } else {
        dummyReadByte = RD0BUF;
    }
}

//******************************************************************************
// name         : DRV_RX1_IRQ()
// description  : RX1 interrupt service routine
// note         :
//******************************************************************************
void __interrupt DRV_RX1_IRQ(void)
{
    uint8_t dummyReadByte;
    __asm("NOP");
    
    if (UART1_RX_CallBackFunc != NULL) {
        UART1_RX_CallBackFunc();
    } else {
        dummyReadByte = RD1BUF;
    }
}

//******************************************************************************
// name         : DRV_RX2_IRQ()
// description  : RX2 interrupt service routine
// note         :
//******************************************************************************
void __interrupt DRV_RX2_IRQ(void)
{
    uint8_t dummyReadByte;
    __asm("NOP");
    
    if (UART2_RX_CallBackFunc != NULL) {
        UART2_RX_CallBackFunc();
    } else {
        dummyReadByte = RD2BUF;
    }
}

//******************************************************************************
// name         : DRV_TX0_IRQ()
// description  : TX0 interrupt service routine
// note         :
//******************************************************************************
void __interrupt DRV_TX0_IRQ(void)
{
    if (UART0_TX_CallBackFunc != NULL) {
        UART0_TX_CallBackFunc();
    }
}

//******************************************************************************
// name         : DRV_TX1_IRQ()
// description  : TX1 interrupt service routine
// note         :
//******************************************************************************
void __interrupt DRV_TX1_IRQ(void)
{
    if (UART1_TX_CallBackFunc != NULL) {
        UART1_TX_CallBackFunc();
    }
}

//******************************************************************************
// name         : DRV_TX2_IRQ()
// description  : TX2 interrupt service routine
// note         :
//******************************************************************************
void __interrupt DRV_TX2_IRQ(void)
{
    if (UART2_TX_CallBackFunc != NULL) {
        UART2_TX_CallBackFunc();
    }
}
