//******************************************************************************
// file         : main.c
// version      : V1.3  2023/11/20
// description  : main program
// note         : this example program is a demostration for UART TX sending data to RX 
//                within a single SQ7705 chip
//                1. TX1(P1.1) connects to RX1(P1.0)
//                2. UART1 configuration : Baud rate to 115200, stop bit to 1 bit, no parity
//******************************************************************************
#ifndef IO_MEM
#define IO_MEM
#endif
#include "ADM_main.h"
#include "DRV_SQ7705_SYSCLK.h"
#include "DRV_SQ7705_UART.h"

/* --------------------------- global variables ----------------------------- */
uint8_t rx1RecvCount = 0;              // UART1 RX receive Bytes number
uint8_t sentBuf[]={0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x76, 0x15, 0x5A, 0xA5};
uint8_t receiveBuf[]={0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};

/* ------------------- function prototype declaration ----------------------- */
void CallBack_RX1(void);
void CallBack_TX1(void);

/* ---------------------------- main program -------------------------------- */
void ADM_Main()
{
    DRV_Clock_Init(Clk_Source_SYS, 1);        // Initial system clock (PLL 48MHz/2 = 24MHz)

    /* ------------------------ UART1 initialization ------------------------ */
    DRV_UART1_Init(UART1_RX_PIN_P10, UART1_TX_PIN_P11, PARITY_NONE, STOP_1BIT);
    DRV_UART1_INT(ENABLE, UART_INT_EN_TX_RX, CallBack_RX1, CallBack_TX1);
    DRV_UART_SetBaudRate(UART1, Operat24M_Baud115200);  // set UART1 Baud rate to 115200
    DRV_UART_Start(UART1);

    /* ----------------- UART1 Tx send and Rx receive test ------------------ */
    // UART1 Tx send one-byte and UART1 Rx receive this byte
    rx1RecvCount = 0;
    DRV_UART_Send_OneByte(UART1, 0x55);
    while(rx1RecvCount == 0) ;                // wait UART1 Rx receive data
    __ASM("NOP");                             // add break-point here, receiveBuf[0]  will be 0x55

    // UART1 Tx send one-byte and UART1 Rx receive this byte
    rx1RecvCount = 0;
    DRV_UART_Send_OneByte(UART1, 0xAA);
    while(rx1RecvCount == 0);
    __ASM("NOP");                             // add break-point here, receiveBuf[0]  will be 0xAA    

   // UART1 Tx send 12-byte and UART1 Rx receive these bytes
    rx1RecvCount = 0;
    DRV_UART_Send_Bytes(UART1, sentBuf, 12);  // TX transmits 21 bytes data
    while(rx1RecvCount < 12);                 // wait UART1 Rx receive 12-bytes data
    __ASM("NOP");                             // add break-point here, and watch receiveBuf[] buffer content    

    while(1){
        __ASM("NOP");
    }
}

//******************************************************************************
// name         : CallBack_RX1()
// description  : UART1_RX interrupt callback function
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void CallBack_RX1(void)
{
    receiveBuf[rx1RecvCount] = DRV_UART_Recv_OneByte(UART1);   // Form UART1 receive one byte data    
    rx1RecvCount++;
}

//******************************************************************************
// name         : CallBack_TX1()
// description  : UART1_TX interrupt callback function
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void CallBack_TX1(void)
{
    __ASM("NOP");
}

/* ----------------------- interrupt service routine ------------------------ */
void __interrupt OnlyReti(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n OnlyRetn(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n WDT_IRQ(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n SWI_IRQ(void)               // software reset interrupt service routine
{
    SWRST = 0x5a;
    SWRST = 0xa5;
    SWRST = 0xc3;
    SWRST = 0x3c;
}

void __interrupt IntADM_Req(void)
{

}

/* ---------------------- end of interrupt service routine ------------------ */


/* ----------------------- SQ7705 interrupt vector -------------------------- */
#pragma section const ADM_INT_VECTOR
void * const IntTbl1[] = {
    ADM_STARTUP,             /* 0x0000 : RESET               */    //non-maskable
    SWI_IRQ,                 /* 0x0002 : SWI_IRQ/INTUNDEF    */    //non-maskable
    (void *)0xffff,          /* 0x0004 : Reserved            */    
    WDT_IRQ,                 /* 0x0006 : WDT_IRQ             */    //non-maskable interrupt source:watchdog timer
    OnlyReti,                /* 0x0008 : TAMPER_IRQ          */    //interrupt source:tamper detection
    OnlyReti,                /* 0x000A : CFD_IRQ             */    //interrupt source:clock faildetection
    OnlyReti,                /* 0x000C : HVD_IRQ             */
    IntADM_Req,              /* 0x000E : U2A_IRQ/IntADM_Req  */    //interrupt source:user mode to admin mode
    OnlyReti,                /* 0x0010 : RTC_IRQ             */    //interrupt source:real-time clock timer
    OnlyReti,                /* 0x0012 : TMRA00_IRQ          */    //interrupt source:TMRA00 8-bit timer
    OnlyReti,                /* 0x0014 : TMRA01_IRQ          */    //interrupt source:TMRA01 8-bit timer
    OnlyReti,                /* 0x0016 : TMRA10_IRQ          */    //interrupt source:TMRA10 8-bit timer
    OnlyReti,                /* 0x0018 : TMRA11_IRQ          */    //interrupt source:TMRA11 8-bit timer
    OnlyReti,                /* 0x001A : TMRA20_IRQ          */
    OnlyReti,                /* 0x001C : TMRA21_IRQ          */
    OnlyReti,                /* 0x001E : TBT_IRQ             */
    OnlyReti,                /* 0x0020 : RX0_IRQ             */    //interrupt source:UART0 RX0
    OnlyReti,                /* 0x0022 : TX0_IRQ             */    //interrupt source:UART0 TX0
    OnlyReti,                /* 0x0024 : I2C0_IRQ            */    //interrupt source:I2C0
    OnlyReti,                /* 0x0026 : SPI0_IRQ            */    //interrupt source:SPI0
    OnlyReti,                /* 0x0028 : EXT0_IRQ            */    //external interrupt0
    OnlyReti,                /* 0x002A : EXT1_IRQ            */    //external interrupt1
    OnlyReti,                /* 0x002C : EXT2_IRQ            */    //external interrupt2
    OnlyReti,                /* 0x002E : EXT3_IRQ            */    //external interrupt3
    OnlyReti,                /* 0x0030 : ADC_IRQ             */    //interrupt source:ADC
    OnlyReti,                /* 0x0032 : LVD0_IRQ            */
    OnlyReti,                /* 0x0034 : LVD1_IRQ            */
    OnlyReti,                /* 0x0036 : Reserved            */
    OnlyReti,                /* 0x0038 : Reserved            */
    OnlyReti,                /* 0x003A : FMC_IRQ             */    //interrupt source:flash memory controller
    OnlyReti,                /* 0x003C : DMACH0_IRQ          */
    OnlyReti,                /* 0x003E : DMACH1_IRQ          */
    OnlyReti,                /* 0x0040 : DMACH2_IRQ          */
    OnlyReti,                /* 0x0042 : DMACH3_IRQ          */
    OnlyReti,                /* 0x0044 : Reserved            */
    OnlyReti,                /* 0x0046 : Reserved            */
    OnlyReti,                /* 0x0048 : Reserved            */
    OnlyReti,                /* 0x004A : Reserved            */
    OnlyReti,                /* 0x004C : Reserved            */
    OnlyReti,                /* 0x004E : DIC_IRQ             */    //interrupt source:DIC
    OnlyReti,                /* 0x0050 : LEUART_RX_IRQ       */
    OnlyReti,                /* 0x0052 : LEUART_TX_IRQ       */
    OnlyReti,                /* 0x0054 : TRNG_IRQ            */
    OnlyReti,                /* 0x0056 : EXT4_IRQ            */    //external interrupt4
    OnlyReti,                /* 0x0058 : EXT5_IRQ            */    //external interrupt5
    OnlyReti,                /* 0x005A : EXT6_IRQ            */    //external interrupt6
    OnlyReti,                /* 0x005C : EXT7_IRQ            */    //external interrupt7
    OnlyReti,                /* 0x005E : EEMC_IRQ            */    //interrupt source:EEPROM controller
    OnlyReti,                /* 0x0060 : TCA0_IRQ            */    //interrupt source:TCA0 16-bit timer
    OnlyReti,                /* 0x0062 : TCA1_IRQ            */    //interrupt source:TCA1 16-bit timer
    OnlyReti,                /* 0x0064 : STM0_IRQ            */
    OnlyReti,                /* 0x0066 : STM1_IRQ            */
    OnlyReti,                /* 0x0068 : AES_IRQ             */
    OnlyReti,                /* 0x006A : SHA_IRQ             */
    DRV_RX1_IRQ,             /* 0x006C : RX1_IRQ             */    //interrupt source:UART1 RX1
    DRV_TX1_IRQ,             /* 0x006E : TX1_IRQ             */    //interrupt source:UART1 TX1
    OnlyReti,                /* 0x0070 : I2C1_IRQ            */    //interrupt source:I2C1
    OnlyReti,                /* 0x0072 : SPI1_IRQ            */    //interrupt source:SPI1
    OnlyReti,                /* 0x0074 : RX2_IRQ             */    //interrupt source:UART2 RX2
    OnlyReti,                /* 0x0076 : TX2_IRQ             */    //interrupt source:UART2 TX2
    OnlyReti                 /* 0x0078 : TCA2_IRQ            */    //interrupt source:TCA2 16-bit timer
};
#pragma section const
/* -------------------- end of SQ7705 interrupt vector ---------------------- */

