//******************************************************************************
// file         : DRV_SQ7705_Tamper.c
// version      : V1.3 2025/03/13
// description  : Tamper related functions
// note         : DRV_TamperPinInit for TAMPERPIN0 TAMPERPIN1
//                DRV_TamperMonitorInit for
//                typedef enum {
//                    TAMPER_HVD,  // High voltage
//                    TAMPER_MTO,  // Circuit open
//                    TAMPER_MTS   // Circuit short
//                } TAMPER_TYPE;
//******************************************************************************
#include "DRV_SQ7705_Tamper.h"

/*flags for tamper*/
uint8_t tamperPin0 = 0;
uint8_t tamperPin1 = 0;
uint8_t tamperHvd  = 0;
uint8_t tamperMo   = 0;
uint8_t tamperMts  = 0;

//******************************************************************************
// name         : DRV_TamperPinInit()
// description  : TAMPERPIN initialization
// input param  : - tPin : TAMPIN1 or TIMPIN0
//                - selGPIO : GPIO for this pin
//                - trigger : selects the interrupt request generating condition
//                  RISING_TRIGGER = 0, an interrupt request is generated at the rising edge
//                  FALLING_TRIGGER = 1, an interrupt request is generated at the falling edgee
//                  RIS_FAL_TRIGGER = 2, an interrupt request is generated at the both edges
//                - sample : sets the noise canceller sampling interval
//                - act : action when tamper happen
// output param : 
// retval       : 
// note         : Valid tPin, selGPIO, parameter
//                example : TAMPPIN0_P00 means  tamper PIN0 usage GPIO P0,0
//                #define TAMPPIN0_P00  TAMPERPIN0,0
//                #define TAMPPIN0_P10  TAMPERPIN0,1
//                #define TAMPPIN0_P20  TAMPERPIN0,2
//                #define TAMPPIN0_P50  TAMPERPIN0,5
//                #define TAMPPIN0_P60  TAMPERPIN0,6
//                #define TAMPPIN0_P70  TAMPERPIN0,7
//                #define TAMPPIN1_P04  TAMPERPIN1,0
//                #define TAMPPIN1_P14  TAMPERPIN1,1
//                #define TAMPPIN1_P24  TAMPERPIN1,2
//                #define TAMPPIN1_P44  TAMPERPIN1,4
//                #define TAMPPIN1_P54  TAMPERPIN1,5
//                #define TAMPPIN1_P64  TAMPERPIN1,6
//                #define TAMPPIN1_P74  TAMPERPIN1,7
//******************************************************************************
void DRV_TamperPinInit(TAMPER_PIN tPin, uint8_t selGPIO, TAMPER_TRIGGER trigger, TAMPER_NC_SAMPLE sample, TAMPER_RSP_ACT act)
{
    __asm("DI");

    if (tPin == TAMPERPIN0) {
        PCKEN4_TAMPIN0 = 1;                    // enable TAMPERPIN0
        /*Trigger PIN and trigger condiction*/
        TAMPPINCR0_PINSEL = selGPIO;           // select gpio
        TAMPPINCR0_TAMPERES = trigger;         // trigger type
        TAMPPINCR0_TAMPERNC = sample;          // sample clock
        /*Set tamper respons action command*/
        TAMPERDR = 0Xc3;
        TAMPERDR = 0X3c;
        switch (act) {
            case TAMPER_RSP_ACT_NONE:
                TAMPERRSP1_TPIN0RSP = 0;
                break;

            case TAMPER_RSP_ACT_INT:
                TAMPERRSP1_TPIN0RSP = 2;
                break;

            case TAMPER_RSP_ACT_CLEAR_MSG:
                TAMPERRSP1_TPIN0RSP = 3;
                break;

            default:
                break;
        }

    } else {
        PCKEN4_TAMPIN1 = 1;
        /*Trigger PIN and trigger condiction*/
        TAMPPINCR1_PINSEL = selGPIO;           // select gpio
        TAMPPINCR1_TAMPERES = trigger;         // trigger type
        TAMPPINCR1_TAMPERNC = sample;          // sample clock
        /*Set tamper respons action command*/
        TAMPERDR = 0Xc3;
        TAMPERDR = 0X3c;

        switch (act) {
            case TAMPER_RSP_ACT_NONE:
                TAMPERRSP1_TPIN1RSP = 0;
                break;

            case TAMPER_RSP_ACT_INT:
                TAMPERRSP1_TPIN1RSP = 2;
                break;

            case TAMPER_RSP_ACT_CLEAR_MSG:
                TAMPERRSP1_TPIN1RSP = 3;
                break;

            default:
                break;
        }
    }

    __asm("EI");
}

//******************************************************************************
// name         : DRV_TamperPinDeInit()
// description  : Deinit TAMPERPIN
// input param  : - tPin : tamper pin
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_TamperPinDeInit(TAMPER_PIN tPin)
{
    __asm("DI");

    if (tPin == TAMPERPIN0) {
        PCKEN4_TAMPIN0 = 0;

    } else {
        PCKEN4_TAMPIN1 = 0;
    }

    __asm("EI");
}

//******************************************************************************
// name         : DRV_TamperMonitorInit()
// description  : Tamper monitor init
// input param  : - type : tamper type
//                - act  : action when tamper happen
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_TamperMonitorInit(TAMPER_TYPE type, TAMPER_RSP_ACT act)
{
    __asm("DI");
    //TAMPERCR_TAMPERIE = 1;
    TAMPERDR = 0Xc3;
    TAMPERDR = 0X3c;

    switch (type) {
        case TAMPER_HVD:          // High voltage
            TAMPERCR_HVDEN = 1;
            HVDCR = 1;
            TAMPERDR = 0Xc3;
            TAMPERDR = 0X3c;

            switch (act) {
                case TAMPER_RSP_ACT_NONE:
                    TAMPERRSP_HVDRSP = 0;
                    break;

                case TAMPER_RSP_ACT_INT:
                    TAMPERRSP_HVDRSP = 2;
                    break;

                case TAMPER_RSP_ACT_CLEAR_MSG:
                    TAMPERRSP_HVDRSP = 3;
                    break;

                default:
                    break;
            }

            break;

        case TAMPER_MTO:          // Circuit open
            TAMPERCR_MTOEN = 1;

            switch (act) {
                case TAMPER_RSP_ACT_NONE:
                    TAMPERRSP_MTPRS = 0;
                    break;

                case TAMPER_RSP_ACT_INT:
                    TAMPERRSP_MTPRS = 2;
                    break;

                case TAMPER_RSP_ACT_CLEAR_MSG:
                    TAMPERRSP_MTPRS = 3;
                    break;

                default:
                    break;
            }

            break;

        case TAMPER_MTS:          // Circuit short
            TAMPERCR_MTSEN = 1;

            switch (act) {
                case TAMPER_RSP_ACT_NONE:
                    TAMPERRSP_MTPRS = 0;
                    break;

                case TAMPER_RSP_ACT_INT:
                    TAMPERRSP_MTPRS = 2;
                    break;

                case TAMPER_RSP_ACT_CLEAR_MSG:
                    TAMPERRSP_MTPRS = 3;
                    break;

                default:
                    break;
            }

            break;
    }

    TAMPERCR_TAMPERIE = 1;
    __asm("EI");

}

//******************************************************************************
// name         : DRV_TamperMonitorDeInit()
// description  : Deinit tamper monitor
// input param  : - type : tamper type
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_TamperMonitorDeInit(TAMPER_TYPE type)
{
    switch (type) {
        case TAMPER_HVD:          // High voltage
            HVDCR = 0;
            TAMPERCR_HVDEN = 0;
            break;

        case TAMPER_MTO:          // Circuit open
            TAMPERCR_MTOEN = 0;
            break;

        case TAMPER_MTS:          // Circuit short
            TAMPERCR_MTSEN = 0;
            break;
    }
}

//******************************************************************************
// name         : DRV_TamperEnable()
// description  : Enable Tamper
// input param  :
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_TamperEnable()
{
    __asm("DI");
    TAMPERCR_TAMPERIE = 1;  // enable tamper
    IFR_TAMPER = 0;   // clear tamper interrupt flag
    IER_TAMPER = 1;   // enable Tamper interrupt
    __asm("EI");
}

//******************************************************************************
// name         : DRV_TamperDisable()
// description  : Disable Tamper
// input param  :
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_TamperDisable()
{
    __asm("DI");
    TAMPERCR_TAMPERIE = 0;  // Disable tamper
    IFR_TAMPER = 0;   // clear tamper interrupt flag
    IER_TAMPER = 0;   // disable Tamper interrupt
    __asm("EI");
}

//******************************************************************************
// name         : DRV_TAMPER_IRQ()
// description  : Tamper interrupt service routine
// input param  :
// retval       : 
// note         :
//******************************************************************************
void __interrupt DRV_TAMPER_IRQ(void)
{
    if (TAMPERSR_TPIN0 == 1) {
        tamperPin0 = 1;
    }

    if (TAMPERSR_TPIN1 == 1) {
        tamperPin1 = 1;
    }

    if (TAMPERSR_HVD == 1) {
        tamperHvd = 1;
    }

    if (TAMPERSR_MTO == 1) {
        tamperMo = 1;
    }

    if (TAMPERSR_MTS == 1) {
        tamperMts = 1;
    }
}