//******************************************************************************
// file         : DRV_SQ7705_SYSCLK.c
// version      : V1.2 2023/09/07
// description  : system clock related functions
// note         : system clock related functions are gathered in this subroutine
//******************************************************************************
#include "DRV_SQ7705_SYSCLK.h"

//******************************************************************************
// name         : DRV_Sys_Clock_Init()
// description  : Initial system PLL clock.
// input param  : clk_src: SYS, LIRC, HXTAL, LXTAL
//                sckpsc(select system clock prescaler): 0:/1, 1:/2, 2:/4, 3:/8, 4:/16, 5:/32, 6:/64, 7:/128
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_Clock_Init(enum Clk_Source clk_src, uint8_t sckpsc)
{
    switch(clk_src){
        case Clk_Source_SYS:
            PLLCR0_PLLEN = 1;                   // enable PLL
            while (PLLCR0_PLLRDY == 0);         // wait for PLL ready
            CLKCR0_SCKPSC = sckpsc;              
            CLKCR0_SCKSEL = 0x1;                //=001b, switch system clock source to PLL
            while (CLKST_SCKRDY == 0);          // wait for LIRC as system clock ready
            break;
        case Clk_Source_LIRC:
            CLKCR1_LIRCEN = 1; // enable LIRC
            //while (CLKST_LIRCRDY == 0); // wait for LIRC ready (CLKST_LIRCRDY has been removed-2023/9/7)
            __ASM("NOP");     
            __ASM("NOP");     
            __ASM("NOP");     
            __ASM("NOP");     
            __ASM("NOP");     
            __ASM("NOP");     
            CLKCR0_SCKPSC = sckpsc;              
            CLKCR0_SCKSEL = 0x2; //=010b, switch system clock source to LIRC
            while (CLKST_SCKSRC != 0x2);    // wait for LIRC as system clock ready
            break;
        case Clk_Source_HXTAL:
            CLKCR1_HXEN = 1; // enable external high crystal
            while (CLKST_HXRDY == 0); // wait for external high crystal ready
            CLKCR0_SCKPSC = sckpsc;              
            CLKCR0_SCKSEL = 0x3; //=011b, switch system clock source to external high crystal
            while (CLKST_SCKSRC != 0x3);    // wait for external high crystal as system clock ready
            break;
        case Clk_Source_LXTAL:
            CLKCR1_LXEN = 1; // enable external low crystal
            while (CLKST_LXRDY == 0); // wait for external low crystal ready
            CLKCR0_SCKPSC = sckpsc;              
            CLKCR0_SCKSEL = 0x4; //=100b, switch system clock source to external low crystal
            while (CLKST_SCKSRC != 0x4);    // wait for external low crystal as system clock ready
            break;
    }
}

//******************************************************************************
// name         : DRV_PLL_WUP()
// description  : set PLL warm-up counter
// input param  : clk_src: SYS, LIRC, HXTAL, LXTAL
//                wup_counter: 0: 1x (0.67ms), 1: 1.25x, 2:1.5x, 3:2x, 4:3x, 5:4x, others : reserved
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_Clock_WUP(enum Clk_Source clk_src, uint8_t wup_counter)
{
    switch(clk_src){
        case Clk_Source_SYS:
            PLLCR0_PLLWUP = wup_counter;
            break;
        case Clk_Source_LIRC:
            break;
        case Clk_Source_HXTAL:
            CLKWUP_HXWUP = wup_counter;
            break;
        case Clk_Source_LXTAL:
            CLKWUP_LXWUP = wup_counter;
            break;
    }
}
//******************************************************************************
// name         : DRV_PLL_REF()
// description  : select PLL reference clock source
// input param  : ref_select: 00:HIRC, 01:external high crystal 8MHz, others : reserved
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_PLL_REF(uint8_t ref_select)
{
    PLLCR0_PLLREF = ref_select;
}
//******************************************************************************
// name         : DRV_CM_Enable()
// description  : enable clock monitor
// input param  : clkcr2:
//                bit7: CMIE enable clock monitor 0 : disable, 1 : enable
//                bit6: LXCMEN enable external low crystal clock monitor 0 : disable, 1 : enable
//                bit5: HXCMEN enable external hjgh crystal clock monitor 0 : disable, 1 : enable
//                bit4-1: Reserved, = 0000b
//                bit0: SCKLF low frequency clock selection 0 : LIRC, 1 : LXTAL
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_CM_Enable(uint8_t clkcr2)
{
    CLKCR2 = clkcr2;
}

//******************************************************************************
// name         : DRV_CMCR_Set()
// description  : set clock monitor control register
// input param  : cmcr: clock monitor control register setting value, bit=0 : disable, 1 : enable
//                bit7: LF_GLITCH external low crystal glitch monitor
//                bit6: LF_SLOW external low crystal slow monitor (slower than 95%)
//                bit5: LF_FAST external low crystal fast monitor (faster than 105%)
//                bit4: LF_FAIL external low crystal fail monitor (slower than 50%)
//                bit3: HF_GLITCH external high crystal glitch monitor
//                bit2: HF_SLOW external high crystal slow monitor (slower than 95%)
//                bit1: HF_FAST external high crystal fast monitor (faster than 105%)
//                bit0: HF_FAIL external high crystal fail monitor (slower than 50%)
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_CMCR_Set(uint8_t cmcr)
{
    CMCR = cmcr;
}

//******************************************************************************
// name         : DRV_CMSR_Get()
// description  : get clock monitor control register status, bit=0 : not occured, 1 : occured
//                bit7: LF_GLITCH external low crystal glitch monitor
//                bit6: LF_SLOW external low crystal slow monitor (slower than 95%)
//                bit5: LF_FAST external low crystal fast monitor (faster than 105%)
//                bit4: LF_FAIL external low crystal fail monitor (slower than 50%)
//                bit3: HF_GLITCH external high crystal glitch monitor
//                bit2: HF_SLOW external high crystal slow monitor (slower than 95%)
//                bit1: HF_FAST external high crystal fast monitor (faster than 105%)
//                bit0: HF_FAIL external high crystal fail monitor (slower than 50%)
// input param  :
// output param :
// retval       : CMSR: enable clock monitor register status value
// note         :
//******************************************************************************
uint8_t DRV_CMSR_Get()
{
    return CMSR;
}

