//******************************************************************************
// file         : DRV_SQ7705_TMRA.h
// version      : V1.2 2023/11/20
// description  : TMRA header file
// note         : 
//******************************************************************************
#ifndef _DRV_SQ7705_TMRA_H__
#define _DRV_SQ7705_TMRA_H__

#include "TypeDefs.h"
#include "SQ7705_SFR.h"

extern uint8_t TMRA00_Int_Flag;      // flag for TMRA00 interrupt
extern uint8_t TMRA01_Int_Flag;      // flag for TMRA01 interrupt
extern uint8_t TMRA10_Int_Flag;      // flag for TMRA10 interrupt
extern uint8_t TMRA11_Int_Flag;      // flag for TMRA11 interrupt
extern uint8_t TMRA20_Int_Flag;      // flag for TMRA20 interrupt
extern uint8_t TMRA21_Int_Flag;      // flag for TMRA21 interrupt

typedef enum{               // select TMRAxx 8-bit
    TMRASEL_TMRA00 = 0,     // TMRA00
    TMRASEL_TMRA01 = 1,     // TMRA01
    TMRASEL_TMRA10 = 2,     // TMRA10
    TMRASEL_TMRA11 = 3,     // TMRA11
    TMRASEL_TMRA20 = 4,     // TMRA20
    TMRASEL_TMRA21 = 5      // TMRA21
} TMRA_SEL_8BIT;

typedef enum{               // select TMRAx 16-bit
    TMRASEL_TMRA0 = 0,      // TMRA0
    TMRASEL_TMRA1 = 1,      // TMRA1
    TMRASEL_TMRA2 = 2       // TMRA2
} TMRA_SEL_16BIT;

typedef enum{               // timer configuration
    TCFG_000 = 0,           // 8-bit mode (both timers operate independently)
    TCFG_001 = 1,           // Timer 0 and 1 outputs AND together and output to TMRA01 
    TCFG_010 = 2,           // 16-bit mode (both timers cascade together to form one single 16-bit timer)
    TCFG_100 = 4,           // pre-scale mode
    TCFG_101 = 5,           // complementary Mode
    TCFG_110 = 6            // center-aligned mode
} TMRA_TCFG;

typedef enum{               // timer overflow interrupt enable
    TMROVIE_DISABLE = 0,    // overflow interrupt enable: disable
    TMROVIE_ENABLE = 1      // overflow interrupt enable: enable
} TMRA_TMROVIE;

typedef enum{               // edge detect interrupt enable
    EDGEIE_DISABLE = 0,     // edge detect interrupt enable: disable
    EDGEIE_ENABLE = 1       // edge detect interrupt enable: enable
} TMRA_EDGEIE;

typedef enum{               // timer output enable
    OE_DISABLE = 0,         // timer output enable: disable
    OE_ENABLE = 1           // timer output enable: enable
} TMRA_OE;

typedef enum{               // edge select
    EDGE_RISING_EDGE = 0,   // edge select: rising edge
    EDGE_FALLING_EDGE = 2,  // edge select: falling edge
    EDGE_BOTH_EDGE = 3      // edge select: both edges
} TMRA_EDGE;

typedef enum{               // edge select (Pulse Measurement mode)
    EDGE_000 = 0,           // rising edge to falling edge (H pulse)
    EDGE_001 = 1,           // falling edgeto rising edge (L pulse)
    EDGE_010 = 2,           // rising edge to rising edge (period)
    EDGE_011 = 3,           // falling edge to falling edge (period)
    EDGE_100 = 4            // rising edge to falling edge (H pulse) rising edge to rising edge (period)
} TMRA_EDGE_PulseMeasure;

typedef enum{               // timer F/F control
    TFF_CLEAR = 0,          // timer F/F control: clear
    TFF_SET = 1             // timer F/F control: set
} TMRA_TFF;

typedef enum{               // double buffer control
    DBE_DISABLE = 0,        // disable double buffer
    DBE_ENABLE = 1          // enable double buffer
} TMRA_DBE;

typedef enum{        // timer counter source clock selection  
                     //       Normal/Sleep Mode      | Normal/Sleep Mode (slow clock)
                     // DV9CK=0       | DV9CK=1      | 
                     // -------------------------------------------------------
    TCK_000 = 0,     // fsysclk/2^11  | flclk/2^4    | flclk/2^4  
    TCK_001 = 1,     // fsysclk/2^10  | flclk/2^3    | flclk/2^3 
    TCK_010 = 2,     // fsysclk/2^8   | fsysclk/2^8  | -
    TCK_011 = 3,     // fsysclk/2^6   | fsysclk/2^6  | -
    TCK_100 = 4,     // fsysclk/2^4   | fsysclk/2^4  | -
    TCK_101 = 5,     // fsysclk/2^2   | fsysclk/2^2  | - 
    TCK_110 = 6,     // fsysclk/2     | fsysclk/2    | -
    TCK_111 = 7      // fsysclk       | fsysclk      | flclk
} TMRA_TCK;

typedef enum{               // timer operation mode
    TCM_TIMER = 0,          // timer mode 
    TCM_EVENT_COUNTER = 1,  // event counter mode
    TCM_CAPTURE = 3,        // capture mode
    TCM_PWM = 4,            // PWM mode
    TCM_PULSE_MEASURE = 5,  // pulse width meadurement mode
    TCM_PPG_OUTPUT = 6      // PPG output mode
} TMRA_TCM;

typedef enum{               //TMRA IO Pin
    TMRA00_P20 = 0x20,
    TMRA00_P36 = 0x36,
    TMRA00_P72 = 0x72,
    TMRA01_P21 = 0x21,
    TMRA01_P37 = 0x37,
    TMRA01_P74 = 0x74,
    TMRA10_P22 = 0x22,
    TMRA10_P60 = 0x60,
    TMRA10_P75 = 0x75,
    TMRA11_P00 = 0x00,
    TMRA11_P23 = 0x23,
    TMRA11_P61 = 0x61,
    TMRA20_P01 = 0x01,
    TMRA20_P24 = 0x24,
    TMRA20_P62 = 0x62,
    TMRA21_P02 = 0x02,
    TMRA21_P25 = 0x25,
    TMRA21_P63 = 0x63
} TMRA_IOPin;


typedef struct{                        // TMRA 8bit timer mode parameter struct
    TMRA_SEL_8BIT tmraSEL;             // TMRA 8bit Select TMRA00 ~ TMRA21
    TMRA_TCFG tmraTCFG;                // timer mode selection
    TMRA_DBE tmraDBE;                  // double buffer control
    TMRA_TCK tmraTCK;                  // source clock selection
    uint8_t period;                    // period
} TMRA_TIMER_8BIT_REG;

typedef struct{                        // TMRA 16-bit timer mode parameter struct
    TMRA_SEL_16BIT tmraSEL;            // TMRA 16bit Select TMRA0 ~ TMRA2
    TMRA_TCFG tmraTCFG;                // timer mode selection
    TMRA_DBE tmraDBE;                  // double buffer control
    TMRA_TCK tmraTCK;                  // source clock selection
    uint16_t period;                   // period
} TMRA_TIMER_16BIT_REG;

typedef struct{                        // TMRA 8bit PPG Output mode parameter struct
    TMRA_SEL_8BIT tmraSEL;             // TMRA 8-bit Select TMRA00 ~ TMRA21
    TMRA_TCFG tmraTCFG;                // timer mode selection
    TMRA_TFF tmraTFF;                  // timer F/F control
    TMRA_DBE tmraDBE;                  // double buffer control
    TMRA_TCK tmraTCK;                  // source clock selection
    TMRA_IOPin tmraOutputPin;          // output pin
    uint8_t period;                    // period
    uint8_t duty;                      // duty cycle
} TMRA_PPG_8BIT_REG;

typedef struct{                        // TMRA 16-bit PPG Output mode parameter struct
    TMRA_SEL_16BIT tmraSEL;            // TMRA 16-bit Select TMRA0 ~ TMRA2
    TMRA_TCFG tmraTCFG;                // timer mode selection
    TMRA_TFF tmraTFF;                  // timer F/F control
    TMRA_DBE tmraDBE;                  // double buffer control
    TMRA_TCK tmraTCK;                  // source clock selection
    TMRA_IOPin tmraOutputPin;          // output pin
    uint16_t period;                   // period
    uint16_t duty;                     // duty cycle
} TMRA_PPG_16BIT_REG;

typedef struct{                        // TMRA 8-bit Pulse Width Measure mode parameter struct
    TMRA_SEL_8BIT tmraSEL;             // TMRA 8-bit Select TMRA00 ~ TMRA21
    TMRA_TCFG tmraTCFG;                // timer mode selection
    TMRA_TMROVIE tmraOVIE;             // overflow interrupt enable
    TMRA_EDGEIE tmraEDGEIE;            // edge detect interrupt enable
    TMRA_OE tmraOE;                    // timer output enable
    TMRA_EDGE_PulseMeasure tmraEDGE;   // edge select (Pulse Measure mode)
    TMRA_TFF tmraTFF;                  // timer F/F control
    TMRA_DBE tmraDBE;                  // double buffer control
    TMRA_TCK tmraTCK;                  // source clock selection  
    TMRA_IOPin tmraInputPin;           // output pin
} TMRA_PULSE_MEASURE_8BIT_REG;

typedef struct{                        // TMRA 16-bit Pulse Width Measure mode parameter struct
    TMRA_SEL_16BIT tmraSEL;            // TMRA 16-bit Select TMRA0 ~ TMRA2
    TMRA_TCFG tmraTCFG;                // timer mode selection
    TMRA_TMROVIE tmraOVIE;             // overflow interrupt enable
    TMRA_EDGEIE tmraEDGEIE;            // edge detect interrupt enable
    TMRA_OE tmraOE;                    // timer output enable
    TMRA_EDGE_PulseMeasure tmraEDGE;   // edge select (Pulse Measure mode)
    TMRA_TFF tmraTFF;                  // timer F/F control
    TMRA_DBE tmraDBE;                  // double buffer control
    TMRA_TCK tmraTCK;                  // source clock selection 
    TMRA_IOPin tmraInputPin;           // output pin
} TMRA_PULSE_MEASURE_16BIT_REG;


//******************************************************************************
// name         : DRV_TMRA_Timer_8bit_Init()
// description  : TMRAx 8-bit timer mode initialization
// input param  : - tmraReg : TMRA 8-bit Timer Mode initialization data structure
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_Timer_8bit_Init(TMRA_TIMER_8BIT_REG tmraReg);

//******************************************************************************
// name         : DRV_TMRA_Timer_16bit_Init()
// description  : TMRAx 16-bit timer mode initialization
// input param  : - tmraReg : TMRA 16-bit Timer Mode initialization data structure
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_Timer_16bit_Init(TMRA_TIMER_16BIT_REG tmraReg);

//******************************************************************************
// name         : DRV_TMRA_PPG_8bit_Init()
// description  : TMRAx 8-bit PPG output mode initialization
// input param  : - tmraReg : TMRA 8-bit PPG output Mode initialization data structure
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_PPG_8bit_Init(TMRA_PPG_8BIT_REG tmraReg);

//******************************************************************************
// name         : DRV_TMRA_PPG_16bit_Init()
// description  : TMRAx 16-bit PPG output mode initialization
// input param  : - tmraReg : TMRA 16-bit PPG output Mode initialization data structure
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_PPG_16bit_Init(TMRA_PPG_16BIT_REG tmraReg);

//******************************************************************************
// name         : DRV_TMRA_PulseMeasure_8bit_Init()
// description  : TMRAx 8-bit pulse width measurement initialization
// input param  : - tmraReg : TMRA 8-bit pulse width measurement initialization data structure
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_PulseMeasure_8bit_Init(TMRA_PULSE_MEASURE_8BIT_REG tmraReg);

//******************************************************************************
// name         : DRV_TMRA_PulseMeasure_16bit_Init()
// description  : TMRAx 16-bit pulse width measurement initialization
// input param  : - tmraReg : TMRA 16-bit pulse width measurement initialization data structure
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_PulseMeasure_16bit_Init(TMRA_PULSE_MEASURE_16BIT_REG tmraReg);

//******************************************************************************
// name         : DRV_TMRA_Deinit()
// description  : TMRA deinitialization
// input param  : - tmraSel : TMRAx selection : TMRA0 ~ TMRA2
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_Deinit(TMRA_SEL_16BIT tmraSel);

//******************************************************************************
// name         : DRV_TMRA_8bit_Start()
// description  : Start TMRAx 8-bit
// input param  : - tmraSel : TMRAx selection : TMRA00 ~ TMRA21
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_8bit_Start(TMRA_SEL_8BIT tmraSel);

//******************************************************************************
// name         : DRV_TMRA_8bit_Stop()
// description  : Stop TMRAx 8-bit
// input param  : - tmraSel : TMRAx selection : TMRA00 ~ TMRA21
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_8bit_Stop(TMRA_SEL_8BIT tmraSel);

//******************************************************************************
// name         : DRV_TMRA_16bit_Start()
// description  : Start TMRAx 16-bit
// input param  : - tmraSel : TMRAx selection : TMRA0 ~ TMRA2
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_16bit_Start(TMRA_SEL_16BIT tmraSel);

//******************************************************************************
// name         : DRV_TMRA_16bit_Stop()
// description  : Stop TMRAx 16-bit
// input param  : - tmraSel : TMRAx selection : TMRA0 ~ TMRA2
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_16bit_Stop(TMRA_SEL_16BIT tmraSel);

//******************************************************************************
// name         : DRV_TMRA_8bit_ReadMeasure_PulseWidth()
// description  : Read single-edge captured value of TMRAx 8-bit 
// input param  : - tmraSel : TMRAx selection : TMRA00 ~ TMRA21
// output param : - readValue : single-edge captured value
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_8bit_ReadMeasure_PulseWidth(TMRA_SEL_8BIT tmraSel, uint8_t *readValue);

//******************************************************************************
// name         : DRV_TMRA_8bit_ReadMeasure_Period()
// description  : Read double-edge captured value of TMRAx 8-bit
// input param  : - tmraSel : TMRAx selection : TMRA00 ~ TMRA21
// output param : - readValue : double-edge captured value
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_8bit_ReadMeasure_Period(TMRA_SEL_8BIT tmraSel, uint8_t *readValue);

//******************************************************************************
// name         : DRV_TMRA_16bit_ReadMeasure_PulseWidth()
// description  : Read single-edge captured value of TMRAx 16-bit 
// input param  : - tmraSel : TMRAx selection : TMRA0 ~ TMRA2
// output param : - readValue : single-edge captured value
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_16bit_ReadMeasure_PulseWidth(TMRA_SEL_16BIT tmraSel, uint16_t *readValue);

//******************************************************************************
// name         : DRV_TMRA_16bit_ReadMeasure_Period()
// description  : Read double-edge captured value of TMRAx 16-bit
// input param  : - tmraSel : TMRAx selection : TMRA0 ~ TMRA2
// output param : - readValue : double-edge captured value
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_16bit_ReadMeasure_Period(TMRA_SEL_16BIT tmraSel, uint16_t *readValue);

//******************************************************************************
// name         : DRV_TMRA00_IRQ()
// description  : TMRA00 interrupt service routine
// input param  : 
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_TMRA00_IRQ(void);

//******************************************************************************
// name         : DRV_TMRA01_IRQ()
// description  : TMRA01 interrupt service routine
// input param  : 
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_TMRA01_IRQ(void);

//******************************************************************************
// name         : DRV_TMRA10_IRQ()
// description  : TMRA10 interrupt service routine
// input param  : 
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_TMRA10_IRQ(void);

//******************************************************************************
// name         : DRV_TMRA11_IRQ()
// description  : TMRA11 interrupt service routine
// input param  : 
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_TMRA11_IRQ(void);

//******************************************************************************
// name         : DRV_TMRA20_IRQ()
// description  : TMRA20 interrupt service routine
// input param  : 
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_TMRA20_IRQ(void);

//******************************************************************************
// name         : DRV_TMRA21_IRQ()
// description  : TMRA21 interrupt service routine
// input param  : 
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_TMRA21_IRQ(void);

#endif  // _DRV_SQ7705_TMRA_H__.


