//******************************************************************************
// file         : DRV_SQ7705_TMRA.c
// version      : V1.2 2023/11/20
// description  : TMRA related functions
// note         : TMRA related functions are gathered in this subroutine
//******************************************************************************
#include "DRV_SQ7705_TMRA.h"

extern void DRV_TMRA_OutputPin_Init(TMRA_IOPin outputPin);
extern void DRV_TMRA_InputPin_Init(TMRA_IOPin inputPin);

uint8_t TMRA00_Int_Flag;      // flag for TMRA00 interrupt
uint8_t TMRA01_Int_Flag;      // flag for TMRA01 interrupt
uint8_t TMRA10_Int_Flag;      // flag for TMRA10 interrupt
uint8_t TMRA11_Int_Flag;      // flag for TMRA11 interrupt
uint8_t TMRA20_Int_Flag;      // flag for TMRA20 interrupt
uint8_t TMRA21_Int_Flag;      // flag for TMRA21 interrupt

//******************************************************************************
// name         : DRV_TMRA_Timer_8bit_Init()
// description  : TMRAx 8-bit timer mode initialization
// input param  : - tmraReg : TMRA 8-bit Timer Mode initialization data structure
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_Timer_8bit_Init(TMRA_TIMER_8BIT_REG tmraReg)
{
    BIT_LIST_8* tmraxCFG = (BIT_LIST_8*)(0x0050 + ((tmraReg.tmraSEL >> 1) * 16));
    BIT_LIST_8* tmraxMOD = (BIT_LIST_8*)(0x0054 + ((tmraReg.tmraSEL >> 1) * 16) + (tmraReg.tmraSEL & 1));
    BIT_LIST_8* tmraxDR = (BIT_LIST_8*)(0x0058 + ((tmraReg.tmraSEL >> 1) * 16) + (tmraReg.tmraSEL & 1));

    PCKEN0 |= (1 << (4 + (tmraReg.tmraSEL >> 1)));  // enable TMRAx
    
    tmraxCFG->byte = tmraReg.tmraTCFG;              // TCFG
    tmraxMOD->b2_0.b2_0 = TCM_TIMER;                // TCM = Timer mode
    tmraxMOD->b5_3.b5_3 = tmraReg.tmraTCK;          // TCK 
    tmraxMOD->bit.b6 = tmraReg.tmraDBE;             // DBE
    tmraxDR->byte = tmraReg.period;                 // TMRAxDR
     
    __ASM("DI");
    IER1 |= (1 << (1 + tmraReg.tmraSEL));           // enable TMRAx interrupt
    __ASM("EI");
}

//******************************************************************************
// name         : DRV_TMRA_Timer_16bit_Init()
// description  : TMRAx 16-bit timer mode initialization
// input param  : - tmraReg : TMRA 16-bit Timer Mode initialization data structure
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_Timer_16bit_Init(TMRA_TIMER_16BIT_REG tmraReg)
{
    BIT_LIST_8* tmraxCFG = (BIT_LIST_8*)(0x0050 + (tmraReg.tmraSEL * 16));
    BIT_LIST_8* tmraxMOD = (BIT_LIST_8*)(0x0055 + (tmraReg.tmraSEL * 16));
    BIT_LIST_16* tmraxDR = (BIT_LIST_16*)(0x0058 + (tmraReg.tmraSEL * 16));

    PCKEN0 |= (1 << (4 + tmraReg.tmraSEL));         // enable TMRAx
    
    tmraxCFG->byte = tmraReg.tmraTCFG;              // TCFG
    tmraxMOD->b2_0.b2_0 = TCM_TIMER;                // TCM = Timer mode
    tmraxMOD->b5_3.b5_3 = tmraReg.tmraTCK;          // TCK 
    tmraxMOD->bit.b6 = tmraReg.tmraDBE;             // DBE
    tmraxDR->word = tmraReg.period;                 // TMRAxDR
     
    __ASM("DI");
    IER1 |= (1 << (2 + (tmraReg.tmraSEL * 2)));     // enable TMRAx interrupt
    __ASM("EI");
}

//******************************************************************************
// name         : DRV_TMRA_PPG_8bit_Init()
// description  : TMRAx 8-bit PPG output mode initialization
// input param  : - tmraReg : TMRA 8-bit PPG output Mode initialization data structure
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_PPG_8bit_Init(TMRA_PPG_8BIT_REG tmraReg)
{
    BIT_LIST_8* tmraxCFG = (BIT_LIST_8*)(0x0050 + ((tmraReg.tmraSEL >> 1) * 16));
    BIT_LIST_8* tmraxMOD = (BIT_LIST_8*)(0x0054 + ((tmraReg.tmraSEL >> 1) * 16) + (tmraReg.tmraSEL & 1));
    BIT_LIST_8* tmraxDR = (BIT_LIST_8*)(0x0058 + ((tmraReg.tmraSEL >> 1) * 16) + (tmraReg.tmraSEL & 1));
    BIT_LIST_8* tmraxPWM = (BIT_LIST_8*)(0x005A + ((tmraReg.tmraSEL >> 1) * 16) + (tmraReg.tmraSEL & 1));

    DRV_TMRA_OutputPin_Init(tmraReg.tmraOutputPin);   // Init TMRA Output Pin
    PCKEN0 |= (1 << (4 + (tmraReg.tmraSEL >> 1)));    // enable TMRAx
    
    tmraxCFG->byte = tmraReg.tmraTCFG;                // TCFG
    tmraxMOD->b2_0.b2_0 = TCM_PPG_OUTPUT;             // TCM = PPG Output mode
    tmraxMOD->b5_3.b5_3 = tmraReg.tmraTCK;            // TCK 
    tmraxMOD->bit.b6 = tmraReg.tmraDBE;               // DBE
    tmraxMOD->bit.b7 = tmraReg.tmraTFF;               // TFF
    tmraxDR->byte = tmraReg.period;                   // TMRAxDR
    tmraxPWM->byte = tmraReg.duty;                    // TMRAxPWM
    
    __ASM("DI");
    IER1 |= (1 << (1 + tmraReg.tmraSEL));             // enable TMRAx interrupt
    __ASM("EI");
}

//******************************************************************************
// name         : DRV_TMRA_PPG_16bit_Init()
// description  : TMRAx 16-bit PPG output mode initialization
// input param  : - tmraReg : TMRA 16-bit PPG output Mode initialization data structure
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_PPG_16bit_Init(TMRA_PPG_16BIT_REG tmraReg)
{
    BIT_LIST_8* tmraxCFG = (BIT_LIST_8*)(0x0050 + (tmraReg.tmraSEL * 16));
    BIT_LIST_8* tmraxMOD = (BIT_LIST_8*)(0x0055 + (tmraReg.tmraSEL * 16));
    BIT_LIST_16* tmraxDR = (BIT_LIST_16*)(0x0058 + (tmraReg.tmraSEL * 16));
    BIT_LIST_16* tmraxPWM = (BIT_LIST_16*)(0x005A + (tmraReg.tmraSEL * 16));

    DRV_TMRA_OutputPin_Init(tmraReg.tmraOutputPin);   // Init TMRA Output Pin
    PCKEN0 |= (1 << (4 + tmraReg.tmraSEL));           // enable TMRAx
    
    tmraxCFG->byte = tmraReg.tmraTCFG;                // TCFG
    tmraxMOD->b2_0.b2_0 = TCM_PPG_OUTPUT;             // TCM = PPG Output mode
    tmraxMOD->b5_3.b5_3 = tmraReg.tmraTCK;            // TCK 
    tmraxMOD->bit.b6 = tmraReg.tmraDBE;               // DBE
    tmraxMOD->bit.b7 = tmraReg.tmraTFF;               // TFF
    tmraxDR->word = tmraReg.period;                   // TMRAxDR
    tmraxPWM->word = tmraReg.duty;                    // TMRAxPWM
    
    __ASM("DI");
    IER1 |= (1 << (2 + (tmraReg.tmraSEL * 2)));       // enable TMRAx interrupt
    __ASM("EI");
}

//******************************************************************************
// name         : DRV_TMRA_PulseMeasure_8bit_Init()
// description  : TMRAx 8-bit pulse width measurement initialization
// input param  : - tmraReg : TMRA 8-bit pulse width measurement initialization data structure
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_PulseMeasure_8bit_Init(TMRA_PULSE_MEASURE_8BIT_REG tmraReg)
{
    BIT_LIST_8* tmraxCFG = (BIT_LIST_8*)(0x0050 + ((tmraReg.tmraSEL >> 1) * 16));
    BIT_LIST_8* tmraxCR = (BIT_LIST_8*)(0x0052 + ((tmraReg.tmraSEL >> 1) * 16) + (tmraReg.tmraSEL & 1));
    BIT_LIST_8* tmraxMOD = (BIT_LIST_8*)(0x0054 + ((tmraReg.tmraSEL >> 1) * 16) + (tmraReg.tmraSEL & 1));
    
    DRV_TMRA_InputPin_Init(tmraReg.tmraInputPin);     // Init TMRA Input Pin
    PCKEN0 |= (1 << (4 + (tmraReg.tmraSEL >> 1)));    // enable TMRAx
    
    tmraxCFG->byte = tmraReg.tmraTCFG;                // TCFG
    tmraxCR->b3_1.b3_1 = tmraReg.tmraEDGE;            // EDGE
    tmraxCR->bit.b4 = tmraReg.tmraOE;                 // OE
    tmraxCR->bit.b6 = tmraReg.tmraEDGEIE;             // EDGEIE
    tmraxCR->bit.b7 = tmraReg.tmraOVIE;               // OVIE
    tmraxMOD->b2_0.b2_0 = TCM_PULSE_MEASURE;          // TCM = Pulse Width Measurement Mode
    tmraxMOD->b5_3.b5_3 = tmraReg.tmraTCK;            // TCK 
    tmraxMOD->bit.b6 = tmraReg.tmraDBE;               // DBE
    tmraxMOD->bit.b7 = tmraReg.tmraTFF;               // TFF
    
    __ASM("DI");
    IER1 |= (1 << (1 + tmraReg.tmraSEL));             // enable TMRAx interrupt
    __ASM("EI");
}

//******************************************************************************
// name         : DRV_TMRA_PulseMeasure_16bit_Init()
// description  : TMRAx 16-bit pulse width measurement initialization
// input param  : - tmraReg : TMRA 16-bit pulse width measurement initialization data structure
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_PulseMeasure_16bit_Init(TMRA_PULSE_MEASURE_16BIT_REG tmraReg)
{
    BIT_LIST_8* tmraxCFG = (BIT_LIST_8*)(0x0050 + (tmraReg.tmraSEL * 16));
    BIT_LIST_8* tmraxCR = (BIT_LIST_8*)(0x0053 + (tmraReg.tmraSEL * 16));
    BIT_LIST_8* tmraxMOD = (BIT_LIST_8*)(0x0055 + (tmraReg.tmraSEL * 16));
    
    DRV_TMRA_InputPin_Init(tmraReg.tmraInputPin);     // Init TMRA Input Pin
    PCKEN0 |= (1 << (4 + tmraReg.tmraSEL));           // enable TMRAx
    
    tmraxCFG->byte = tmraReg.tmraTCFG;                // TCFG
    tmraxCR->b3_1.b3_1 = tmraReg.tmraEDGE;            // EDGE
    tmraxCR->bit.b4 = tmraReg.tmraOE;                 // OE
    tmraxCR->bit.b6 = tmraReg.tmraEDGEIE;             // EDGEIE
    tmraxCR->bit.b7 = tmraReg.tmraOVIE;               // OVIE
    tmraxMOD->b2_0.b2_0 = TCM_PULSE_MEASURE;          // TCM = Pulse Width Measurement Mode
    tmraxMOD->b5_3.b5_3 = tmraReg.tmraTCK;            // TCK 
    tmraxMOD->bit.b6 = tmraReg.tmraDBE;               // DBE
    tmraxMOD->bit.b7 = tmraReg.tmraTFF;               // TFF
    
    __ASM("DI");
    IER1 |= (1 << (2 + (tmraReg.tmraSEL * 2)));       // enable TMRAx interrupt
    __ASM("EI");
}

//******************************************************************************
// name         : DRV_TMRA_Deinit()
// description  : TMRA deinitialization
// input param  : - tmraSel : TMRAx selection : TMRA0 ~ TMRA2
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_Deinit(TMRA_SEL_16BIT tmraSel)
{
    if(tmraSel == TMRASEL_TMRA0){
        PRSTR0_TMRA0 = 1;
    } else if(tmraSel == TMRASEL_TMRA1){
        PRSTR1_TMRA1 = 1;
    } else if(tmraSel == TMRASEL_TMRA2){
        PRSTR1_TMRA2 = 1;
    }
}

//******************************************************************************
// name         : DRV_TMRA_OutputPin_Init()
// description  : TMRAx output pin initialization
// input param  : - outputPin : output pin selection
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_OutputPin_Init(TMRA_IOPin outputPin)
{
    uint8_t pA = ((outputPin >> 4) & 7);        // PA.B
    uint8_t pB = (outputPin & 7);               // PA.B
    uint8_t pxCFGCRvalue = (0x84 | (pB << 4));  // TMRA IO Base Value: 0x84 = 1xxx0100b    
    uint8_t* pxCFGCR = (uint8_t*)(0x0140 + (pA * 2));
    uint8_t* pxOE = (uint8_t*)(0x0100 + pA);
    
    *pxCFGCR = pxCFGCRvalue;                    // PxCFGCE
    *pxOE |= (1 << pB);                         // PxOE Set 1 (output mode)
}

//******************************************************************************
// name         : DRV_TMRA_InputPin_Init()
// description  : TMRAx input pin initialization
// input param  : - inputPin : input pin selection
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_InputPin_Init(TMRA_IOPin inputPin)
{
    uint8_t pA = ((inputPin >> 4) & 7);         // PA.B
    uint8_t pB = (inputPin & 7);                // PA.B
    uint8_t pxCFGCRvalue = (0x84 | (pB << 4));  // TMRA IO Base Value: 0x84 = 1xxx0100b
    uint8_t* pxCFGCR = (uint8_t*)(0x0140 + (pA * 2));
    uint8_t* pxOE = (uint8_t*)(0x0100 + pA);
    
    *pxCFGCR = pxCFGCRvalue;                    //PxCFGCE
    *pxOE &= ~(1 << pB);                        //PxOE Set 0 (input mode)
}

//******************************************************************************
// name         : DRV_TMRA_8bit_Start()
// description  : Start TMRAx 8-bit
// input param  : - tmraSel : TMRAx selection : TMRA00 ~ TMRA21
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_8bit_Start(TMRA_SEL_8BIT tmraSel)
{
    BIT_LIST_8* tmraxCR = (BIT_LIST_8*)(0x0052 + ((tmraSel >> 1) * 16) + (tmraSel & 1));
    tmraxCR->bit.b0 = 1;     // TMRAxCR RUN: start
    
}

//******************************************************************************
// name         : DRV_TMRA_8bit_Stop()
// description  : Stop TMRAx 8-bit
// input param  : - tmraSel : TMRAx selection : TMRA00 ~ TMRA21
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_8bit_Stop(TMRA_SEL_8BIT tmraSel)
{
    BIT_LIST_8* tmraxCR = (BIT_LIST_8*)(0x0052 + ((tmraSel >> 1) * 16) + (tmraSel & 1));
    tmraxCR->bit.b0 = 0;     // TMRAxCR RUN: stop and clear
}

//******************************************************************************
// name         : DRV_TMRA_16bit_Start()
// description  : Start TMRAx 16-bit
// input param  : - tmraSel : TMRAx selection : TMRA0 ~ TMRA2
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_16bit_Start(TMRA_SEL_16BIT tmraSel)
{
    BIT_LIST_8* tmraxCR = (BIT_LIST_8*)(0x0053 + (tmraSel * 16));
    tmraxCR->bit.b0 = 1;     // TMRAxCR RUN: start
    
}

//******************************************************************************
// name         : DRV_TMRA_16bit_Stop()
// description  : Stop TMRAx 16-bit
// input param  : - tmraSel : TMRAx selection : TMRA0 ~ TMRA2
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_16bit_Stop(TMRA_SEL_16BIT tmraSel)
{
    BIT_LIST_8* tmraxCR = (BIT_LIST_8*)(0x0053 + (tmraSel * 16));
    tmraxCR->bit.b0 = 0;     // TMRAxCR RUN: stop and clear
}

//******************************************************************************
// name         : DRV_TMRA_8bit_ReadMeasure_PulseWidth()
// description  : Read single-edge captured value of TMRAx 8-bit 
// input param  : - tmraSel : TMRAx selection : TMRA00 ~ TMRA21
// output param : - readValue : single-edge captured value
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_8bit_ReadMeasure_PulseWidth(TMRA_SEL_8BIT tmraSel, uint8_t *readValue)
{
    BIT_LIST_8* tmraxDR = (BIT_LIST_8*)(0x0058 + ((tmraSel >> 1) * 16) + (tmraSel & 1));
    *readValue = tmraxDR->byte;
}

//******************************************************************************
// name         : DRV_TMRA_8bit_ReadMeasure_Period()
// description  : Read double-edge captured value of TMRAx 8-bit
// input param  : - tmraSel : TMRAx selection : TMRA00 ~ TMRA21
// output param : - readValue : double-edge captured value
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_8bit_ReadMeasure_Period(TMRA_SEL_8BIT tmraSel, uint8_t *readValue)
{
    BIT_LIST_8* tmraxPWM = (BIT_LIST_8*)(0x005A + ((tmraSel >> 1) * 16) + (tmraSel & 1));
    *readValue = tmraxPWM->byte;
}

//******************************************************************************
// name         : DRV_TMRA_16bit_ReadMeasure_PulseWidth()
// description  : Read single-edge captured value of TMRAx 16-bit 
// input param  : - tmraSel : TMRAx selection : TMRA0 ~ TMRA2
// output param : - readValue : single-edge captured value
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_16bit_ReadMeasure_PulseWidth(TMRA_SEL_16BIT tmraSel, uint16_t *readValue)
{
    BIT_LIST_16* tmraxDR = (BIT_LIST_16*)(0x0058 + (tmraSel * 16));    
    *readValue = tmraxDR->word;
}

//******************************************************************************
// name         : DRV_TMRA_16bit_ReadMeasure_Period()
// description  : Read double-edge captured value of TMRAx 16-bit
// input param  : - tmraSel : TMRAx selection : TMRA0 ~ TMRA2
// output param : - readValue : double-edge captured value
// retval       : 
// note         : 
//******************************************************************************
void DRV_TMRA_16bit_ReadMeasure_Period(TMRA_SEL_16BIT tmraSel, uint16_t *readValue)
{
    BIT_LIST_16* tmraxPWM = (BIT_LIST_16*)(0x005A + (tmraSel * 16));
    *readValue = tmraxPWM->word;
}

//******************************************************************************
// name         : DRV_TMRA00_IRQ()
// description  : TMRA00 interrupt service routine
// input param  : 
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_TMRA00_IRQ(void)
{
    TMRA00_Int_Flag = 1;
}

//******************************************************************************
// name         : DRV_TMRA01_IRQ()
// description  : TMRA01 interrupt service routine
// input param  : 
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_TMRA01_IRQ(void)
{
    TMRA01_Int_Flag = 1;
}

//******************************************************************************
// name         : DRV_TMRA10_IRQ()
// description  : TMRA10 interrupt service routine
// input param  : 
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_TMRA10_IRQ(void)
{
    TMRA10_Int_Flag = 1;
}

//******************************************************************************
// name         : DRV_TMRA11_IRQ()
// description  : TMRA11 interrupt service routine
// input param  : 
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_TMRA11_IRQ(void)
{
    TMRA11_Int_Flag = 1;
}

//******************************************************************************
// name         : DRV_TMRA20_IRQ()
// description  : TMRA20 interrupt service routine
// input param  : 
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_TMRA20_IRQ(void)
{
    TMRA20_Int_Flag = 1;
}

//******************************************************************************
// name         : DRV_TMRA21_IRQ()
// description  : TMRA21 interrupt service routine
// input param  : 
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_TMRA21_IRQ(void)
{
    TMRA21_Int_Flag = 1;
}

