//******************************************************************************
// file         : DRV_SQ7705_TCA.h
// version      : V1.2 2023/08/22
// description  : TCA header file
// note         : 
//******************************************************************************
#ifndef _DRV_SQ7705_TCA_H__
#define _DRV_SQ7705_TCA_H__

#include "TypeDefs.h"
#include "SQ7705_SFR.h"

extern uint8_t TCA0_Int_Flag;      // flag for TCA0 interrupt
extern uint8_t TCA1_Int_Flag;      // flag for TCA1 interrupt
extern uint8_t TCA2_Int_Flag;      // flag for TCA2 interrupt

typedef enum{
    TCASEL_TCA0 = 0,               // TCA0
    TCASEL_TCA1 = 1,               // TCA1
    TCASEL_TCA2 = 2                // TCA2
} TCA_SEL;

typedef enum{
    TADBE_DISABLE = 0,             // disable double buffer
    TADBE_ENABLE = 1               // enable double buffer
} TCA_TADBE;

typedef enum{
    TATED_RISING_EDGE = 0,         // external trigger input selection: rising edge / H level
    TATED_FALLING_EDGE = 1         // external trigger input selection: falling edge / L level
} TCA_TATED;

typedef enum{
    TAMCAP_DOUBLE_EDGE = 0,        // pulse width measurement mode control: double edge capture
    TAMCAP_SINGLE_EDGE = 1         // pulse width measurement mode control: single edge capture
} TCA_TAMCAP;

typedef enum{
    TAMETT_TRIGGER_START = 0,      // external trigger timer mode contro: trigger start
    TAMETT_TRIGGER_START_STOP = 1  // external trigger timer mode contro: trigger start & stop
} TCA_TAMETT;

typedef enum{                      // timer counter source clock selection
                                   //         Normal/Sleep Mode          | Normal/Sleep Mode (slow clock)
                                   // TBTCR<DV9CK>=0  |  TBTCR<DV9CK>=1  |  
                                   // ------------------------------------------------------------------
    TACK_00 = 0,                   // fsysclk/2^10    |  flclk/2^3       | flclk/2^3  
    TACK_01 = 1,                   // fsysclk/2^6     |  flclk/2^6       | - 
    TACK_10 = 2,                   // fsysclk/2^2     |  flclk/2^2       | -
    TACK_11 = 3                    // fsysclk/2       |  flclk/2         | -
} TCA_TACK;

typedef enum{
    TAM_TIMER = 0,                 // timer mode 
    TAM_EVENT_COUNTER = 2,         // event counter mode
    TAM_PPG_OUTPUT = 3,            // PPG output mode
    TAM_EXTERNAL_TRIGER = 4,       // external trigger timer mode
    TAM_WINDOW = 5,                // window mode
    TAM_PULSE_MEASURE = 6          // pulse width measurement mode
} TCA_TAM;

typedef enum{                      // overflow interrupt control
    TAOVE_DISABLE = 0,             // generate no INTTCA0 interrupt request when the counter overflow occurs
    TAOVE_ENABLE = 1               // generate an INTTCA0 interrupt request when the counter overflow occurs
} TCA_TAOVE;

typedef enum{
    TATFF_CLEAR = 0,               // timer F/F control: clear
    TATFF_SET = 1                  // timer F/F control: set
} TCA_TATFF;

typedef enum{                      // noise canceller sampling interval setting
                                   // Normal/Sleep Mode   |  Normal/Sleep Mode (slow clock)
                                   // -----------------------------------------------------
    TANC_00 = 0,                   // no noise canceller  |  no noise canceller
    TANC_01 = 1,                   // fsysclk/2           |  -
    TANC_10 = 2,                   // fsysclk/2^2         |  -
    TANC_11 = 3                    // fsysclk/2^8         |  flclk/2
} TCA_TANC;

typedef enum{                      // auto capture function
    TACAP_DISABLE = 0,             // disable the auto capture
    TACAP_ENABLE = 1               // enable the auto capture
} TCA_TACAP;

typedef enum{
    TAMPPG_CONTINUOUS = 0,         // PPG output control: continuous
    TAMPPG_ONESHOT= 1              // PPG output control: one-shot
} TCA_TAMPPG;

typedef enum{                      //TCA IO Pin
    TCA0_P00 = 0x00,
    TCA0_P07 = 0x07,
    TCA0_P14 = 0x14,
    TCA0_P16 = 0x16,
    TCA0_P43 = 0x43,
    TCA0_P50 = 0x50,
    TCA0_P53 = 0x53,
    TCA0_P65 = 0x65,
    TCA1_P01 = 0x01,
    TCA1_P15 = 0x15,
    TCA1_P17 = 0x17,
    TCA1_P21 = 0x21,
    TCA1_P44 = 0x44,
    TCA1_P51 = 0x51,
    TCA1_P63 = 0x63,
    TCA1_P66 = 0x66,
    TCA1_P74 = 0x74,
    TCA2_P02 = 0x02,
    TCA2_P22 = 0x22,
    TCA2_P26 = 0x26,
    TCA2_P52 = 0x52,
    TCA2_P56 = 0x56,
    TCA2_P57 = 0x57,
    TCA2_P64 = 0x64
} TCA_IOPin;

typedef struct{                    // TCA timer mode parameter struct
    TCA_SEL tcaSEL;                // TCA Select TCA0 ~ TCA2
    TCA_TADBE tcaTADBE;            // TADBE double buffer control
    TCA_TACK tcaTACK;              // TACK TCAx source clock selection
    TCA_TACAP tcaTACAP;            // TACAP auto capture function
    uint16_t period;               // period
} TCA_TIMER_REG;

typedef struct{                    // TCA PPG output mode parameter struct
    TCA_SEL tcaSEL;                // TCA Select TCA0 ~ TCA2
    TCA_TADBE tcaTADBE;            // TADBE double buffer control
    TCA_TACK tcaTACK;              // TACK TCAx source clock selection
    TCA_TATFF tcaTATFF;            // TATFF timer F/F control
    TCA_TAMPPG tcaTAMPPG;          // TAMPPG PPG output control
    TCA_IOPin tcaOutputPin;        // output pin
    uint16_t period;               // period
    uint16_t duty;                 // duty cycle
} TCA_PPG_REG;

typedef struct{                    // TCA Pulse Width Measure mode parameter struct
    TCA_SEL tcaSEL;                // TCA Select TCA0 ~ TCA2
    TCA_TATED tcaTATED;            // TATED external trigger input selection
    TCA_TAMCAP tcaTAMCAP;          // TAMCAP pulse width measurement mode control
    TCA_TACK tcaTACK;              // TACK TCAx source clock selection
    TCA_TAOVE tcaTAOVE;            // TAOVE overflow interrupt control
    TCA_IOPin tcaInputPin;         // input pin
} TCA_PULSE_MEASURE_REG;


//******************************************************************************
// name         : DRV_TCA_Timer_Init()
// description  : TCA Timer Mode initialization
// input param  : - tcaReg : TCA Timer Mode initialization data structure
// output param :
// retval       : 
// note         : 
//******************************************************************************
void DRV_TCA_Timer_Init(TCA_TIMER_REG tcaReg);

//******************************************************************************
// name         : DRV_TCA_PPG_Init()
// description  : TCA PPG Output Mode initialization
// input param  : - tcaReg : TCA PPG Output Mode initialization data structure
// output param :
// retval       : 
// note         : 
//******************************************************************************
void DRV_TCA_PPG_Init(TCA_PPG_REG tcaReg);

//******************************************************************************
// name         : DRV_TCA_PulseMeasure_Init()
// description  : TCA PulseMeasure Mode initialization
// input param  : - tcaReg : TCA PulseMeasure Mode initialization data structure
// output param :
// retval       : 
// note         : 
//******************************************************************************
void DRV_TCA_PulseMeasure_Init(TCA_PULSE_MEASURE_REG tcaReg);

//******************************************************************************
// name         : DRV_TCA_Deinit()
// description  : TCA deinitialization
// input param  : - tcaSel : TCAx selection : TCA0 ~ TCA2
// output param :
// retval       : 
// note         : 
//******************************************************************************
void DRV_TCA_Deinit(TCA_SEL tcaSel);

//******************************************************************************
// name         : DRV_TCA_Start()
// description  : Start TCA
// input param  : - tcaSel : TCAx selection : TCA0 ~ TCA2
// output param :
// retval       : 
// note         : 
//******************************************************************************
void DRV_TCA_Start(TCA_SEL tcaSel);

//******************************************************************************
// name         : DRV_TCA_Stop()
// description  : Stop TCA
// input param  : - tcaSel : TCAx selection : TCA0 ~ TCA2
// output param :
// retval       : 
// note         : 
//******************************************************************************
void DRV_TCA_Stop(TCA_SEL tcaSel);

//******************************************************************************
// name         : DRV_TCA_ReadMeasure_PulseWidth()
// description  : Read single-edge captured value
// input param  : - tcaSel : TCAx selection : TCA0 ~ TCA2
// output param : - readValue : single-edge captured value
// retval       : 
// note         : 
//******************************************************************************
void DRV_TCA_ReadMeasure_PulseWidth(TCA_SEL tcaSel, uint16_t *readValue);

//******************************************************************************
// name         : DRV_TCA_ReadMeasure_Period()
// description  : Read double-edge captured value
// input param  : - tcaSel : TCAx selection : TCA0 ~ TCA2
// output param : - readValue : double-edge captured value
// retval       : 
// note         : 
//******************************************************************************
void DRV_TCA_ReadMeasure_Period(TCA_SEL tcaSel, uint16_t *readValue);

//******************************************************************************
// name         : DRV_TCA0_IRQ()
// description  : TCA0 interrupt service routine
// input param  : 
// output param :
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_TCA0_IRQ(void);

//******************************************************************************
// name         : DRV_TCA1_IRQ()
// description  : TCA1 interrupt service routine
// input param  : 
// output param :
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_TCA1_IRQ(void);

//******************************************************************************
// name         : DRV_TCA2_IRQ()
// description  : TCA2 interrupt service routine
// input param  : 
// output param :
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_TCA2_IRQ(void);

#endif  // _DRV_SQ7705_TCA_H__.


