//******************************************************************************
// file         : DRV_SQ7705_TCA.c
// version      : V1.2 2023/08/22
// description  : TCA related functions
// note         : TCA related functions are gathered in this subroutine
//******************************************************************************
#include "DRV_SQ7705_TCA.h"

extern void DRV_TCA_OutputPin_Init(TCA_IOPin outputPin);
extern void DRV_TCA_InputPin_Init(TCA_IOPin inputPin);

uint8_t TCA0_Int_Flag;                   // flag for TCA0 interrupt
uint8_t TCA1_Int_Flag;                   // flag for TCA1 interrupt
uint8_t TCA2_Int_Flag;                   // flag for TCA2 interrupt

//******************************************************************************
// name         : DRV_TCA_Timer_Init()
// description  : TCA Timer Mode initialization
// input param  : - tcaReg : TCA Timer Mode initialization data structure
// output param :
// retval       : 
// note         : 
//******************************************************************************
void DRV_TCA_Timer_Init(TCA_TIMER_REG tcaReg)
{
    BIT_LIST_8* tcaxCR = (BIT_LIST_8*)(0x0080 + (tcaReg.tcaSEL*8));
    BIT_LIST_8* tcaxMOD = (BIT_LIST_8*)(0x0081 + (tcaReg.tcaSEL*8));
    BIT_LIST_16* tcaxDRA = (BIT_LIST_16*)(0x0084 + (tcaReg.tcaSEL*8));

    PCKEN1 |= (1 << tcaReg.tcaSEL);         // enalbe TCAx
    
    tcaxCR->byte = 0;                       // Clear CR
    tcaxMOD->byte = 0;                      // Clear MOD
    tcaxDRA->word = 0;                      // Clear DRA

    tcaxMOD->b2_0.b2_0 = TAM_TIMER;         // TAM timer mode
    tcaxMOD->b4_3.b4_3 = tcaReg.tcaTACK;    // TACK
    tcaxMOD->bit.b7 = tcaReg.tcaTADBE;      // TADBE
    tcaxCR->bit.b0 = tcaReg.tcaTACAP;       // TACAP
    tcaxDRA->word = tcaReg.period;          // TAxDRA
     
    __ASM("DI");
    if(tcaReg.tcaSEL < TCASEL_TCA2)
    {
        IER6 |= (1 << (tcaReg.tcaSEL));     // enable TCA0, TCA1 interrupt
    }
    else if(tcaReg.tcaSEL == TCASEL_TCA2)
    {
        IER_TCA2 = 1;                       // enable TCA2 interrupt
    }
    __ASM("EI");
}

//******************************************************************************
// name         : DRV_TCA_PPG_Init()
// description  : TCA PPG Output Mode initialization
// input param  : - tcaReg : TCA PPG Output Mode initialization data structure
// output param :
// retval       : 
// note         : 
//******************************************************************************
void DRV_TCA_PPG_Init(TCA_PPG_REG tcaReg)
{
    BIT_LIST_8* tcaxCR = (BIT_LIST_8*)(0x0080 + (tcaReg.tcaSEL*8));
    BIT_LIST_8* tcaxMOD = (BIT_LIST_8*)(0x0081 + (tcaReg.tcaSEL*8));
    BIT_LIST_16* tcaxDRA = (BIT_LIST_16*)(0x0084 + (tcaReg.tcaSEL*8));
    BIT_LIST_16* tcaxDRB = (BIT_LIST_16*)(0x0086 + (tcaReg.tcaSEL*8)); 

    DRV_TCA_OutputPin_Init(tcaReg.tcaOutputPin); // Init TCA Output Pin
    PCKEN1 |= (1 << tcaReg.tcaSEL);              // enable TCAx
    
    tcaxCR->byte = 0;                            // Clear CR
    tcaxMOD->byte = 0;                           // Clear MOD
    tcaxDRA->word = 0;                           // Clear DRA

    tcaxMOD->b2_0.b2_0 = TAM_PPG_OUTPUT;         // TAM PPG output mode
    tcaxMOD->b4_3.b4_3 = tcaReg.tcaTACK;         // TACK
    tcaxMOD->bit.b7 = tcaReg.tcaTADBE;           // TADBE
    tcaxCR->bit.b1 = tcaReg.tcaTAMPPG;           // TAMPPG
    tcaxCR->bit.b6 = tcaReg.tcaTATFF;            // TATFF
    tcaxDRA->word = tcaReg.period;               // TAxDRA
    tcaxDRB->word = tcaReg.duty;                 // TAxDRB
     
    __ASM("DI");
    if(tcaReg.tcaSEL < TCASEL_TCA2)
    {
        IER6 |= (1 << (tcaReg.tcaSEL));          // enable TCA0, TCA1 interrupt
    }
    else if(tcaReg.tcaSEL == TCASEL_TCA2)
    {
        IER_TCA2 = 1;                            // enable TCA2 interrupt
    }
    __ASM("EI");
}

//******************************************************************************
// name         : DRV_TCA_PulseMeasure_Init()
// description  : TCA PulseMeasure Mode initialization
// input param  : - tcaReg : TCA PulseMeasure Mode initialization data structure
// output param :
// retval       : 
// note         : 
//******************************************************************************
void DRV_TCA_PulseMeasure_Init(TCA_PULSE_MEASURE_REG tcaReg)
{
    BIT_LIST_8* tcaxCR = (BIT_LIST_8*)(0x0080 + (tcaReg.tcaSEL * 8));
    BIT_LIST_8* tcaxMOD = (BIT_LIST_8*)(0x0081 + (tcaReg.tcaSEL * 8));
    BIT_LIST_16* tcaxDRA = (BIT_LIST_16*)(0x0084 + (tcaReg.tcaSEL * 8));
    BIT_LIST_16* tcaxDRB = (BIT_LIST_16*)(0x0086 + (tcaReg.tcaSEL * 8)); 

    DRV_TCA_InputPin_Init(tcaReg.tcaInputPin);   // Init TCA Output Pin
    PCKEN1 |= (1 << tcaReg.tcaSEL);              // enalbe TCAx
    
    tcaxCR->byte = 0;                            // Clear CR
    tcaxMOD->byte = 0;                           // Clear MOD
    tcaxDRA->word = 0;                           // Clear DRA

    tcaxMOD->b2_0.b2_0 = TAM_PULSE_MEASURE;      // TAM pulse measurement mode
    tcaxMOD->b4_3.b4_3 = tcaReg.tcaTACK;         // TACK
    tcaxMOD->bit.b5 = tcaReg.tcaTAMCAP;          // TAMCAP
    tcaxMOD->bit.b6 = tcaReg.tcaTATED;           // TATED
    tcaxCR->bit.b7 = tcaReg.tcaTAOVE;            // TAOVE
     
    __ASM("DI");
    if(tcaReg.tcaSEL < TCASEL_TCA2)
    {
        IER6 |= (1 << (tcaReg.tcaSEL));          // enable TCA0, TCA1 interrupt
    }
    else if(tcaReg.tcaSEL == TCASEL_TCA2)
    {
        IER_TCA2 = 1;                            // enable TCA2 interrupt
    }
    __ASM("EI");
}

//******************************************************************************
// name         : DRV_TCA_Deinit()
// description  : TCA deinitialization
// input param  : - tcaSel : TCAx selection : TCA0 ~ TCA2
// output param :
// retval       : 
// note         : 
//******************************************************************************
void DRV_TCA_Deinit(TCA_SEL tcaSel)
{
    PRSTR1 |= (1 << (3 + tcaSel));               // Reset TCAx
}

//******************************************************************************
// name         : DRV_TCA_OutputPin_Init()
// description  : TCAx output pin initialization
// input param  : - outputPin : output pin selection
// output param :
// retval       : 
// note         : 
//******************************************************************************
void DRV_TCA_OutputPin_Init(TCA_IOPin outputPin)
{
    uint8_t pA = ((outputPin >> 4) & 7);         // PA.B
    uint8_t pB = (outputPin & 7);                // PA.B
    uint8_t pxCFGCRvalue = (0x81 | (pB << 4));   // TCA IO Base Value: 81 = 1xxx0001b    
    uint8_t* pxCFGCR = (uint8_t*)(0x0140 + (pA * 2));
    uint8_t* pxOE = (uint8_t*)(0x0100 + pA);
    
    *pxCFGCR = pxCFGCRvalue;                     // PxCFGCE
    *pxOE |= (1 << pB);                          // PxOE Set 1 (input mode)
}

//******************************************************************************
// name         : DRV_TCA_InputPin_Init()
// description  : TCAx input pin initialization
// input param  : - inputPin : input pin selection
// output param :
// retval       : 
// note         : 
//******************************************************************************
void DRV_TCA_InputPin_Init(TCA_IOPin inputPin)
{
    uint8_t pA = ((inputPin >> 4) & 7);          // PA.B
    uint8_t pB = (inputPin & 7);                 // PA.B
    uint8_t pxCFGCRvalue = (0x81 | (pB << 4));   // TCA IO Base Value: 81 = 1xxx0001b
    uint8_t* pxCFGCR = (uint8_t*)(0x0140 + (pA * 2));
    uint8_t* pxOE = (uint8_t*)(0x0100 + pA);
    
    *pxCFGCR = pxCFGCRvalue;                     // PxCFGCE
    *pxOE &= ~(1 << pB);                         // PxOE Set 0 (input mode)
}

//******************************************************************************
// name         : DRV_TCA_Start()
// description  : Start TCA
// input param  : - tcaSel : TCAx selection : TCA0 ~ TCA2
// output param :
// retval       : 
// note         : 
//******************************************************************************
void DRV_TCA_Start(TCA_SEL tcaSel)
{
    BIT_LIST_8* tcaxCR = (BIT_LIST_8*)(0x0080 + (tcaSel * 8)); 
    tcaxCR->bit.b0 = 1;      // TCAxCR TAxS: enable TCAx
}

//******************************************************************************
// name         : DRV_TCA_Stop()
// description  : Stop TCA
// input param  : - tcaSel : TCAx selection : TCA0 ~ TCA2
// output param :
// retval       : 
// note         : 
//******************************************************************************
void DRV_TCA_Stop(TCA_SEL tcaSel)
{
    BIT_LIST_8* tcaxCR = (BIT_LIST_8*)(0x0080 + (tcaSel * 8)); 
    tcaxCR->bit.b0 = 0;      // TCAxCR TAxS: TCAxCR TAxS: disable TCAx
}


//******************************************************************************
// name         : DRV_TCA_ReadMeasure_PulseWidth()
// description  : Read single-edge captured value
// input param  : - tcaSel : TCAx selection : TCA0 ~ TCA2
// output param : - readValue : single-edge captured value
// retval       : 
// note         : 
//******************************************************************************
void DRV_TCA_ReadMeasure_PulseWidth(TCA_SEL tcaSel, uint16_t *readValue)
{
    BIT_LIST_16* tcaxDRB = (BIT_LIST_16*)(0x0086 + (tcaSel * 8));
    *readValue = tcaxDRB->word;
}

//******************************************************************************
// name         : DRV_TCA_ReadMeasure_Period()
// description  : Read double-edge captured value
// input param  : - tcaSel : TCAx selection : TCA0 ~ TCA2
// output param : - readValue : double-edge captured value
// retval       : 
// note         : 
//******************************************************************************
void DRV_TCA_ReadMeasure_Period(TCA_SEL tcaSel, uint16_t *readValue)
{
    BIT_LIST_16* tcaxDRA = (BIT_LIST_16*)(0x0084 + (tcaSel * 8));
    *readValue = tcaxDRA->word;
}

//******************************************************************************
// name         : DRV_TCA0_IRQ()
// description  : TCA0 interrupt service routine
// input param  : 
// output param :
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_TCA0_IRQ(void)
{
    TCA0_Int_Flag = 1;       // TCA0 interrupt flag
}

//******************************************************************************
// name         : DRV_TCA1_IRQ()
// description  : TCA1 interrupt service routine
// input param  : 
// output param :
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_TCA1_IRQ(void)
{
    TCA1_Int_Flag = 1;       // TCA1 interrupt flag
}

//******************************************************************************
// name         : DRV_TCA2_IRQ()
// description  : TCA2 interrupt service routine
// input param  : 
// output param :
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_TCA2_IRQ(void)
{
    TCA2_Int_Flag = 1;       // TCA2 interrupt flag
}


