//******************************************************************************
// file         : DRV_SQ7705_STM.c
// version      : V1.2 2023/08/22
// description  : STM related functions
// note         : STM related functions are gathered in this subroutine
//******************************************************************************
#include "DRV_SQ7705_STM.h"

uint8_t STM0_Int_Flag;                   // flag for STM0 interrupt
uint8_t STM1_Int_Flag;                   // flag for STM1 interrupt

//******************************************************************************
// name         : DRV_STM_Init()
// description  : STM initialization
// input param  : - stmReg : STM initialization data structure
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_STM_Init(STM_REG stmReg)
{
    BIT_LIST_8* stmxCR = (BIT_LIST_8*)(0x0280 + (stmReg.stmSEL*8));
    BIT_LIST_16* stmxV = (BIT_LIST_16*)(0x0284 + (stmReg.stmSEL*8));
    BIT_LIST_16* stmxDR = (BIT_LIST_16*)(0x0286 + (stmReg.stmSEL*8)); 

    PCKEN0 |= (1 << stmReg.stmSEL);         // enable STMx
    
    stmxCR->bit.b0 = 1;                     // STMEN
    stmxCR->bit.b1 = stmReg.stmCKS;         // STMCKS
    stmxCR->bit.b2 = 1;                     // STMIEN
    stmxCR->bit.b3 = stmReg.stmDSLP;        // DSLP
    stmxCR->b5_4.b5_4 = stmReg.stmCKD;      // STMCKD
    stmxV->word = stmReg.count;             // STMVL-STMVH
    stmxDR->word = stmReg.period;           // STMDRL-STMDRH 
    
    __ASM("DI");
    IER6 |= (1 << (2 + stmReg.stmSEL));     // enable STMx interrupt 
    __ASM("EI");
}

//******************************************************************************
// name         : DRV_STM_Deinit()
// description  : STM deinitialization
// input param  : - stmSel : STM selection : STM0 ~ STM1
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_STM_Deinit(STM_SEL stmSel)
{
    PRSTR0 |= (1 << (3 + stmSel));          // Reset STMx
}

//******************************************************************************
// name         : DRV_STM_Start()
// description  : Start STM
// input param  : - stmSel : STM selection : STM0 ~ STM1
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_STM_Start(STM_SEL stmSel)
{
    BIT_LIST_8* stmxCR = (BIT_LIST_8*)(0x0280 + (stmSel*8));
    BIT_LIST_8* stmxSR = (BIT_LIST_8*)(0x0281 + (stmSel*8));
    stmxSR->bit.b0 = 1;           // STMSR STMS:  clear start status
    stmxCR->bit.b0 = 1;           // STMCR STMEN: STMx Enable
}

//******************************************************************************
// name         : DRV_STM_Stop()
// description  : Stop STM
// input param  : - stmSel : STM selection : STM0 ~ STM1
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_STM_Stop(STM_SEL stmSel)
{
    BIT_LIST_8* stmxCR = (BIT_LIST_8*)(0x0280 + (stmSel*8));
    BIT_LIST_8* stmxSR = (BIT_LIST_8*)(0x0281 + (stmSel*8));
    stmxSR->bit.b0 = 1;           // STMSR STMS:  clear start status
    stmxCR->bit.b0 = 0;           // STMCR STMEN: disable STMx
}

//******************************************************************************
// name         : DRV_STM0_IRQ()
// description  : STM0 interrupt service routine
// input param  : 
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_STM0_IRQ(void)
{
    STM0_Int_Flag = 1;            // flag for STM0 interrupt
}

//******************************************************************************
// name         : DRV_STM1_IRQ()
// description  : STM1 interrupt service routine
// input param  : 
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_STM1_IRQ(void)
{
    STM1_Int_Flag = 1;            // flag for STM1 interrupt
}