//******************************************************************************
// file         : DRV_SQ7705_SPI.c
// version      : V1.2 2023/08/31
// description  : SPI related functions
// note         : SPI related functions are gathered in this subroutine
//******************************************************************************
#include "DRV_SQ7705_SPI.h"

//******************************************************************************
// name         : DRV_SPI0_Init()
// description  : SPI0 initialization
// input param  : initDef : SPI0 structure
// retval       : 
// note         : 
//******************************************************************************
void DRV_SPI0_Init(SPI0_INIT_STRUCT *initDef)
{
    DRV_SPI0_MOSI_Init(initDef->mosi);
    DRV_SPI0_MISO_Init(initDef->miso);
    DRV_SPI0_SCK_Init(initDef->sck);
    DRV_SPI0_CS_Init(initDef->cs);
    
    PCKEN2_SPI0 = 1;     // Enable
    PRSTR2_SPI0 = 1;     // Reset
    PRSTR2_SPI0 = 0;     // Reset
    
    DI;
    IFR_SPI0 = 0;
    IER_SPI0 = 1;
    EI;
    
    SPI0CR_MODE = 3;     // Tx and Rx
    SPI0CR |= initDef->mode; //CPOL/CPHA
    SPI0CFG_CSSEL = 1;   // CS active state 1 : active high
    SPI0CFG_CSSEL = 0;   // CS active state 0 : active low
    SPI0CFG_TXEMP = 1;   // set tramsmit status to high
    SPI0CFG_TXUIG = 1;   // ignore transmit underrun
    
//  SPI0CFG_RXERRIE = 0; // disable receive error interrupt
    SPI0CFG_ETXFIE = 1;  // enable end of transfer interrupt

    SPI0CFG_MSTR = initDef->mstr; // set operation mode

    SPI0CSTIM = 0x80;    // set hode time and setup time of chip select
    SPI0CKD = 0x00;      // set SPI clock divide register
    
    SPI0MCR_STOP = 1;    // cancel current transaction
    SPI0MCR_STOP = 0;    // not effect stop transaction
}

//******************************************************************************
// name         : DRV_SPI0_MOSI_Init()
// description  : SPI0 MOSI initialization
// input vparam : mosi : MOSI pin assign
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPI0_MOSI_Init(SPI0_MOSI_LIST mosi)
{
    switch(mosi) {
        case SPI0_MOSI_P00:
            P0OE_P0 = 1;
            P0CFGCR = 0x83;
            break;

        case SPI0_MOSI_P04:
            P0OE_P4 = 1;
            P0CFGCR = 0xC3;
            break;

        case SPI0_MOSI_P14:
            P1OE_P4 = 1;
            P1CFGCR = 0xC3;
            break;

        case SPI0_MOSI_P36:
            P3OE_P6 = 1;
            P3CFGCR = 0xE3;
            break;
    }
}

//******************************************************************************
// name         : DRV_SPI0_MISO_Init()
// description  : SPI0 MISO initialization
// input param  : miso :  MISO pin assign
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPI0_MISO_Init(SPI0_MISO_LIST miso)
{
    switch(miso) {
        case SPI0_MISO_P01:
            P0OE_P1 = 1;
            P0CFGCR = 0x93;
            break;

        case SPI0_MISO_P05:
            P0OE_P5 = 1;
            P0CFGCR = 0xD3;
            break;

        case SPI0_MISO_P15:
            P1OE_P5 = 1;
            P1CFGCR = 0xD3;
            break;

        case SPI0_MISO_P37:
            P3OE_P7 = 1;
            P3CFGCR = 0xF3;
            break;
    }
}

//******************************************************************************
// name         : DRV_SPI0_SCK_Init()
// description  : SPI0 SCK initialization
// input param  : sck : SPI clock setting
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPI0_SCK_Init(SPI0_SCK_LIST sck)
{
    switch(sck) {
        case SPI0_SCK_P02:
            P0OE_P2 = 1;
            P0CFGCR = 0xA3;
            break;

        case SPI0_SCK_P06:
            P0OE_P6 = 1;
            P0CFGCR = 0xE3;
            break;

        case SPI0_SCK_P16:
            P1OE_P6 = 1;
            P1CFGCR = 0xE3;
            break;

        case SPI0_SCK_P60:
            P6OE_P0 = 1;
            P6CFGCR = 0x83;
            break;
    }
}

//******************************************************************************
// name         : DRV_SPI0_CS_Init()
// description  : SPI0 CS initialization
// input param  : cs : assign chip-select pin
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPI0_CS_Init(SPI0_CS_LIST cs)
{
    switch(cs){
        case SPI0_CS_P03:
            P0OE_P3 = 1;
            P0CFGCR = 0xB3;
            break;

        case SPI0_CS_P07:
            P0OE_P7 = 1;
            P0CFGCR = 0xF3;
            break;

        case SPI0_CS_P17:
            P1OE_P7 = 1;
            P1CFGCR = 0xF3;
            break;

        case SPI0_CS_P61:
            P6OE_P1 = 1;
            P6CFGCR = 0x93;
            break;
    }
}

//******************************************************************************
// name         : DRV_SPI1_Init()
// description  : SPI1 initialization
// input param  : initDef : SPI1 initialization
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPI1_Init(SPI1_INIT_STRUCT *initDef)
{
    DRV_SPI1_CS_Init(initDef->cs);
    DRV_SPI1_SCK_Init(initDef->sck);
    DRV_SPI1_MOSI_Init(initDef->mosi);
    DRV_SPI1_MISO_Init(initDef->miso);
    
    PCKEN2_SPI1 = 1;     // Enable
    PRSTR3_SPI1 = 1;     // Reset
    PRSTR3_SPI1 = 0;     // Reset
    
    DI;
    IFR_SPI1 = 0;
    IER_SPI1 = 1;
    EI;
    
    SPI1CR_MODE = 3;     // Tx and Rx
    SPI1CR |= initDef->mode;  //CPOL/CPHA
    SPI1CFG_CSSEL = 1;   // CS active state 1 : active high
    SPI1CFG_CSSEL = 0;   // CS active state 0 : active low
    SPI1CFG_TXEMP = 1;   // set tramsmit status to high
    SPI1CFG_TXUIG = 1;   // ignore transmit underrun
     
    
//  SPI1CFG_RXERRIE = 0; // disable receive error interrupt
    SPI1CFG_ETXFIE = 1;  // enable end of transfer interrupt

    SPI1CFG_MSTR = initDef->mstr;  // set operation mode
    
    SPI1CSTIM = 0x00;    // set hode time and setup time of chip select
    SPI1CKD = 0x1;       // set SPI clock divide register
    
    SPI1MCR_STOP = 1;    // cancel current transaction
    SPI1MCR_STOP = 0;    // not effect stop transaction   
}

//******************************************************************************
// name         : DRV_SPI1_MOSI_Init()
// description  : SPI1 MOSI initialization
// input param  : mosi :M0SI initialization 
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPI1_MOSI_Init(SPI1_MOSI_LIST mosi)
{
    switch(mosi) {
        case SPI1_MOSI_P20:
            P2OE_P0 = 1;
            P2CFGCR = 0x82;
            break;

        case SPI1_MOSI_P54:
            P5OE_P4 = 1;
            P5CFGCR = 0xC2;
            break;
    }
}

//******************************************************************************
// name         : DRV_SPI1_MISO_Init()
// description  : SPI1 MISO initialization
// input param  : miso : MISO initialization
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPI1_MISO_Init(SPI1_MISO_LIST miso)
{
    switch(miso) {
        case SPI1_MISO_P21:
            P2OE_P1 = 1;
            P2CFGCR = 0x92;
            break;

        case SPI1_MISO_P55:
            P5OE_P5 = 1;
            P5CFGCR = 0xD2;
            break;
    }   
}

//******************************************************************************
// name         : DRV_SPI1_SCK_Init()
// description  : SPI1 SCK initialization
// input param  : sck :  assoign SPI clock pin  
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPI1_SCK_Init(SPI1_SCK_LIST sck)
{
    switch(sck) {
        case SPI1_SCK_P22:
            P2OE_P2 = 1;
            P2CFGCR = 0xA2;
            break;

        case SPI1_SCK_P56:
            P5OE_P6 = 1;
            P4CFGCR = 0xE2;
            break;
    }
}

//******************************************************************************
// name         : DRV_SPI1_CS_Init()
// description  : SPI1 CS initialization
// input param  : cs : assign chip-slect pin
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPI1_CS_Init(SPI1_CS_LIST cs)
{
    switch(cs) {
        case SPI1_CS_P23:
            P2OE_P3 = 1;
            P2CFGCR = 0xB2;
            break;

        case SPI1_CS_P57:
            P5OE_P7 = 1;
            P5CFGCR = 0xF2;
            break;

        case SPI1_CS_P70:
            P7OE_P0 = 1;
            P7CFGCR = 0x82;
            break;
    }
}

//******************************************************************************
// name         : DRV_SPIx_SendBytes()
// description  : SPI transmits data
// input param  : spi : select SPI0 or SPI1
//                data: data buffer to be transmit
//                length: send data length
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPIx_SendBytes(uint8_t spi, uint8_t *data, uint8_t length)
{
    uint8_t txIdx = 0;
    if(spi == SPI0) {
        while(length > txIdx) {
            while(SPI0SR_BUSY == 1);
            SPI0TX = data[txIdx];
            txIdx++;
        }
        SPI0CR_SPIEN = 1; // enable SPI0
    } else {
        while(length > txIdx) {
            while(SPI1SR_BUSY == 1);
            SPI1TX = data[txIdx];
            txIdx++;
        }
        SPI1CR_SPIEN = 1; // enable SPI1
    }
}

//******************************************************************************
// name         : DRV_SPIx_ReceiveBytes()
// description  : SPI receives data
// input param  : spi : select SPI0 or SPI1
//                length: read data length
// output param : data: received data buffer
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPIx_ReceiveBytes(uint8_t spi, uint8_t *data, uint8_t length)
{
    uint8_t rxIdx = 0;
    if(spi == SPI0) {
        SPI0CNT = 0;
        while(SPI0MCNT > rxIdx) {
            while(SPI0SR_BUSY == 1);
            data[rxIdx] = SPI0RX;
            rxIdx++;
        }   
    } else {
        SPI1CNT = 0;
        while(SPI1MCNT > rxIdx) {
            while(SPI1SR_BUSY == 1);
            data[rxIdx] = SPI1RX;
            rxIdx++;
        }
    }
}

//******************************************************************************
// name         : DRV_SPIx_MasterTxLength()
// description  : set transmit data length of master mode
// input param  : spi : select SPI0 or SPI1
//                cnt : transmit data length
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPIx_MasterTxLength(uint8_t spi, uint8_t cnt)
{
    if(spi == SPI0) {
        SPI0TXCNT = cnt;
        SPI0RXCNT = 0;
        SPI0MCNT = cnt;
    } else {
        SPI1TXCNT = cnt;
        SPI1RXCNT = 0;
        SPI1OFFSET = 0;
        SPI1MCNT = cnt;
    }
}

//******************************************************************************
// name         : DRV_SPIx_SlaveRxLength()
// description  : set received data length of slave mode
// input param  : spi : select SPI0 or SPI1
//                cnt: received data length
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPIx_SlaveRxLength(uint8_t spi, uint8_t cnt)
{
    if(spi == SPI0) {
        SPI0TXCNT = 0;
        SPI0RXCNT = cnt;
        SPI0MCNT = cnt;
    } else {
        SPI1TXCNT = 0;
        SPI1RXCNT = cnt;
        SPI1MCNT = cnt;
    }
}

//******************************************************************************
// name         : DRV_SPIx_TxInit()
// description  : SPIx TX initialization
// input param  : spi : select SPI0 or SPI1
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPIx_TxInit(uint8_t spi)
{
    if(spi == 0) {
        SPI0TXCR_TXFLSH = 1;
        SPI0TXCR_TXUDIE = 0;
        SPI0TXCR_TXOVIE = 0;
        SPI0TXCR_TXIE = 1;
        SPI0TXCR_TXTHR = 0; 
    } else {
        SPI1TXCR_TXFLSH = 1;
        SPI1TXCR_TXUDIE = 0;
        SPI1TXCR_TXOVIE = 0;
        SPI1TXCR_TXIE = 1;
        SPI1TXCR_TXTHR = 3;
    }
    DRV_SPIx_ClrTxFlag(spi);
}

//******************************************************************************
// name         : DRV_SPIx_ClrTxFlag()
// description  : clear TX related flag of SPIx
// input param  : spi : select SPI0 or SPI1
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPIx_ClrTxFlag(uint8_t spi)
{
    if(spi == 0) {
        SPI0TXFSR |= 0x07;
    } else {
        SPI1TXFSR |= 0x07;
    }
}

//******************************************************************************
// name         : DRV_SPIx_ClrTxFlag()
// description  : clear RX related flag of SPIx
// input param  : spi : select SPI0 or SPI1
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPIx_ClrRxFlag(uint8_t spi)
{
    if(spi == 0) {
        SPI0TXFSR |= 0x70;
    } else {
        SPI1TXFSR |= 0x70;
    }
}

//******************************************************************************
// name         : DRV_SPIxSlaveRxInit()
// description  : SPIx RX of slave mode initialization
// input param  : spi : select SPI0 or SPI1
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPIxSlaveRxInit(uint8_t spi)
{
    if(spi == 0) {
        SPI0RXCR_RXFLSH = 1;
        SPI0RXCR_RXUDIE = 0;
        SPI0RXCR_RXOVIE = 0;
        SPI0RXCR_RXIE = 1;
        SPI0RXCR_RXTHR = 0;
    } else {
        SPI1RXCR_RXFLSH = 1;
        SPI1RXCR_RXUDIE = 0;
        SPI1RXCR_RXOVIE = 0;
        SPI1RXCR_RXIE = 1;
        SPI1RXCR_RXTHR = 0; 
    }
    DRV_SPIx_ClrRxFlag(spi);
}

//******************************************************************************
// name         : DRV_SPIxMasterRxInit()
// description  : SPIx RX of master mode initialization
// input param  : spi : select SPI0 or SPI1
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPIxMasterRxInit(uint8_t spi)
{
    if(spi == 0) {
        SPI0RXCR_RXFLSH = 1;
        SPI0RXCR_RXUDIE = 0;
        SPI0RXCR_RXOVIE = 0;
        SPI0RXCR_RXIE = 0;
        SPI0RXCR_RXTHR = 0; 
    } else {
        SPI1RXCR_RXFLSH = 1;
        SPI1RXCR_RXUDIE = 0;
        SPI1RXCR_RXOVIE = 0;
        SPI1RXCR_RXIE = 0;
        SPI1RXCR_RXTHR = 0;     
    }
    DRV_SPIx_ClrRxFlag(spi);
}

//******************************************************************************
// name         : DRV_SPIx_Start()
// description  : start SPIx
// input param  : spi : select SPI0 or SPI1
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPIx_Start(uint8_t spi)
{
    if(spi == SPI0) {
        SPI0CR_SPIEN = 1;
    } else {
        SPI1CR_SPIEN = 1;
    }
}

//******************************************************************************
// name         : DRV_SPIx_Stop()
// description  : stop SPIx
// input param  : spi : select SPI0 or SPI1
// retval       :  
// note         : 
//******************************************************************************
void DRV_SPIx_Stop(uint8_t spi)
{
    if(spi == SPI0) {
        SPI0CR_SPIEN = 0;
    } else {
        SPI1CR_SPIEN = 0;
    }
}

//******************************************************************************
// name         : DRV_SPI0_IRQ()
// description  : SPI0 interrupt service routine
// retval       :  
// note         : 
//******************************************************************************
void __interrupt DRV_SPI0_IRQ(void)
{
    SPI0SR_ETXFI = 1;   //W1C
//  SPI0SR_RXERR = 1;
//  SPI0SR_RXCRC = 1;
//  SPI0SR_CSERR = 1;
    SPI0TXFSR_RXF = 1;  //W1C
    __ASM("NOP");       // no operation
}

//******************************************************************************
// name         : DRV_SPI1_IRQ()
// description  : SPI1 interrupt service routine
// retval       :  
// note         : 
//******************************************************************************
void __interrupt DRV_SPI1_IRQ(void)
{
    SPI1SR_ETXFI = 1;   //W1C
//  SPI1SR_CSERR = 1;
//  SPI1SR_RXERR = 1;
//  SPI1SR_RXCRC = 1;
    SPI1TXFSR_RXF = 1;  //W1C
    __ASM("NOP");       // no operation
}