//******************************************************************************
// file         : main.c
// version      : V1.3  2023/11/20
// description  : main program
// note         : this example program is a demostration for SPI1 sending data to SPI0 within a single SQ7705 chip
//                SPI1 sends data, if SPI0 is received correctly then light on LED1
//                if not yet communicating or receiving error then not light on LED1
//                1. set SPI1 as Master 
//                   set P2.0 to MOSI
//                   set P2.1 to MISO
//                   set P2.2 to SCK
//                   set P2.3 to CS
// 
//                2. set set SPI0 as Slave 
//                   set P0.0 to MOSI
//                   set P0.1 to MISO
//                   set P0.2 to SCK
//                   set P0.3 to CS
// 
//                3. MOSI0(P0.0) connects to MOSI1(P2.0)
//                   MISO0(P0.1) connects to MISO1(P2.1)
//                   SCK0(P0.2) connects to SCK1(P2.2)
//                   CS0(P0.3) connects to CS1(P2.3) 
// 
//                [LED Configuration] LED1  (P5.0) 
//******************************************************************************
#ifndef IO_MEM
#define IO_MEM
#endif
#include "ADM_main.h"
#include "DRV_SQ7705_SYSCLK.h"
#include "DRV_SQ7705_SPI.h"

/* --------------------------- global variables ----------------------------- */
uint8_t sendBuf[8];                          // data transmit buffers
uint8_t recvBuf[8];                          // data receive buffers
uint16_t rIdx;
uint8_t comparePass;                         // data comparison result
SPI0_INIT_STRUCT spi0Struct;                 // create a structure variable for SPI0
SPI1_INIT_STRUCT spi1Struct;                 // create a structure variable for SPI1

/* ---------------------------- main program -------------------------------- */
void ADM_Main()
{
    DRV_Clock_Init(Clk_Source_SYS, 1);        // Initial system clock (PLL 48MHz/2 = 24MHz)

    /* ------------------------- GPIO initialization ------------------------ */
    DRV_GPIO_Init(GPIO_P50, GPIO_OUTPUT, GPIO_PULL_UP); // set P5.0 as output pin
                                                        // enable P5.0 built-in pull-up resist
    LED1_off;                                           // turn off LED
    
    sendBuf[0] = 0x11;
    sendBuf[1] = 0x22;
    sendBuf[2] = 0x33;
    sendBuf[3] = 0x44;
    sendBuf[4] = 0x55;
    sendBuf[5] = 0x66;
    sendBuf[6] = 0x77;
    sendBuf[7] = 0x88;
    /* ------------------------ SPI0 initialization ------------------------- */
    spi0Struct.mosi = SPI0_MOSI_P00;              // P00 MOSI
    spi0Struct.miso = SPI0_MISO_P01;              // P01 MISO
    spi0Struct.sck =  SPI0_SCK_P02;               // P02 SCK
    spi0Struct.cs =   SPI0_CS_P03;                // P03 CS
    spi0Struct.mode = SPI_MODE_0;                 // CPOL=0 /CPHA=0
    spi0Struct.mstr = SPI_SLAVE;                  // Slave mode
    DRV_SPI0_Init(&spi0Struct);                   // initialize SPI0
    DRV_SPIx_TxInit(SPI0);                        // SPI0 Tx initialization
    DRV_SPIxSlaveRxInit(SPI0);                    // SPI0 Slave Rx initialization
    DRV_SPIx_Start(SPI0);
    /* ------------------------ SPI1 initialization ------------------------- */
    spi1Struct.mosi = SPI1_MOSI_P20;              // P20 MOSI
    spi1Struct.miso = SPI1_MISO_P21;              // P21 MISO
    spi1Struct.sck =  SPI1_SCK_P22;               // P22 SCK
    spi1Struct.cs =   SPI1_CS_P23;                // P23 CS
    spi1Struct.mode = SPI_MODE_0;                 // CPOL=0 /CPHA=0
    spi1Struct.mstr = SPI_MASTER;                 // Master mode
    DRV_SPI1_Init(&spi1Struct);                   // initialize SPI1
    DRV_SPIx_TxInit(SPI1);                        // SPI1 Tx initialization
    
    while(1) {
        comparePass = TRUE;
    
        DRV_SPIx_MasterTxLength(SPI1, 8);         // set transmit data length to 8
        DRV_SPIx_SlaveRxLength(SPI0, 8);          // set receive data length to 8
        memset(recvBuf,0xFF,8);
        DRV_SPIx_SendBytes(SPI1,&sendBuf[0],8);   // SPI1 transmit data
        DRV_SPIx_ReceiveBytes(SPI0,&recvBuf[0],8);// SPI0 receive data

        /* ---- data comparison ---- */
        for(rIdx = 0;rIdx < 8;rIdx++) {
            if(sendBuf[rIdx] != recvBuf[rIdx]) {
                comparePass = FALSE;
                break;
            }
        }
        if(comparePass == TRUE)
            LED1_on;      // PASS
        else
            LED1_off;     // ERROR
    }
}

/* ----------------------- interrupt service routine ------------------------ */
void __interrupt OnlyReti(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n OnlyRetn(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n WDT_IRQ(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n SWI_IRQ(void)               // software reset interrupt service routine
{
    SWRST = 0x5a;
    SWRST = 0xa5;
    SWRST = 0xc3;
    SWRST = 0x3c;
}

void __interrupt IntADM_Req(void)
{

}

/* ---------------------- end of interrupt service routine ------------------ */


/* ----------------------- SQ7705 interrupt vector -------------------------- */
#pragma section const ADM_INT_VECTOR
void * const IntTbl1[] = {
    ADM_STARTUP,             /* 0x0000 : RESET               */    //non-maskable
    SWI_IRQ,                 /* 0x0002 : SWI_IRQ/INTUNDEF    */    //non-maskable
    (void *)0xffff,          /* 0x0004 : Reserved            */    
    WDT_IRQ,                 /* 0x0006 : WDT_IRQ             */    //non-maskable interrupt source:watchdog timer
    OnlyReti,                /* 0x0008 : TAMPER_IRQ          */    //interrupt source:tamper detection
    OnlyReti,                /* 0x000A : CFD_IRQ             */    //interrupt source:clock faildetection
    OnlyReti,                /* 0x000C : HVD_IRQ             */
    IntADM_Req,              /* 0x000E : U2A_IRQ/IntADM_Req  */    //interrupt source:user mode to admin mode
    OnlyReti,                /* 0x0010 : RTC_IRQ             */    //interrupt source:real-time clock timer
    OnlyReti,                /* 0x0012 : TMRA00_IRQ          */    //interrupt source:TMRA00 8-bit timer
    OnlyReti,                /* 0x0014 : TMRA01_IRQ          */    //interrupt source:TMRA01 8-bit timer
    OnlyReti,                /* 0x0016 : TMRA10_IRQ          */    //interrupt source:TMRA10 8-bit timer
    OnlyReti,                /* 0x0018 : TMRA11_IRQ          */    //interrupt source:TMRA11 8-bit timer
    OnlyReti,                /* 0x001A : TMRA20_IRQ          */
    OnlyReti,                /* 0x001C : TMRA21_IRQ          */
    OnlyReti,                /* 0x001E : TBT_IRQ             */
    OnlyReti,                /* 0x0020 : RX0_IRQ             */    //interrupt source:UART0 RX0
    OnlyReti,                /* 0x0022 : TX0_IRQ             */    //interrupt source:UART0 TX0
    OnlyReti,                /* 0x0024 : I2C0_IRQ            */    //interrupt source:I2C0
    DRV_SPI0_IRQ,            /* 0x0026 : SPI0_IRQ            */    //interrupt source:SPI0
    OnlyReti,                /* 0x0028 : EXT0_IRQ            */    //external interrupt0
    OnlyReti,                /* 0x002A : EXT1_IRQ            */    //external interrupt1
    OnlyReti,                /* 0x002C : EXT2_IRQ            */    //external interrupt2
    OnlyReti,                /* 0x002E : EXT3_IRQ            */    //external interrupt3
    OnlyReti,                /* 0x0030 : ADC_IRQ             */    //interrupt source:ADC
    OnlyReti,                /* 0x0032 : LVD0_IRQ            */
    OnlyReti,                /* 0x0034 : LVD1_IRQ            */
    OnlyReti,                /* 0x0036 : Reserved            */
    OnlyReti,                /* 0x0038 : Reserved            */
    OnlyReti,                /* 0x003A : FMC_IRQ             */    //interrupt source:flash memory controller
    OnlyReti,                /* 0x003C : DMACH0_IRQ          */
    OnlyReti,                /* 0x003E : DMACH1_IRQ          */
    OnlyReti,                /* 0x0040 : DMACH2_IRQ          */
    OnlyReti,                /* 0x0042 : DMACH3_IRQ          */
    OnlyReti,                /* 0x0044 : Reserved            */
    OnlyReti,                /* 0x0046 : Reserved            */
    OnlyReti,                /* 0x0048 : Reserved            */
    OnlyReti,                /* 0x004A : Reserved            */
    OnlyReti,                /* 0x004C : Reserved            */
    OnlyReti,                /* 0x004E : DIC_IRQ             */    //interrupt source:DIC
    OnlyReti,                /* 0x0050 : LEUART_RX_IRQ       */
    OnlyReti,                /* 0x0052 : LEUART_TX_IRQ       */
    OnlyReti,                /* 0x0054 : TRNG_IRQ            */
    OnlyReti,                /* 0x0056 : EXT4_IRQ            */    //external interrupt4
    OnlyReti,                /* 0x0058 : EXT5_IRQ            */    //external interrupt5
    OnlyReti,                /* 0x005A : EXT6_IRQ            */    //external interrupt6
    OnlyReti,                /* 0x005C : EXT7_IRQ            */    //external interrupt7
    OnlyReti,                /* 0x005E : EEMC_IRQ            */    //interrupt source:EEPROM controller
    OnlyReti,                /* 0x0060 : TCA0_IRQ            */    //interrupt source:TCA0 16-bit timer
    OnlyReti,                /* 0x0062 : TCA1_IRQ            */    //interrupt source:TCA1 16-bit timer
    OnlyReti,                /* 0x0064 : STM0_IRQ            */
    OnlyReti,                /* 0x0066 : STM1_IRQ            */
    OnlyReti,                /* 0x0068 : AES_IRQ             */
    OnlyReti,                /* 0x006A : SHA_IRQ             */
    OnlyReti,                /* 0x006C : RX1_IRQ             */    //interrupt source:UART1 RX1
    OnlyReti,                /* 0x006E : TX1_IRQ             */    //interrupt source:UART1 TX1
    OnlyReti,                /* 0x0070 : I2C1_IRQ            */    //interrupt source:I2C1
    DRV_SPI1_IRQ,            /* 0x0072 : SPI1_IRQ            */    //interrupt source:SPI1
    OnlyReti,                /* 0x0074 : RX2_IRQ             */    //interrupt source:UART2 RX2
    OnlyReti,                /* 0x0076 : TX2_IRQ             */    //interrupt source:UART2 TX2
    OnlyReti                 /* 0x0078 : TCA2_IRQ            */    //interrupt source:TCA2 16-bit timer
};
#pragma section const
/* -------------------- end of SQ7705 interrupt vector ---------------------- */

