//******************************************************************************
// file         : main.c
// version      : V1.3  2023/11/20
// description  : main program, after initialization SHA, perform SHA encryption
//                if get the correct result, light on LED1
//
// note         : [LED Configuration] LED1 (P2.0)
//******************************************************************************
#ifndef IO_MEM
#define IO_MEM
#endif
#include "ADM_main.h"
#include "DRV_SQ7705_GPIO.h"
#include "DRV_SQ7705_SHA.h"
#include "DRV_SQ7705_SYSCLK.h"
#include "DRV_SQ7705_EEPROM.h"

/* --------------------------- global variables ----------------------------- */
unsigned char ADM_ANS_SRAM_SRAM[32] = {
    0x4d, 0xab, 0x03, 0x9a, 0x26, 0x36, 0x62, 0x2f, 0xea, 0x19, 0xd9, 0x62, 0x4c, 0x35, 0x07, 0x55,
    0x95, 0x69, 0xf5, 0xfa, 0x16, 0x14, 0xe9, 0x2f, 0x6b, 0x99, 0xac, 0x57, 0x24, 0x1b, 0x5f, 0xb9
};

unsigned char ADM_ANS_SRAM_EE[32] = {
    0x4d, 0xc2, 0x8e, 0x2a, 0x8c, 0x99, 0xd7, 0xca, 0xb7, 0x4d, 0xce, 0xd9, 0x75, 0x08, 0x30, 0x7c,
    0xd7, 0x7c, 0x32, 0xf1, 0x49, 0x81, 0xc7, 0xc5, 0x94, 0x71, 0xa0, 0x11, 0xce, 0x2f, 0xd9, 0x28
};

unsigned char ADM_MSG_EE[64] = {
    0x5A, 0x86, 0xB7, 0x37, 0xEA, 0xEA, 0x8E, 0xE9, 0x76, 0xA0, 0xA2, 0x4D, 0xA6, 0x3E, 0x7E, 0xD7,
    0xEE, 0xFA, 0xD1, 0x8A, 0x10, 0x1C, 0x12, 0x11, 0xE2, 0xB3, 0x65, 0x0C, 0x51, 0x87, 0xC2, 0xA8,
    0xA6, 0x50, 0x54, 0x72, 0x08, 0x25, 0x1F, 0x6D, 0x42, 0x37, 0xE6, 0x61, 0xC7, 0xBF, 0x4C, 0x77,
    0xF3, 0x35, 0x39, 0x03, 0x94, 0xC3, 0x7F, 0xA1, 0xA9, 0xF9, 0xBE, 0x83, 0x6A, 0xC2, 0x85, 0x09
};

unsigned char ADM_MSG_SRAM1[64] = {
    0x6b, 0x91, 0x8f, 0xb1, 0xa5, 0xad, 0x1f, 0x9c, 0x5e, 0x5d, 0xbd, 0xf1, 0x0a, 0x93, 0xa9, 0xc8,
    0xf6, 0xbc, 0xa8, 0x9f, 0x37, 0xe7, 0x9c, 0x9f, 0xe1, 0x2a, 0x57, 0x22, 0x79, 0x41, 0xb1, 0x73,
    0xac, 0x79, 0xd8, 0xd4, 0x40, 0xcd, 0xe8, 0xc6, 0x4c, 0x4e, 0xbc, 0x84, 0xa4, 0xc8, 0x03, 0xd1,
    0x98, 0xa2, 0x96, 0xf3, 0xde, 0x06, 0x09, 0x00, 0xcc, 0x42, 0x7f, 0x58, 0xca, 0x6e, 0xc3, 0x73,
};

unsigned char ADM_MSG_SRAM2[64] = {
    0x6b, 0x91, 0x8f, 0xb1, 0xa5, 0xad, 0x1f, 0x9c, 0x5e, 0x5d, 0xbd, 0xf1, 0x0a, 0x93, 0xa9, 0xc8,
    0xf6, 0xbc, 0xa8, 0x9f, 0x37, 0xe7, 0x9c, 0x9f, 0xe1, 0x2a, 0x57, 0x22, 0x79, 0x41, 0xb1, 0x73,
    0xac, 0x79, 0xd8, 0xd4, 0x40, 0xcd, 0xe8, 0xc6, 0x4c, 0x4e, 0xbc, 0x84, 0xa4, 0xc8, 0x03, 0xd1,
    0x98, 0xa2, 0x96, 0xf3, 0xde, 0x06, 0x09, 0x00, 0xcc, 0x42, 0x7f, 0x58, 0xca, 0x6e, 0xc3, 0x73,
};

uint8_t digestResult[32];
uint8_t plainText[16] = {                   // plaintext
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
};
uint8_t cipherText[32] = {                  // ciphertext
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};
const uint8_t ansBuf[32] = {                // correct ciphertext
    0xBE, 0x45, 0xCB, 0x26, 0x05, 0xBF, 0x36, 0xBE, 0xBD, 0xE6, 0x84, 0x84, 0x1A, 0x28, 0xF0, 0xFD,
    0x43, 0xC6, 0x98, 0x50, 0xA3, 0xDC, 0xE5, 0xFE, 0xDB, 0xA6, 0x99, 0x28, 0xEE, 0x3A, 0x89, 0x91
};
uint8_t idx;
uint8_t isCorrect;
/* --------------------------- Example function  ----------------------------- */
//******************************************************************************
// name         : EEPROM_SetUp()
// description  : set up data in EEPROM for example oce
//
// input param  :
//                startAddress : start address
//                endAdress    : end address
//                data         : data array for testing
//                size         : data size
//
// output param :
// retval       : 0 means OK , other failed
// note         :Example for different size data in SHA continue mode processing
//******************************************************************************
uint8_t  EEPROM_SetUp(uint16_t startAddress, uint16_t endAdress, uint8_t *data, uint16_t Size)
{
    int i = 0;
    uint16_t wAddr = startAddress;
    uint8_t readData = 0;
    uint16_t address = 0;
    //Erase sector
    for (address = startAddress; address < endAdress; address += 32) {
        DRV_EEPROM_EraseSector(address);
    }
    //write data
    for (i = 0; i < Size; i++) {
        wAddr = startAddress + i;
        DRV_EEPROM_WriteByte(wAddr, data[i]);
    }
    //read and check
    for (i = 0; i < Size; i++) {
        wAddr = startAddress + i;
        DRV_EEPROM_ReadByte(wAddr, &readData);
        if (data[i] != readData)
            return 1;
    }
    return 0;
}

//******************************************************************************
// name         : SHA256ContiueExample_SRAM_SRAM()
// description  : SHA256b continue mode example with memory type SRAM+SRAM
// input param  :
// output param :
// retval       : 0 means OK , other failed
// note         :Example for different size data in SHA continue mode processing
//               Total message size(totalMsgSize) <= 64 example
//               For example  totalMsgSize 19
//               >>DRV_SHA_Init(TRUE,TRUE);
//               >>DRV_SHA_ContiStart(); reset continue mode process counter to 0
//               >>DRV_SHA_ContiFinal(memoryType, messageAddress, 19,digestResult);
//
//               Total message size(totalMsgSize) > 64 example
//               For example  totalMsgSize 129
//               >>DRV_SHA_Init(TRUE,TRUE);
//               >>DRV_SHA_ContiStart(); reset continue mode counter to 0
//               >>DRV_SHA_ContiUpdate(SRAM, messageAddress1);//process byte 0~63
//               >>DRV_SHA_ContiUpdate(SRAM, messageAddress2);//process byte 64~127
//               >>DRV_SHA_ContiFinal(memoryType, messageAddress3, 1,digestResult); //process byte 128

//******************************************************************************

uint8_t SHA256ContiueExample_SRAM_SRAM()
{
    uint8_t  ansCorrect = TRUE;
    DRV_SHA_Init(TRUE, TRUE);
    DRV_SHA_ContiStart();
    DRV_SHA_ContiUpdate(SRAM, ADM_MSG_SRAM1);
    DRV_SHA_ContiFinal(SRAM, ADM_MSG_SRAM2, 11, digestResult);
    /* ------------------------- check result ------------------------------- */
    for (idx = 0; idx < 32; idx++) {
        if (digestResult[idx] != ADM_ANS_SRAM_SRAM[idx]) {
            ansCorrect = FALSE;
            break;
        }
    }
    return ansCorrect;
}
//******************************************************************************
// name         : SHA256ContiueExample_SRAM_EEPROM()
// description  : SHA256b continue mode example with memory type SRAM+EEPROM
// input param  :
// output param :
// retval       : 0 means OK , other failed
// note         :Example for different size data in SHA continue mode processing
//               Total message size(totalMsgSize) <= 64 example
//               For example  totalMsgSize 19
//               >>DRV_SHA_Init(TRUE,TRUE);
//               >>DRV_SHA_ContiStart(); reset continue mode process counter to 0
//               >>DRV_SHA_ContiFinal(memoryType, messageAddress, 19,digestResult);
//
//               Total message size(totalMsgSize) > 64 example
//               For example  totalMsgSize 129
//               >>DRV_SHA_Init(TRUE,TRUE);
//               >>DRV_SHA_ContiStart(); reset continue mode counter to 0
//               >>DRV_SHA_ContiUpdate(SRAM, messageAddress1);//process byte 0~63
//               >>DRV_SHA_ContiUpdate(EEPROM, messageAddress2);//process byte 64~127
//               >>DRV_SHA_ContiFinal(memoryType, messageAddress3, 1,digestResult); //process byte 128

//******************************************************************************
uint8_t SHA256ContiueExample_SRAM_EEPROM()
{
    uint8_t  ansCorrect = TRUE;
    uint16_t ADM_Start_Address = 0x3040, ADM_End_Address = 0x3146;

    EEPROM_SetUp(ADM_Start_Address, ADM_End_Address, ADM_MSG_EE, 64);
    DRV_SHA_ContiStart();
    DRV_SHA_ContiUpdate(SRAM, ADM_MSG_SRAM1);
    DRV_SHA_ContiFinal(EEPROM, (uint8_t *)ADM_Start_Address, 11, digestResult);
    for (idx = 0; idx < 32; idx++) {
        if (digestResult[idx] != ADM_ANS_SRAM_EE[idx]) {
            ansCorrect = FALSE;
            break;
        }
    }
    return ansCorrect;
}
//******************************************************************************
// name         : SHA256NonContiueExample()
// description  : SHA256 non continue mode example
// input param  :
// output param :
// retval       : 0 means OK , other failed
// note         :
//******************************************************************************
uint8_t SHA256NonContiueExample()
{
    uint8_t  ansCorrect = TRUE;
    DRV_SHA_Init(TRUE, TRUE);
    DRV_SHA_Size(32);
    DRV_SHA_Process_DataIn((uint8_t *)plainText, 16, 1, 0); // input plaintext
    DRV_SHA_Process_End((uint8_t *)cipherText);             // end of encryption and output the result to cipherText
    /* ------------------------- check result ------------------------------- */
    for (idx = 0; idx < 32; idx++) {
        if (cipherText[idx] != ansBuf[idx]) {
            ansCorrect = FALSE;                              // result is wrong
            break;
        }
    }
    return ansCorrect;


}
/* ---------------------------- main program -------------------------------- */
void ADM_Main()
{
    uint8_t demoCase = 1;
    uint8_t i = 0;
    DIS_WDT;                            // Disable watch dog timer.

    DRV_Clock_Init(Clk_Source_SYS, 1);  // PLL 24MHz Initial system clock.
    /* ----------------------- GPIO initialization -------------------------- */
    DRV_GPIO_Init(GPIO_P20, GPIO_OUTPUT, GPIO_PULL_UP); // set P2.0 as output pin
    // enable P2.0 built-in pull-up resistor
    LED1_off;
    isCorrect = FALSE;
    for ( i = 0 ; i < 3; i++ ) {
        switch (demoCase) {
        case 0:
            isCorrect = SHA256NonContiueExample();
            break;
        case 1:
            isCorrect = SHA256ContiueExample_SRAM_SRAM();
            break;
        case 2:
            isCorrect = SHA256ContiueExample_SRAM_EEPROM();
            break;
        default:
            break;
        }
        if(isCorrect==FALSE)
           break;
    }
    while (1) {
        if (isCorrect == TRUE) {
            LED1_on;                    // light on LED1
        } else {
            LED1_off;                   // light off LED1
        }
    }
}

/* ----------------------- interrupt service routine ------------------------ */
void __interrupt OnlyReti(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n OnlyRetn(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n WDT_IRQ(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n SWI_IRQ(void)             // software reset interrupt service routine
{
    SWRST = 0x5a;
    SWRST = 0xa5;
    SWRST = 0xc3;
    SWRST = 0x3c;
}

void __interrupt IntADM_Req(void)
{

}

/* ---------------------- end of interrupt service routine ------------------ */


/* ----------------------- SQ7705 interrupt vector -------------------------- */
#pragma section const ADM_INT_VECTOR
void *const IntTbl1[] = {
    ADM_STARTUP,             /* 0x0000 : RESET               */    //non-maskable
    SWI_IRQ,                 /* 0x0002 : SWI_IRQ/INTUNDEF    */    //non-maskable
    (void *)0xffff,          /* 0x0004 : Reserved            */
    WDT_IRQ,                 /* 0x0006 : WDT_IRQ             */    //non-maskable interrupt source:watchdog timer
    OnlyReti,                /* 0x0008 : TAMPER_IRQ          */    //interrupt source:tamper detection
    OnlyReti,                /* 0x000A : CFD_IRQ             */    //interrupt source:clock faildetection
    OnlyReti,                /* 0x000C : HVD_IRQ             */
    IntADM_Req,              /* 0x000E : U2A_IRQ/IntADM_Req  */    //interrupt source:user mode to admin mode
    OnlyReti,                /* 0x0010 : RTC_IRQ             */    //interrupt source:real-time clock timer
    OnlyReti,                /* 0x0012 : TMRA00_IRQ          */    //interrupt source:TMRA00 8-bit timer
    OnlyReti,                /* 0x0014 : TMRA01_IRQ          */    //interrupt source:TMRA01 8-bit timer
    OnlyReti,                /* 0x0016 : TMRA10_IRQ          */    //interrupt source:TMRA10 8-bit timer
    OnlyReti,                /* 0x0018 : TMRA11_IRQ          */    //interrupt source:TMRA11 8-bit timer
    OnlyReti,                /* 0x001A : TMRA20_IRQ          */
    OnlyReti,                /* 0x001C : TMRA21_IRQ          */
    OnlyReti,                /* 0x001E : TBT_IRQ             */
    OnlyReti,                /* 0x0020 : RX0_IRQ             */    //interrupt source:UART0 RX0
    OnlyReti,                /* 0x0022 : TX0_IRQ             */    //interrupt source:UART0 TX0
    OnlyReti,                /* 0x0024 : I2C0_IRQ            */    //interrupt source:I2C0
    OnlyReti,                /* 0x0026 : SPI0_IRQ            */    //interrupt source:SPI0
    OnlyReti,                /* 0x0028 : EXT0_IRQ            */    //external interrupt0
    OnlyReti,                /* 0x002A : EXT1_IRQ            */    //external interrupt1
    OnlyReti,                /* 0x002C : EXT2_IRQ            */    //external interrupt2
    OnlyReti,                /* 0x002E : EXT3_IRQ            */    //external interrupt3
    OnlyReti,                /* 0x0030 : ADC_IRQ             */    //interrupt source:ADC
    OnlyReti,                /* 0x0032 : LVD0_IRQ            */
    OnlyReti,                /* 0x0034 : LVD1_IRQ            */
    OnlyReti,                /* 0x0036 : Reserved            */
    OnlyReti,                /* 0x0038 : Reserved            */
    OnlyReti,                /* 0x003A : FMC_IRQ             */    //interrupt source:flash memory controller
    OnlyReti,                /* 0x003C : DMACH0_IRQ          */
    OnlyReti,                /* 0x003E : DMACH1_IRQ          */
    OnlyReti,                /* 0x0040 : DMACH2_IRQ          */
    OnlyReti,                /* 0x0042 : DMACH3_IRQ          */
    OnlyReti,                /* 0x0044 : Reserved            */
    OnlyReti,                /* 0x0046 : Reserved            */
    OnlyReti,                /* 0x0048 : Reserved            */
    OnlyReti,                /* 0x004A : Reserved            */
    OnlyReti,                /* 0x004C : Reserved            */
    OnlyReti,                /* 0x004E : DIC_IRQ             */    //interrupt source:DIC
    OnlyReti,                /* 0x0050 : LEUART_RX_IRQ       */
    OnlyReti,                /* 0x0052 : LEUART_TX_IRQ       */
    OnlyReti,                /* 0x0054 : TRNG_IRQ            */
    OnlyReti,                /* 0x0056 : EXT4_IRQ            */    //external interrupt4
    OnlyReti,                /* 0x0058 : EXT5_IRQ            */    //external interrupt5
    OnlyReti,                /* 0x005A : EXT6_IRQ            */    //external interrupt6
    OnlyReti,                /* 0x005C : EXT7_IRQ            */    //external interrupt7
    OnlyReti,                /* 0x005E : EEMC_IRQ            */    //interrupt source:EEPROM controller
    OnlyReti,                /* 0x0060 : TCA0_IRQ            */    //interrupt source:TCA0 16-bit timer
    OnlyReti,                /* 0x0062 : TCA1_IRQ            */    //interrupt source:TCA1 16-bit timer
    OnlyReti,                /* 0x0064 : STM0_IRQ            */
    OnlyReti,                /* 0x0066 : STM1_IRQ            */
    OnlyReti,                /* 0x0068 : AES_IRQ             */
    OnlyReti,                /* 0x006A : SHA_IRQ             */
    OnlyReti,                /* 0x006C : RX1_IRQ             */    //interrupt source:UART1 RX1
    OnlyReti,                /* 0x006E : TX1_IRQ             */    //interrupt source:UART1 TX1
    OnlyReti,                /* 0x0070 : I2C1_IRQ            */    //interrupt source:I2C1
    OnlyReti,                /* 0x0072 : SPI1_IRQ            */    //interrupt source:SPI1
    OnlyReti,                /* 0x0074 : RX2_IRQ             */    //interrupt source:UART2 RX2
    OnlyReti,                /* 0x0076 : TX2_IRQ             */    //interrupt source:UART2 TX2
    OnlyReti                 /* 0x0078 : TCA2_IRQ            */    //interrupt source:TCA2 16-bit timer
};
#pragma section const
/* -------------------- end of SQ7705 interrupt vector ---------------------- */

