//******************************************************************************
// file         : DRV_SQ7705_RTC.h
// version      : V1.2 2024/01/08
// description  : RTC header file
// note         : 
//******************************************************************************
#ifndef _DRV_SQ7705_RTC_H__
#define _SRV_SQ7705_RTC_H__

#include "TypeDefs.h"
#include "SQ7705_SFR.h"

extern uint8_t RTC_1sec_actFlag;          // flag for second interrupt
extern uint8_t RTC_alarm_actFlag;         // flag for alarm interrupt

#define RTC_TIE     0x08
#define RTC_SIE     0x04
#define RTC_AIE     0x02
#define RTC_WREIE   0x01

// user-defined type for perpetual calendar time
typedef struct{
    uint8_t sec;                   // second
    uint8_t min;                   // minute
    uint8_t hour;                  // hour
    uint8_t day;                   // day
    uint8_t week;                  // day of the week
    uint8_t mon;                   // month
    uint8_t year;                  // year
}RTC_PERPETUAL_CALENDAR_TIME;

// user-defined type for alarm time
typedef struct{
    uint8_t min;                   // minute
    uint8_t hour;                  // hour
    uint8_t day;                   // day
    uint8_t week;                  // day of thr week
}RTC_ALARM;

typedef enum{
    ONE_TIME_ONLY=0,               // one-shot mode
    CONTINUE_TIME_OUT              // continuous mode
} WORK_TYPE;

// when CLKSEL=00 (LXTAL/8)
// 00 244 us
// 01 15.6 ms
// 10 1 sec
// 11 1 min
typedef enum{
    RTCCLK_UNIT=0,
    RTCCLK_DIV64_UNIT,
    SEC_UNIT,
    MIN_UNIT
} TIMER_UNIT;

//******************************************************************************
// name         : DRV_RTC_IntEnable
// description  : enable RTC interrupts
// input param  : - intsel : RTC interrupt selection
//                #define RTC_TIE   0x08 enable RTC timeout interrupt
//                #define RTC_SIE   0x04 enable second interrupt
//                #define RTC_AIE   0x02 enable alarm interrupt
//                #define RTC_WREIE 0x01 enable WREN interrupt
// retval       : 
// note         : 
//******************************************************************************
void DRV_RTC_IntEnable(uint8_t intsel);

//******************************************************************************
// name         : DRV_RTC_IntDisable()
// description  : disable RTC interrupt
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_RTC_IntDisable();

//******************************************************************************
// name         : DRV_RTC_TimerStart()
// description  : set RTC timer function and start it
// input param  : - tUnit : when CLKSEL=00 (LXTAL/8)
//                          RTCCLK_UNIT 244 us
//                          RTCCLK_DIV64_UNIT 15.6 ms
//                          SEC_UNIT 1 sec
//                          MIN_UNIT 1 min
//                - timeOut : time for timeout 
//                - type : ONE_TIME_ONLY=0,   // one-shot mode
//                         CONTINUE_TIME_OUT  // continuous mode
// retval       : 
// note         : 
//******************************************************************************
void DRV_RTC_TimerStart(TIMER_UNIT tUnit, uint8_t timeOut, WORK_TYPE type);

//******************************************************************************
// name         : DRV_RTC_TimerStop()
// description  : stop RTC timer
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_RTC_TimerStop(void);

//******************************************************************************
// name         : DRV_RTC_SetDate()
// description  : set RTC time registers
// input param  : - prtc : set time in BCD format
//                      typedef struct{
//                          uint8_t sec;       // second
//                          uint8_t min;       // minute
//                          uint8_t hour;      // hour
//                          uint8_t day;       // day
//                          uint8_t week;      // day of the week
//                          uint8_t mon;       // month
//                          uint8_t year;      // year
//                      }RTC_PERPETUAL_CALENDAR_TIME;
// retval       : 
// note         : 
//******************************************************************************
void DRV_RTC_SetDate(RTC_PERPETUAL_CALENDAR_TIME *prtc);

//******************************************************************************
// name         : DRV_RTC_GetDate()
// description  : get current time
// input param  : - prtc : current time in BCD format
//                      typedef struct{
//                          uint8_t sec;       // second
//                          uint8_t min;       // minute
//                          uint8_t hour;      // hour
//                          uint8_t day;       // day
//                          uint8_t week;      // day of the week
//                          uint8_t mon;       // month
//                          uint8_t year;      // year
//                      }RTC_PERPETUAL_CALENDAR_TIME;
// retval       : 
// note         : 
//******************************************************************************
void DRV_RTC_GetDate(RTC_PERPETUAL_CALENDAR_TIME *prtc);

//******************************************************************************
// name         : DRV_RTC_SetAlarm()
// description  : set RTC alarm time
// input param  : - palarm : alarm time
//                      typedef struct{
//                          uint8_t min;       // minute
//                          uint8_t hour;      // hour
//                          uint8_t day;       // day
//                          uint8_t week;      // day of thr week
//                      }RTC_ALARM;
// retval       : 
// note         : it will transfer to BCD format
//******************************************************************************
void DRV_RTC_SetAlarm(RTC_ALARM* palarm);

//******************************************************************************
// name         : DRV_RTC_Start()
// description  : start RTC
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_RTC_Start(void);

//******************************************************************************
// name         : DRV_RTC_Stop()
// description  : stop RTC
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_RTC_Stop(void);

//******************************************************************************
// name         : DRV_RTC_IRQ()
// description  : TC interrupt service routine 
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_RTC_IRQ(void);

#endif

