//******************************************************************************
// file         : DRV_SQ7705_RTC.c
// version      : V1.2 2024/01/08
// description  : RTC related functions
// note         : RTC related functions are gathered in this subroutine
//******************************************************************************
#include "DRV_SQ7705_RTC.h"

uint8_t RTC_1sec_actFlag;          // flag for second interrupt
uint8_t RTC_alarm_actFlag;         // flag for alarm interrupt

//******************************************************************************
// name         : DRV_RTC_IntEnable
// description  : enable RTC interrupts
// input param  : - intsel : RTC interrupt selection
//                #define RTC_TIE   0x08 enable RTC timeout interrupt
//                #define RTC_SIE   0x04 enable second interrupt
//                #define RTC_AIE   0x02 enable alarm interrupt
//                #define RTC_WREIE 0x01 enable WREN interrupt
// retval       : 
// note         : 
//******************************************************************************
void DRV_RTC_IntEnable(uint8_t intsel)
{
    __ASM("DI");                            // disable all maskable interrupts
    IER_RTC = 1;
    RTCCR1 |= intsel;
    __ASM("EI");                            // enable all maskable interrupts
}

//******************************************************************************
// name         : DRV_RTC_IntDisable()
// description  : disable RTC interrupt
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_RTC_IntDisable()
{
    __ASM("DI");                            // disable all maskable interrupts
    IER_RTC = 0;
    RTCCR1 =  0x00;
    __ASM("EI");                            // enable all maskable interrupts
}

//******************************************************************************
// name         : DRV_RTC_TimerStart()
// description  : set RTC timer function and start it
// input param  : - tUnit : when CLKSEL=00 (LXTAL/8)
//                          RTCCLK_UNIT 244 us
//                          RTCCLK_DIV64_UNIT 15.6 ms
//                          SEC_UNIT 1 sec
//                          MIN_UNIT 1 min
//                - timeOut : time for timeout 
//                - type : ONE_TIME_ONLY=0,   // one-shot mode
//                         CONTINUE_TIME_OUT  // continuous mode
// retval       : 
// note         : 
//******************************************************************************
void DRV_RTC_TimerStart(TIMER_UNIT tUnit, uint8_t timeOut, WORK_TYPE type)
{
    RTCCR1 |= 0x0C;                         // enable RTC timeout interrupt (RTCCR1_TOIE = 1)
    RTCTMRCR_TMRSEL = tUnit;                // timer clock selection
    RTCTMRCR_CONT = type;                   // set timer continuous mode
    RTCTMRTO = timeOut;                     // RTC timeout value (must larger than 0)
    RTCTMRCR_TMREN = 1;
}

//******************************************************************************
// name         : DRV_RTC_TimerStop()
// description  : stop RTC timer
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_RTC_TimerStop(void)
{
    RTCTMRCR_TMREN = 0;                     // disable RTC timer function

    while (RTCTMRCR_TMRST != 0);            // make sure timer stop
}

//******************************************************************************
// name         : DRV_RTC_SetDate()
// description  : set RTC time registers
// input param  : - prtc : set time in BCD format
//                      typedef struct{
//                          uint8_t sec;       // second
//                          uint8_t min;       // minute
//                          uint8_t hour;      // hour
//                          uint8_t day;       // day
//                          uint8_t week;      // day of the week
//                          uint8_t mon;       // month
//                          uint8_t year;      // year
//                      }RTC_PERPETUAL_CALENDAR_TIME;
// retval       : 
// note         : 
//******************************************************************************
void DRV_RTC_SetDate(RTC_PERPETUAL_CALENDAR_TIME *prtc)
{
    RTCSEC = ((prtc->sec / 10) << 4) | (prtc->sec % 10);           // RTC second,  0~59
    RTCMIN = ((prtc->min / 10) << 4) | (prtc->min % 10);           // RTC minute,  0~59
    RTCHR = ((prtc->hour / 10) << 4) | (prtc->hour % 10);          // RTC hour, 0~23
    RTCDAY = ((prtc->day / 10) << 4) | (prtc->day % 10);           // RTC day, 1~31
    RTCWDAY = (prtc->week & 0x7);                                  // RTC days of the week
    RTCMONTH = ((prtc->mon / 10) << 4) | (prtc->mon % 10);         // RTC month, 1~12
    RTCYEAR = ((prtc->year % 100 / 10) << 4) | (prtc->year % 10);  // RTC year, 1~99
}

//******************************************************************************
// name         : DRV_RTC_GetDate()
// description  : get current time
// input param  : - prtc : current time in BCD format
//                      typedef struct{
//                          uint8_t sec;       // second
//                          uint8_t min;       // minute
//                          uint8_t hour;      // hour
//                          uint8_t day;       // day
//                          uint8_t week;      // day of the week
//                          uint8_t mon;       // month
//                          uint8_t year;      // year
//                      }RTC_PERPETUAL_CALENDAR_TIME;
// retval       : 
// note         : 
//******************************************************************************
void DRV_RTC_GetDate(RTC_PERPETUAL_CALENDAR_TIME *prtc)
{
    prtc->sec = (((RTCSEC & 0x70) >> 4) * 10) + (RTCSEC & 0x0F);
    prtc->min = (((RTCMIN & 0x70) >> 4) * 10) + (RTCMIN & 0x0F);
    prtc->hour = (((RTCHR & 0x30) >> 4) * 10) + (RTCHR & 0x0F);
    prtc->day = (((RTCDAY & 0x30) >> 4) * 10) + (RTCDAY & 0x0F);
    prtc->week = (RTCWDAY & 0x0F);
    prtc->mon = (((RTCMONTH & 0x10) >> 4) * 10) + (RTCMONTH & 0x0F);
    prtc->year = (((RTCYEAR & 0xF0) >> 4) * 10) + (RTCYEAR & 0x0F);
}

//******************************************************************************
// name         : DRV_RTC_SetAlarm()
// description  : set RTC alarm time
// input param  : - palarm : alarm time
//                      typedef struct{
//                          uint8_t min;       // minute
//                          uint8_t hour;      // hour
//                          uint8_t day;       // day
//                          uint8_t week;      // day of thr week
//                      }RTC_ALARM;
// retval       : 
// note         : it will transfer to BCD format
//******************************************************************************
void DRV_RTC_SetAlarm(RTC_ALARM *palarm)
{
    // remember enable alarm interrupt first
    RTCALMIN = ((palarm->min / 10) << 4) | (palarm->min % 10);     // set RTC alarm minute
    RTCALHR = ((palarm->hour / 10) << 4) | (palarm->hour % 10);    // set RTC alarm hour
    RTCALDAY = ((palarm->day / 10) << 4) | (palarm->day % 10);     // set RTC alarm day
    RTCALWDAY = ((palarm->week / 10) << 4) | (palarm->week % 10);  // set RTC alarm day of the week
}

//******************************************************************************
// name         : DRV_RTC_Start()
// description  : start RTC
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_RTC_Start(void)
{
    RTCCR0_RTCEN = 1;                         // enalbe RTC
}

//******************************************************************************
// name         : DRV_RTC_Stop()
// description  : stop RTC
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_RTC_Stop(void)
{
    RTCCR0_RTCEN = 0;                         // disable RTC
}

//******************************************************************************
// name         : DRV_RTC_IRQ()
// description  : TC interrupt service routine 
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_RTC_IRQ(void)
{
    __asm("NOP");                             // no operation
    //--- second interrupt ---
    if (RTCCR1_SECF == 1 ) {                  // second interrupt is generated1
        // RTCCR1 = (RTCCR1&0x0F)|0x40;       // clear second interrupt flag
        RTCCR1_SECF = 0;
        RTC_1sec_actFlag = 1;                 // set RTC_1sec_actFlag to 1
    }
    //--- alarm interrupt ---
    if (RTCCR1_ALF == 1) {                    // alarm interrupt is generated
        RTCCR1 = (RTCCR1 & 0x0F) | 0x20;      // clear alarm interrupt flag
        RTC_alarm_actFlag = 1;                // set RTC_alarm_actFlag to 1
        RTCCR1_ALF = 0;
    }
    //--- timer overtime interrupt ---
    if (RTCCR1_TOF == 1) {
        P2DO_P4 = 1;
        P2DO_P4 = 0;
        // timer overtime interrupt is generated
        RTCCR1 = (RTCCR1 & 0x0F) | 0x80;      // clear timer overtime interrupt flag
        RTCCR1_TOF = 0;
    }
}

