//******************************************************************************
// file         : DEV_SQ7705_LVD.c
// version      : V1.2 2024/01/05
// description  : LVD related functions
// note         : LVD related functions are gathered in this subroutine
//******************************************************************************
#include "DRV_SQ7705_LVD.h"

uint8_t LVD_Int_Flag;

//******************************************************************************
// name         : DRV_LVD_Init()
// description  : initialize LVD
// input param  : - lvdCfg : select voltage detection level
//                      LVD_2_00_V  2.00V detection level
//                      LVD_2_35_V  2.35V detection level
//                      LVD_2_65_V  2.65V detection level
//                      LVD_2_85_V  2.85V detection level
//                      LVD_3_15_V  3.15V detection level
//                      LVD_3_98_V  3.98V detection level
//                      LVD_4_2_V   4.2V  detection level
//                      LVD_4_5_V   4.5V  detection level
//                - act : action when low voltage is detected
//                        ACT_INERRUPT : LVD interrupt is generated
//                        ACT_REST     : reset
// retval       : none (all parameters can only use the defined values, otherwise the function will be disabled)
// note         : 
//******************************************************************************
void DRV_LVD_Init(LVD_CFG lvdCfg, LVD_REACTION act)
{
    __ASM("DI");

    if (lvdCfg <= LVD_2_85_V) {         // LVD0
        IER_LVD0  = 1;                  // disable LVD0 interrupt
        LVDCR0_LVDRSTEN = act;          // when low voltage is detected, 0: Interrupt, 1: Reset.
        LVDCR0_LVDCFG = lvdCfg;         // select voltage detection level
        LVDCR0_LVDIEN = 1;              // enable LVD0
    } else {
        IER_LVD1  = 1;
        LVDCR1_LVDRSTEN = act;          // when low voltage is detected, 0: Interrupt, 1: Reset.
        LVDCR1_LVDCFG = lvdCfg;         // select voltage detection level
        LVDCR1_LVDIEN = 1;
    }

    __ASM("EI");
}

//******************************************************************************
// name         : DRV_LVD_DeInit()
// description  : Deinitialize LVD0 or LVD1
// input param  : - lvdType : LVD level type
//                      LVD_2V_TO_3V for LVD_2_00_V ~ LVD_2_85_V
//                      LVD_3V_TO_4V for LVD_3_15_V ~ LVD_4_5_V
// retval       : none (all parameters can only use the defined values, otherwise the function will be disabled)
// note         : 
//******************************************************************************
void DRV_LVD_DeInit(LVD_TYPE lvdType)
{
    if (lvdType == LVD_2V_TO_3V) {      // LVD0
        LVDCR0_LVDEN = 0;               // disable LVD0
        IER_LVD0 = 0;                   // disable LVD0 interrupt
    } else {
        LVDCR1_LVDEN = 0;               // disable LVD1
        IER_LVD1 = 0;                   // disable LVD1 interrupt
    }
}

//******************************************************************************
// name         : DRV_LVD_Start()
// description  : start LVD0 or LVD1
// input param  : - lvdType : LVD level type
//                      LVD_2V_TO_3V for LVD_2_00_V ~ LVD_2_85_V
//                      LVD_3V_TO_4V for LVD_3_15_V ~ LVD_4_5_V
// retval       : none (all parameters can only use the defined values, otherwise the function will be disabled)
// note         : 
//******************************************************************************
void DRV_LVD_Start(LVD_TYPE lvdType)
{
    if (lvdType == LVD_2V_TO_3V) {      // LVD0
        LVDCR0_LVDEN = 1;
    } else {                            // LVD1
        LVDCR1_LVDEN = 1;
    }
}

//******************************************************************************
// name         : DRV_LVD_Stop()
// description  : stop LVD0 or LVD1
// input param  : - lvdType : LVD level type
//                      LVD_2V_TO_3V for LVD_2_00_V ~ LVD_2_85_V
//                      LVD_3V_TO_4V for LVD_3_15_V ~ LVD_4_5_V
// retval       : none (all parameters can only use the defined values, otherwise the function will be disabled)
// note         : 
//******************************************************************************
void DRV_LVD_Stop(LVD_TYPE lvdType)
{
    if (lvdType == LVD_2V_TO_3V) {      // LVD0
        LVDCR0_LVDEN = 0;               // disable LVD0
    } else {
        LVDCR1_LVDEN = 0;               // disable LVD1
    }
}

//******************************************************************************
// name         : DRV_LVD0_IRQ()
// description  : LVD0 interrupt service routine
// retval       : none
// note         : 
//******************************************************************************
void __interrupt DRV_LVD0_IRQ(void)
{
    __asm("NOP");                       // no operation
    LVD_Int_Flag = 1;
}

//******************************************************************************
// name         : DRV_LVD1_IRQ()
// description  : LVD1 interrupt service routine
// retval       : none
// note         : 
//******************************************************************************
void __interrupt DRV_LVD1_IRQ(void)
{
    __asm("NOP");                       // no operation
    LVD_Int_Flag = 1;
}

