//******************************************************************************
// file         : ADM_main.c
// version      : V1.3  2023/11/20
// description  : LED1 (P2.0) blinks continuously when the VDD voltage is above the set low voltage detection level
//                when low voltage detection occurs, LED1 (P2.0) turns off and stops blinking
// note         : when using this program, the VDD value must be within the specification voltage of SQ7705
// example      : at the beginning of this program, the low voltage detection level is 2.8V, LED1 will be off
//                when VDD voltage is higher than 2.8V, LED1 will blink continuously
//******************************************************************************
#ifndef IO_MEM
#define IO_MEM
#endif
#include "ADM_main.h"
#include "DRV_SQ7705_SYSCLK.h"
#include "DRV_SQ7705_GPIO.h"
#include "DRV_SQ7705_TBT.h"
#include "DRV_SQ7705_LVD.h"

/* --------------------------- global variables ----------------------------- */
TBT_REG tbtStruct;                              // TBT data structure
unsigned char LVD_switch;                       // for switching low voltage detection levels

/* ---------------------------- main program -------------------------------- */
void ADM_Main()
{
    DRV_Clock_Init(Clk_Source_SYS, 1);                  // PLL 24MHz Initial system clock.
    /* ----------------------- GPIO initialization -------------------------- */
    DRV_GPIO_Init(GPIO_P20, GPIO_OUTPUT, GPIO_PULL_UP); // set P2.0 as output pin
                                                        // enable P2.0 built-in pull-up resistor
    DRV_GPIO_WritePin(GPIO_P20, GPIO_PIN_HIGH);         // set LED1(P2.0) to light off
    /* ----------------------- TBT initialization --------------------------- */
    tbtStruct.tbtDV9CK = DV9CK_FSYSCLK;                 // selects the input clock to stage 9 of the divider:fsysclk/2^9
    tbtStruct.tbtTBTCK = TBTCK_1000;                    // selects the time base timer interrupt frequency:fsysclk/2^22
    DRV_TBT_Init(tbtStruct);                            // TBT initialization
    DRV_TBT_Start();                                    // enable TBT
    /* ----------------------- LVD initialization --------------------------- */
    DRV_LVD_Init(LVD_2_85_V, ACT_INERRUPT);
    DRV_LVD_Start(LVD_2V_TO_3V);

    while (1) {
        if (TBT_Int_Flag) {                                  // if TBT interrupt flag is 1
            TBT_Int_Flag = 0;                                // set TBT_Int_Flag to 0

            if (LVD_Int_Flag == 0) {                         // if LVD_Int_Flag is 0
                DRV_GPIO_TogglePin(GPIO_P20);                // LED1(P2.0) inverted
            } else {                                         // if LVD_Int_Flag is 1
                DRV_GPIO_WritePin(GPIO_P20, GPIO_PIN_HIGH);  // light off LED1
            }
        }
    }
}

/* ----------------------- interrupt service routine ------------------------ */
void __interrupt OnlyReti(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n OnlyRetn(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n WDT_IRQ(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n SWI_IRQ(void)               // software reset interrupt service routine
{
    SWRST = 0x5a;
    SWRST = 0xa5;
    SWRST = 0xc3;
    SWRST = 0x3c;
}

void __interrupt IntADM_Req(void)
{

}

/* ---------------------- end of interrupt service routine ------------------ */


/* ----------------------- SQ7705 interrupt vector -------------------------- */
#pragma section const ADM_INT_VECTOR
void * const IntTbl1[] = {
    ADM_STARTUP,             /* 0x0000 : RESET               */    //non-maskable
    SWI_IRQ,                 /* 0x0002 : SWI_IRQ/INTUNDEF    */    //non-maskable
    (void *)0xffff,          /* 0x0004 : Reserved            */    
    WDT_IRQ,                 /* 0x0006 : WDT_IRQ             */    //non-maskable interrupt source:watchdog timer
    OnlyReti,                /* 0x0008 : TAMPER_IRQ          */    //interrupt source:tamper detection
    OnlyReti,                /* 0x000A : CFD_IRQ             */    //interrupt source:clock faildetection
    OnlyReti,                /* 0x000C : HVD_IRQ             */
    IntADM_Req,              /* 0x000E : U2A_IRQ/IntADM_Req  */    //interrupt source:user mode to admin mode
    OnlyReti,                /* 0x0010 : RTC_IRQ             */    //interrupt source:real-time clock timer
    OnlyReti,                /* 0x0012 : TMRA00_IRQ          */    //interrupt source:TMRA00 8-bit timer
    OnlyReti,                /* 0x0014 : TMRA01_IRQ          */    //interrupt source:TMRA01 8-bit timer
    OnlyReti,                /* 0x0016 : TMRA10_IRQ          */    //interrupt source:TMRA10 8-bit timer
    OnlyReti,                /* 0x0018 : TMRA11_IRQ          */    //interrupt source:TMRA11 8-bit timer
    OnlyReti,                /* 0x001A : TMRA20_IRQ          */
    OnlyReti,                /* 0x001C : TMRA21_IRQ          */
    DRV_TBT_IRQ,             /* 0x001E : TBT_IRQ             */
    OnlyReti,                /* 0x0020 : RX0_IRQ             */    //interrupt source:UART0 RX0
    OnlyReti,                /* 0x0022 : TX0_IRQ             */    //interrupt source:UART0 TX0
    OnlyReti,                /* 0x0024 : I2C0_IRQ            */    //interrupt source:I2C0
    OnlyReti,                /* 0x0026 : SPI0_IRQ            */    //interrupt source:SPI0
    OnlyReti,                /* 0x0028 : EXT0_IRQ            */    //external interrupt0
    OnlyReti,                /* 0x002A : EXT1_IRQ            */    //external interrupt1
    OnlyReti,                /* 0x002C : EXT2_IRQ            */    //external interrupt2
    OnlyReti,                /* 0x002E : EXT3_IRQ            */    //external interrupt3
    OnlyReti,                /* 0x0030 : ADC_IRQ             */    //interrupt source:ADC
    DRV_LVD0_IRQ,            /* 0x0032 : LVD0_IRQ            */
    DRV_LVD1_IRQ,            /* 0x0034 : LVD1_IRQ            */
    OnlyReti,                /* 0x0036 : Reserved            */
    OnlyReti,                /* 0x0038 : Reserved            */
    OnlyReti,                /* 0x003A : FMC_IRQ             */    //interrupt source:flash memory controller
    OnlyReti,                /* 0x003C : DMACH0_IRQ          */
    OnlyReti,                /* 0x003E : DMACH1_IRQ          */
    OnlyReti,                /* 0x0040 : DMACH2_IRQ          */
    OnlyReti,                /* 0x0042 : DMACH3_IRQ          */
    OnlyReti,                /* 0x0044 : Reserved            */
    OnlyReti,                /* 0x0046 : Reserved            */
    OnlyReti,                /* 0x0048 : Reserved            */
    OnlyReti,                /* 0x004A : Reserved            */
    OnlyReti,                /* 0x004C : Reserved            */
    OnlyReti,                /* 0x004E : DIC_IRQ             */    //interrupt source:DIC
    OnlyReti,                /* 0x0050 : LEUART_RX_IRQ       */
    OnlyReti,                /* 0x0052 : LEUART_TX_IRQ       */
    OnlyReti,                /* 0x0054 : TRNG_IRQ            */
    OnlyReti,                /* 0x0056 : EXT4_IRQ            */    //external interrupt4
    OnlyReti,                /* 0x0058 : EXT5_IRQ            */    //external interrupt5
    OnlyReti,                /* 0x005A : EXT6_IRQ            */    //external interrupt6
    OnlyReti,                /* 0x005C : EXT7_IRQ            */    //external interrupt7
    OnlyReti,                /* 0x005E : EEMC_IRQ            */    //interrupt source:EEPROM controller
    OnlyReti,                /* 0x0060 : TCA0_IRQ            */    //interrupt source:TCA0 16-bit timer
    OnlyReti,                /* 0x0062 : TCA1_IRQ            */    //interrupt source:TCA1 16-bit timer
    OnlyReti,                /* 0x0064 : STM0_IRQ            */
    OnlyReti,                /* 0x0066 : STM1_IRQ            */
    OnlyReti,                /* 0x0068 : AES_IRQ             */
    OnlyReti,                /* 0x006A : SHA_IRQ             */
    OnlyReti,                /* 0x006C : RX1_IRQ             */    //interrupt source:UART1 RX1
    OnlyReti,                /* 0x006E : TX1_IRQ             */    //interrupt source:UART1 TX1
    OnlyReti,                /* 0x0070 : I2C1_IRQ            */    //interrupt source:I2C1
    OnlyReti,                /* 0x0072 : SPI1_IRQ            */    //interrupt source:SPI1
    OnlyReti,                /* 0x0074 : RX2_IRQ             */    //interrupt source:UART2 RX2
    OnlyReti,                /* 0x0076 : TX2_IRQ             */    //interrupt source:UART2 TX2
    OnlyReti                 /* 0x0078 : TCA2_IRQ            */    //interrupt source:TCA2 16-bit timer
};
#pragma section const
/* -------------------- end of SQ7705 interrupt vector ---------------------- */

