//******************************************************************************
// file         : DRV_SQ7705_LEUART.c
// version      : V1.3 (2025/03/13)
// description  : LEUART related functions
// note         : LEUART related functions are gathered in this subroutine
//******************************************************************************
#include <stdlib.h>
#include "DRV_SQ7705_LEUART.h"

static LEUART_CALLBACK LEUART_TX_CallBackFunc;  // TX callback function
static LEUART_CALLBACK LEUART_RX_CallBackFunc;  // RX callback function

uint8_t rxIntFlag;        // 0:init value.  1:Rx interrupt occurs
uint8_t rxRecvData;       // RX IRQ received 1-byte data

//******************************************************************************
// name         : DRV_LEUART_Init()
// description  : LEUART initialization
// input param  : ioSel : LEUART port pin selection(must these TxRx pair)
// retval       : 
// note         : 
//******************************************************************************
void DRV_LEUART_Init(LEUART_IO_LIST ioSel)
{
    LEUART_TX_CallBackFunc = NULL;
    LEUART_RX_CallBackFunc = NULL; 

    PCKEN1_LEUART = 1;      // enable LEUART clock
    switch(ioSel) {
        case LEUART_RX_P05_TX_P06:
            P0CFGCR = 0xD2;
            P0CFGCR = 0xE2;
            break;
        case LEUART_RX_P12_TX_P13:
            P1CFGCR = 0xA2;
            P1CFGCR = 0xB2;
            break;
        case LEUART_RX_P24_TX_P25:
            P2CFGCR = 0xC2;
            P2CFGCR = 0xD2;
            break;
    }

    LEUARTCR1_TXE = 0;     // disable TX
    LEUARTCR1_RXE = 0;     // disable RX
    __asm("DI");
    IFR_LEUART_TX = 0;          // clear LEUART interrupt flag
    IER_LEUART_RX = 1;          // enable LEUART RX IRQ
    IER_LEUART_TX = 1;          // enable LEUART TX IRQ
    __asm("EI");
}

//******************************************************************************
// name         : DRV_LEUART_BaudRate()
// description  : set Baud rate
// input param  : clockSel : clock source selection
//                baudrate : Baud rate selection
// retval       :  
// note         : 
//******************************************************************************
void DRV_LEUART_BaudRate(LEUART_CLOCK_SOURCE_LIST clockSel, uint16_t baudrate)
{
    uint16_t t  = baudrate;

    if(clockSel == 0) {
        LEUARTCR1_CLKSEL = 0; 
        LEUARTCR2_RTSEL = baudrate & 0x000F;
        LEUARTDR = baudrate >> 8 ;
    } else {
        LEUARTCR1_CLKSEL = 1;
        LEUARTDR = baudrate;
    }
}

//******************************************************************************
// name         : DRV_LEUART_DeInit()
// description  : LEUART deinitialization
// input param  : ioSel : io function selection
// retval       :  
// note         : 
//******************************************************************************
void DRV_LEUART_DeInit(LEUART_IO_LIST ioSel)
{
    PCKEN1_LEUART = 0;        // disable LEUART clock
    switch(ioSel) {
        case LEUART_RX_P05_TX_P06:
            P0CFGCR = 0xD0;   // IO port set to GPIO 
            P0CFGCR = 0xE0;
            break;
        case LEUART_RX_P12_TX_P13:
            P1CFGCR = 0xA0;
            P1CFGCR = 0xB0;
            break;
        case LEUART_RX_P24_TX_P25:
            P2CFGCR = 0xC0;
            P2CFGCR = 0xD0;
            break;
    }

    LEUARTCR1_TXE = 0;       // disable TX
    LEUARTCR1_RXE = 0;
    __asm("DI");
    IFR_LEUART_TX = 0;
    IER_LEUART_RX = 0;
    IER_LEUART_TX = 0;
    __asm("EI");
    LEUARTIER = 0x00;
}

//******************************************************************************
// name         : DRV_LEUART_TX_IRQ_Init()
// description  : enable LEUART TX interrupt (select interrupt mode)
// input param  : irq  : LEUART TX interrupt mode selection
//                txCB : TX callback function
// retval       :
// note         : 
//******************************************************************************
void DRV_LEUART_TX_IRQ_Init(LEUART_TXIRQEN_LIST irq, LEUART_CALLBACK txCB)
{
    LEUART_TX_CallBackFunc = txCB;  // set callback function
    LEUARTIER |= irq;               // select interrupt mode
}

//******************************************************************************
// name         : DRV_LEUART_RX_IRQ_Init()
// description  : enable LEUART RX interrupt (select interrupt mode)
// input param  : irq  : LEUART RX interrupt selection
//                rxCB : RX callback function
// retval       :  
// note         : 
//******************************************************************************
void DRV_LEUART_RX_IRQ_Init(LEUART_RXIRQEN_LIST irq, LEUART_CALLBACK rxCB)
{
    LEUART_RX_CallBackFunc = rxCB;
    LEUARTIER |= irq;
}

//******************************************************************************
// name         : DRV_LEUART_Start()
// description  : start LEUART
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_LEUART_Start()
{
    LEUARTCR1_TXE = 1;                   // enable TX 
    LEUARTCR1_RXE = 1;
}

//******************************************************************************
// name         : DRV_LEUART_Stop()
// description  : stop LEUART
// input param  : 
// retval       : 
// note         :
//******************************************************************************
void DRV_LEUART_Stop()
{
    LEUARTCR1_TXE = 0;                   // disable TX 
    LEUARTCR1_RXE = 0;
}

//******************************************************************************
// name         : DRV_LEUART_SendBytes()
// description  : TX transmits bytes
// input param  : buf : the bytes to be transferred
//                lenght : the length of data
// retval       : 
// note         : 
//******************************************************************************
void DRV_LEUART_SendBytes(uint8_t *buf, uint16_t lenght)
{
    uint16_t idx = 0;
    for(idx = 0;idx<lenght;idx++) {
        // wait for TX buffer empty
        while(LEUARTTXST_TXEMPTY == 0) 
            CLR_WDT;

        LEUARTBUF = buf[idx];
        // 1: busy , 0: transfer completed or before transfer
        while(LEUARTTXST_TXBUSY == 1)
            CLR_WDT;
    }
}

//******************************************************************************
// name         : DRV_LEUART_SendByte()
// description  : LEUART TX transmits a single byte
// input param  : data : one byte data to be transmitted by TX
// retval       : 
// note         : 
//******************************************************************************
void  DRV_LEUART_SendByte(uint8_t data)
{
    // wait for TX buffer empty
    while(LEUARTTXST_TXEMPTY == 0) 
        CLR_WDT;
    
    LEUARTBUF = data;
}

//******************************************************************************
// name         : DRV_LERX0_IRQ()
// description  : LEUART0 RX IRQ functiomn
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_LEUART_RX_IRQ(void)
{
    rxRecvData = LEUARTBUF;
    rxIntFlag = 1;   // RX interrupt  occurs 
    if (LEUART_RX_CallBackFunc != NULL) 
        LEUART_RX_CallBackFunc();
}

//******************************************************************************
// name         : DRV_LETX0_IRQ()
// description  : LEUART0 TX IRQ functiomn
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void __interrupt DRV_LEUART_TX_IRQ(void)
{
    __ASM("NOP"); 
}

