//******************************************************************************
// file         : main.c
// version      : V1.3  2023/11/20
// description  : main program
// note         : this example program is a demostration for LEUART TX sending data to RX within a single SQ7705 chip
//                when press button 1, TX sends data 0x55, if RX is received correctly then light on LED1
//                when press button 2, TX sends data 0xAA, if RX is received correctly then light on LED2
//                when press button 3, TX sends multiple datas, if RX is received correctly then light on LED3
//                not yet communicating or receiving error then not light on LED
// 
//                1. TX and RX configuration
//                     set P1.3 to TX
//                     set P1.2 to RX
//
//                2. TX(P1.3) connects to RX(P1.2)
//
//                3. [Button Configuration] 
//                       button 1 :  BTN1_in(P5.0)
//                       button 2 :  BTN2_in(P5.1)
//                       button 3 :  BTN3_in(P5.2)
// 
//                   [LED Configuration] LED 1  (P2.0)
//                   [LED Configuration] LED 2  (P2.1)
//                   [LED Configuration] LED 3  (P2.2)
// 
//                4. UART1 configuration : Baud rate to 9600, stop bit to 1 bit, no parity
//******************************************************************************
#ifndef IO_MEM
#define IO_MEM
#endif
#include "ADM_main.h"
#include "DRV_SQ7705_SYSCLK.h"
#include "DRV_SQ7705_LEUART.h"

/* --------------------------- global variables ----------------------------- */
uint16_t rxRecvCount = 0;    // LEUART RX receive bytes count
uint8_t sendBuf1[12]={0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x77, 0x05, 0x5A, 0xA5};
uint8_t rxRecvBuf[12]={0};

/* ------------------- function prototype declaration ----------------------- */
void Check_Recive_Data_LED();
void CallBack_LEUART_RX(void);        // LEUART RX interrupt callback function
void CallBack_LEUART_TX(void);        // LEUART TX interrupt callback function

/* ---------------------------- main program -------------------------------- */
void ADM_Main()
{
    DRV_Clock_Init(Clk_Source_SYS, 1);        // Initial system clock (PLL 48MHz/2 = 24MHz)
    DRV_LXTAL_Init();                         // external XTAL 32.768 kHz initialize

    /* ------------------------ GPIO initialization ------------------------- */
    DRV_GPIO_Init(GPIO_P20, GPIO_OUTPUT, GPIO_PULL_UP); // set P2.0 as output pin
                                                        // enable P2.0 built-in pull-up resistor
    DRV_GPIO_Init(GPIO_P21, GPIO_OUTPUT, GPIO_PULL_UP); // set P2.1 as output pin
                                                        // enable P2.1 built-in pull-up resistor
    DRV_GPIO_Init(GPIO_P22, GPIO_OUTPUT, GPIO_PULL_UP); // set P2.2 as output pin
                                                        // enable P2.2 built-in pull-up resistor
    DRV_GPIO_Init(GPIO_P23, GPIO_OUTPUT, GPIO_PULL_UP); // set P2.3 as output pin
                                                        // enable P2.3 built-in pull-up resistor
    LED1_to_3_turn_off;
    
    DRV_GPIO_Init(GPIO_P50, GPIO_INPUT, GPIO_PULL_UP);  // set P5.0 as input pin
                                                        // enable P5.0 built-in pull-up resistor
    DRV_GPIO_Init(GPIO_P51, GPIO_INPUT, GPIO_PULL_UP);  // set P5.1 as input pin
                                                        // enable P5.1 built-in pull-up resistor
    DRV_GPIO_Init(GPIO_P52, GPIO_INPUT, GPIO_PULL_UP);  // set P5.2 as input pin
                                                        // enable P5.2 built-in pull-up resistor
    /*  ----------------------- LEUART initialization ----------------------- */
    DRV_LEUART_Init(LEUART_RX_P12_TX_P13);              // set RX:P1.2,TX:P1.3
    DRV_LEUART_BaudRate(LEAURT_Use_ExternalLowSpeedClock, OperatLSCLK_Baud9600);
    DRV_LEUART_RX_IRQ_Init(LEUART_RXIRQ_RXBYTEIEN, CallBack_LEUART_RX);  // enable receive single byte interrupt
    DRV_LEUART_Start();                                 // enable LEUART

    /* ---------------- Use button to test UAER send/receive ---------------- */
    while(1){
        if(BTN1_in == 0){                        // press button 1
            rxRecvCount = 0;            
            DRV_LEUART_SendByte(0x55);           // TX1 send 0x55
            while(BTN1_in == 0);
        } else if(BTN2_in == 0){ 
            rxRecvCount = 0;
            DRV_LEUART_SendByte(0xAA);           // TX send 0xAA
            while(BTN2_in == 0);
        } else if(BTN3_in == 0){                 // press button 3
            rxRecvCount = 0;
            DRV_LEUART_SendBytes(sendBuf1, 12);  // TX send 12-bytes
            while(BTN3_in == 0);
        }

        Check_Recive_Data_LED();                 // compare the received data, if it is correct 
                                                 // then light on the corresponding LED
        LED1_to_3_turn_off;                      // light off all LEDs
    }
}

//******************************************************************************
// name         : Check_Recive_Data_LED()
// description  : check LEUART receive data and turn on LED
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void Check_Recive_Data_LED()
{
    uint8_t i,check_result;
    check_result = 0;

    if(rxRecvBuf[0] == 0x55){
        LED1_to_3_turn_off;
        LED1_on;                             // receive 0x55, LED1(P2.0) on
    }
    else if(rxRecvBuf[0] == 0xAA){
        LED1_to_3_turn_off;             
        LED2_on;                             // receive 0xAA, LED2(P2.1) on
    }
    else if(rxRecvBuf[11] == 0xA5){
        LED1_to_3_turn_off;

        // compare rxRecvBuf[] and sendBuf1[]
        for (i=0; i<12; i++){
            if(rxRecvBuf[i] == sendBuf1[i]){
                check_result++;
            }
        
            if(check_result==12){
                LED3_on;                     // all correct, LED3(P2.2) on    
            }
        }
    }
}

//******************************************************************************
// name         : CallBack_LEUART_RX()
// description  : LEUART0_RX interrupt callback function
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void CallBack_LEUART_RX(void)
{
    rxRecvBuf[rxRecvCount] = rxRecvData;   // get LEUART0 receive one byte data 
    rxRecvCount++;
}

//******************************************************************************
// name         : CallBack_LEUART_TX()
// description  : LEUART0_TX interrupt callback function
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void CallBack_LEUART_TX(void)
{
    __ASM("NOP");
}

/* ----------------------- interrupt service routine ------------------------ */
void __interrupt OnlyReti(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n OnlyRetn(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n WDT_IRQ(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n SWI_IRQ(void)               // software reset interrupt service routine
{
    SWRST = 0x5a;
    SWRST = 0xa5;
    SWRST = 0xc3;
    SWRST = 0x3c;
}

void __interrupt IntADM_Req(void)
{

}

/* ---------------------- end of interrupt service routine ------------------ */


/* ----------------------- SQ7705 interrupt vector -------------------------- */
#pragma section const ADM_INT_VECTOR
void * const IntTbl1[] = {
    ADM_STARTUP,             /* 0x0000 : RESET               */    //non-maskable
    SWI_IRQ,                 /* 0x0002 : SWI_IRQ/INTUNDEF    */    //non-maskable
    (void *)0xffff,          /* 0x0004 : Reserved            */    
    WDT_IRQ,                 /* 0x0006 : WDT_IRQ             */    //non-maskable interrupt source:watchdog timer
    OnlyReti,                /* 0x0008 : TAMPER_IRQ          */    //interrupt source:tamper detection
    OnlyReti,                /* 0x000A : CFD_IRQ             */    //interrupt source:clock faildetection
    OnlyReti,                /* 0x000C : HVD_IRQ             */
    IntADM_Req,              /* 0x000E : U2A_IRQ/IntADM_Req  */    //interrupt source:user mode to admin mode
    OnlyReti,                /* 0x0010 : RTC_IRQ             */    //interrupt source:real-time clock timer
    OnlyReti,                /* 0x0012 : TMRA00_IRQ          */    //interrupt source:TMRA00 8-bit timer
    OnlyReti,                /* 0x0014 : TMRA01_IRQ          */    //interrupt source:TMRA01 8-bit timer
    OnlyReti,                /* 0x0016 : TMRA10_IRQ          */    //interrupt source:TMRA10 8-bit timer
    OnlyReti,                /* 0x0018 : TMRA11_IRQ          */    //interrupt source:TMRA11 8-bit timer
    OnlyReti,                /* 0x001A : TMRA20_IRQ          */
    OnlyReti,                /* 0x001C : TMRA21_IRQ          */
    OnlyReti,                /* 0x001E : TBT_IRQ             */
    OnlyReti,                /* 0x0020 : RX0_IRQ             */    //interrupt source:UART0 RX0
    OnlyReti,                /* 0x0022 : TX0_IRQ             */    //interrupt source:UART0 TX0
    OnlyReti,                /* 0x0024 : I2C0_IRQ            */    //interrupt source:I2C0
    OnlyReti,                /* 0x0026 : SPI0_IRQ            */    //interrupt source:SPI0
    OnlyReti,                /* 0x0028 : EXT0_IRQ            */    //external interrupt0
    OnlyReti,                /* 0x002A : EXT1_IRQ            */    //external interrupt1
    OnlyReti,                /* 0x002C : EXT2_IRQ            */    //external interrupt2
    OnlyReti,                /* 0x002E : EXT3_IRQ            */    //external interrupt3
    OnlyReti,                /* 0x0030 : ADC_IRQ             */    //interrupt source:ADC
    OnlyReti,                /* 0x0032 : LVD0_IRQ            */
    OnlyReti,                /* 0x0034 : LVD1_IRQ            */
    OnlyReti,                /* 0x0036 : Reserved            */
    OnlyReti,                /* 0x0038 : Reserved            */
    OnlyReti,                /* 0x003A : FMC_IRQ             */    //interrupt source:flash memory controller
    OnlyReti,                /* 0x003C : DMACH0_IRQ          */
    OnlyReti,                /* 0x003E : DMACH1_IRQ          */
    OnlyReti,                /* 0x0040 : DMACH2_IRQ          */
    OnlyReti,                /* 0x0042 : DMACH3_IRQ          */
    OnlyReti,                /* 0x0044 : Reserved            */
    OnlyReti,                /* 0x0046 : Reserved            */
    OnlyReti,                /* 0x0048 : Reserved            */
    OnlyReti,                /* 0x004A : Reserved            */
    OnlyReti,                /* 0x004C : Reserved            */
    OnlyReti,                /* 0x004E : DIC_IRQ             */    //interrupt source:DIC
    DRV_LEUART_RX_IRQ,       /* 0x0050 : LEUART_RX_IRQ       */
    OnlyReti,                /* 0x0052 : LEUART_TX_IRQ       */
    OnlyReti,                /* 0x0054 : TRNG_IRQ            */
    OnlyReti,                /* 0x0056 : EXT4_IRQ            */    //external interrupt4
    OnlyReti,                /* 0x0058 : EXT5_IRQ            */    //external interrupt5
    OnlyReti,                /* 0x005A : EXT6_IRQ            */    //external interrupt6
    OnlyReti,                /* 0x005C : EXT7_IRQ            */    //external interrupt7
    OnlyReti,                /* 0x005E : EEMC_IRQ            */    //interrupt source:EEPROM controller
    OnlyReti,                /* 0x0060 : TCA0_IRQ            */    //interrupt source:TCA0 16-bit timer
    OnlyReti,                /* 0x0062 : TCA1_IRQ            */    //interrupt source:TCA1 16-bit timer
    OnlyReti,                /* 0x0064 : STM0_IRQ            */
    OnlyReti,                /* 0x0066 : STM1_IRQ            */
    OnlyReti,                /* 0x0068 : AES_IRQ             */
    OnlyReti,                /* 0x006A : SHA_IRQ             */
    OnlyReti,                /* 0x006C : RX1_IRQ             */    //interrupt source:UART1 RX1
    OnlyReti,                /* 0x006E : TX1_IRQ             */    //interrupt source:UART1 TX1
    OnlyReti,                /* 0x0070 : I2C1_IRQ            */    //interrupt source:I2C1
    OnlyReti,                /* 0x0072 : SPI1_IRQ            */    //interrupt source:SPI1
    OnlyReti,                /* 0x0074 : RX2_IRQ             */    //interrupt source:UART2 RX2
    OnlyReti,                /* 0x0076 : TX2_IRQ             */    //interrupt source:UART2 TX2
    OnlyReti                 /* 0x0078 : TCA2_IRQ            */    //interrupt source:TCA2 16-bit timer
};
#pragma section const
/* -------------------- end of SQ7705 interrupt vector ---------------------- */

