//******************************************************************************
// file         : DRV_SQ7705_I2C.h
// version      : V1.3 (2024/05/09)
// description  : I2C header file
// note         :
//******************************************************************************
#ifndef __DRV_SQ7705_I2C_H__
#define __DRV_SQ7705_I2C_H__

#include "TypeDefs.h"
#include "SQ7705_SFR.h"

typedef enum{                // bufcount: Number of data bits
    BC_8bit = 0,
    BC_1bit = 1,
    BC_2bit = 2,
    BC_3bit = 3,
    BC_4bit = 4,
    BC_5bit = 5,
    BC_6bit = 6,
    BC_7bit = 7
} I2C_BC;

typedef enum{                // Generation and counting of the clocks for an acknowledge signal
    ACK_NACKMODE = 0,        // non-acknowledgement mode
    ACK_ACKMODE = 1          // acknowledgement mode
} I2C_ACK;

typedef enum{                // Enables/disables the slave address match detection and the GENERAL CALL detection
    NOACK_ENABLE = 0,        // (slave mode only) Enable the slave address match detection and the GENERAL CALL detection
    NOACK_DISABLE = 1        // (slave mode only) Disable the slave address match detection and the GENERAL CALL detection
} I2C_NOACK;

typedef enum{                // HIGH and LOW periods of the serial clock in the master mode Time before the release of the SCL pin in the slave mode
                             // tHIGH(m/fsysclk)| tLOW(n/fsysclk)| ex: fsysclk=24MHz
                             //        m        |       n        |  fsysclk/(m+n)
                             // ------------------------------------------------------------------
    SCK_HI_9_LO_12    = 0,   //        9        |      12        |    1143KHz
    SCK_HI_11_LO_14   = 1,   //       11        |      14        |     960KHz
    SCK_HI_15_LO_18   = 2,   //       15        |      18        |     727KHz
    SCK_HI_23_LO_26   = 3,   //       23        |      26        |     490KHz
    SCK_HI_39_LO_42   = 4,   //       39        |      42        |     296KHz
    SCK_HI_71_LO_74   = 5,   //       71        |      74        |     166KHz
    SCK_HI_135_LO_138 = 6,   //      135        |     138        |      88KHz
    SCK_HI_263_LO_266 = 7    //      263        |     266        |      45KHz
} I2C_SCK;

typedef enum{                // Master/slave selection
    MST_SLAVE = 0,           // Slave
    MST_MASTER = 1           // Master
} I2C_MST;

typedef struct{              // I2C parameter struct
    I2C_BC i2cBC;            // I2C Number of data bits
    I2C_ACK i2cACK;          // I2C ACK - Generation and counting of the clocks for an acknowledge signal
    I2C_NOACK i2cNOACK;      // I2C NOACK - Enables/disables the slave address match detection and the GENERAL CALL detection
    I2C_SCK i2cSCK;          // I2C SCK - HIGH and LOW periods of the serial clock in the master mode Time before the release of the SCL pin in the slave mode
    I2C_MST i2cMST;          // I2C MST - Master/slave selection 
    uint8_t i2cSlaveAddr;    // I2C Slave address in the slave mode
} I2C_REG;

#define waitCountMax            250     // maximum waiting counting value
#define initBusLowErr           0xE0    // if one of SDA or SCL is not H level, its error code is 0xE0
#define nackTimeoutErr          0xE1    // if there is no response, its error code is 0xE1

#define I2C_OP_W     (0x00)             // I2C write operation.
#define I2C_OP_R     (0x01)             // I2C read operation.

//------------------------------------------------------------------------------
// function return value list
// 0 : successful, others : error
//------------------------------------------------------------------------------
#define DRV_RET_SUCCESS                (0)      // Execution success.
#define DRV_RET_FAILURE                (-1)     // Error.
#define DRV_RET_INVALID_PARAM          (-2)     // Invalid input parameters.
#define DRV_RET_TIMEOUT                (-3)     // Execution timeout.
#define DRV_RET_RESOURCE_CONFLICT      (-4)     // Resource conflict.
#define DRV_RET_SETTING_CONFLICT       (-5)     // Setting conflict.
#define DRV_RET_PERMISSION_DENIED      (-6)     // Permission denied.
#define DRV_RET_BUS_ERROR              (-7)     // BUS error.
#define DRV_RET_INVALID_PACKET         (-8)     // Packet error.
#define DRV_RET_CRC_ERROR              (-9)     // CRC error.

typedef void (* I2C_CALLBACK) (void);           // type define of UART callback function pointer

void  DRV_I2C0_Init(I2C_REG i2cReg);                      // I2C0 initialization.
void  DRV_I2C0_Int(uint8_t config, I2C_CALLBACK i2cCB);   // enable / disable I2C0 interrupt
uint8_t DRV_I2C0_Master_Write(uint8_t devAddr, uint8_t *data, uint8_t len); // I2C0 transmits data.
uint8_t DRV_I2C0_Master_Read(uint8_t devAddr, uint8_t *data, uint8_t len);  // I2C0 receives data.

uint8_t DRV_I2C0_Start(uint8_t addr, uint8_t opRW);       // I2C0 generate the start condition
uint8_t DRV_I2C0_Restart(uint8_t addr);                   // I2C0 generate the restart condition.
void  DRV_I2C0_Stop(void);                                // I2C0 generate the stop condition.
void  DRV_I2C0_Reset(void);                               // software reset I2C0.
void  DRV_I2C0_Confirm_Tx_Completed(void);                // wait for tranfer completion
uint8_t DRV_I2C0_Wait_Slave_ACK(void);                    // wait for Slave return ACK.
void  DRV_I2C0_SetDevAddr_Direction(uint8_t addr, uint8_t opEW);  // set slave device address and direction.
void  DRV_I2C0_SetSlaveModeAddr(uint8_t addr);            // set slave mode address.
void  DRV_I2C0_Error_Handling(void);                      // I2C0 error handling.

void  DRV_I2C1_Init(I2C_REG i2cReg);                      // I2C1 initialization.
void  DRV_I2C1_Int(uint8_t config, I2C_CALLBACK i2cCB);   // enable / disable I2C1 interrupt
uint8_t DRV_I2C1_Master_Write(uint8_t devAddr, uint8_t *data, uint8_t len); // I2C1 transmits data.
uint8_t DRV_I2C1_Master_Read(uint8_t devAddr, uint8_t *data, uint8_t len);  // I2C1 receives data.

uint8_t DRV_I2C1_Start(uint8_t addr, uint8_t opRW);       // I2C1 generate the start condition
uint8_t DRV_I2C1_Restart(uint8_t addr);                   // I2C1 generate the restart condition.
void  DRV_I2C1_Stop(void);                                // I2C1 generate the stop condition.
void  DRV_I2C1_Reset(void);                               // software reset I2C1.
void  DRV_I2C1_Confirm_Tx_Completed(void);                // wait for tranfer completion
uint8_t DRV_I2C1_Wait_Slave_ACK(void);                    // wait for Slave return ACK.
void  DRV_I2C1_SetDevAddr_Direction(uint8_t addr, uint8_t opEW);  // set slave device address and direction.
void  DRV_I2C1_SetSlaveModeAddr(uint8_t addr);            // set slave mode address.
void  DRV_I2C1_Error_Handling(void);                      // I2C1 error handling.

void  __interrupt I2C0_IRQ(void);                         // I2C0 interrupt service routine.
void  __interrupt I2C1_IRQ(void);                         // I2C1 interrupt service routine.

#endif  // __DRV_SQ7705_I2C_H__.
