//******************************************************************************
// file         : DRV_SQ7705_I2C.c
// version      : V1.3 (2024/05/09)
// description  : I2C related functions
// note         : I2C related functions are gathered in this subroutine
//******************************************************************************
#include <stdlib.h>
#include "DRV_SQ7705_I2C.h"

static I2C_CALLBACK I2C0_CallBackFunc;
static I2C_CALLBACK I2C1_CallBackFunc;

uint8_t i2c0ErrCode = 0x00;
uint8_t i2c1ErrCode = 0x00;

//******************************************************************************
// name         : DRV_I2C0_Init()
// description  : I2C0 initialization
// input param  : - i2cReg : I2C initialization data structure
// retval       : none
// note         : set I2C0 pins and transfer mode 
//******************************************************************************
void DRV_I2C0_Init(I2C_REG i2cReg)
{
    I2C0_CallBackFunc = NULL;

    //====== I2C0 pin configuration initialization ======
    while (P7DI_P4 != 1 || P7DI_P5 != 1) {  // check SDA0 and SCL0 are both H level
        CLR_WDT;                            // clear watchdog timer.
        i2c0ErrCode = initBusLowErr;
        DRV_I2C0_Error_Handling();              // error handling.
    }
    P7CFGCR = 0x80 | 0x40 | 0x03;           // set P7.4 to SDA0
    P7CFGCR = 0x80 | 0x50 | 0x03;           // set P7.5 to SCL0

    //====== enable function ======
    PCKEN2_I2C0 = ENABLE;                   // this must be enabled first, the relevant 
                                            // staging settings later to be useful,I2C0 Enable
    SBI0CR2_SBIM = 1;                       // set to serial bus interface mode.

    // SBI0CR1 
    SBI0CR1_ACK = i2cReg.i2cACK;
    SBI0CR1_BC = i2cReg.i2cBC;
    SBI0CR1_SCK = i2cReg.i2cSCK;
    SBI0CR1_NOACK = i2cReg.i2cNOACK;
    
    // SBI0CR2 
    SBI0CR2 = (i2cReg.i2cMST << 7) | 0x18;  // Master/Slave mode. 

    if(i2cReg.i2cMST == MST_SLAVE) {
        DRV_I2C0_SetSlaveModeAddr(i2cReg.i2cSlaveAddr);  //set slave mode address
    }
}

//******************************************************************************
// name         : DRV_I2C0_Int()
// description  : enable / disable I2C0 interrupt
// input param  : config : ENABLE  enable I2C0 interrupt
//                         DISABLE  disable I2C0 interrupt
//                i2cCB  : interrupt CALLBACK function
// retval       : none
// note         : 
//******************************************************************************
void DRV_I2C0_Int(uint8_t config, I2C_CALLBACK i2cCB)
{
    __asm("DI");                        // disable all maskable interrupts.

    IFR_I2C0 = 0;                       // clear interrupt flag.
    IER_I2C0 = config;                  // if config = ENABLE  ,enable I2C0 interrupt
                                        // if config = DISABLE ,disable I2C0 interrupt
    if(config == ENABLE) {
        I2C0_CallBackFunc = i2cCB;
    } else {
        I2C0_CallBackFunc = NULL;
    }
    __asm("EI");                        // enable all maskable interrupts.
}

//******************************************************************************
// name         : DRV_I2C0_Master_Write()
// description  : I2C0 transmits data
// input param  : devAddr : salve device address
//                data    : data buffer to be transmitted
//                len     : write data length
// retval       : errCode
// note         :            
//******************************************************************************
uint8_t DRV_I2C0_Master_Write(uint8_t devAddr, uint8_t *data, uint8_t len)
{
    uint8_t i;

    if (DRV_I2C0_Start(devAddr, I2C_OP_W) < 0) {
        DRV_I2C0_Stop();
        return DRV_RET_BUS_ERROR;
    }

    for (i = 0; i < len; i++) {
        SBI0DBR = *data;
        data++;
        DRV_I2C0_Confirm_Tx_Completed();      // wait for tranfer completion
        if (DRV_I2C0_Wait_Slave_ACK() < 0) {  // if Slave return ACK
            return DRV_RET_BUS_ERROR;
        }
    }

    DRV_I2C0_Stop();

    return DRV_RET_SUCCESS;
}

//******************************************************************************
// name         : DRV_I2C0_Master_Read()
// description  : I2C0 receives data
// input param  : devAddr : start address to read
//                len     : read data length
// output param : data    : read data buffer
// retval       : errCode
// note         :            
//******************************************************************************
uint8_t DRV_I2C0_Master_Read(uint8_t devAddr, uint8_t *data, uint8_t len)
{
    uint8_t i;

    if (DRV_I2C0_Wait_Slave_ACK() < 0) {    // if Slave return ACK
        return DRV_RET_BUS_ERROR;
    }

    if (DRV_I2C0_Start(devAddr, I2C_OP_R) < 0) {
        DRV_I2C0_Stop();
        return DRV_RET_BUS_ERROR;
    }

    // Read (len-1) byte
    for (i = 0; i < len; i++) {
        if(i == len - 1) SBI0CR1_ACK = 0;   // ACK=0
        SBI0DBR = 0x00;                     // write dummy data to set PIN = 1
        DRV_I2C0_Confirm_Tx_Completed();
        *data = SBI0DBR;
        data++;
    }

    //  Termination of data transfer by master mode
    //  Sent 1 bit negative acknowledge............
    SBI0CR1_BC = 1;                     // BC, bit count = 1
    SBI0DBR    = 0x00;                  // write dummy data to set PIN = 1
    DRV_I2C0_Confirm_Tx_Completed();

    DRV_I2C0_Stop();

    return DRV_RET_SUCCESS;
}

//******************************************************************************
// name         : DRV_I2C0_Start()
// description  : I2C0 generate the start condition
// input param  : addr : slave device address
//                       e.g. slave address of AT24C08 is 0xA0
//                opRW : read or write mode
// retval       : i2c0ErrCode
// note         :
//******************************************************************************
uint8_t DRV_I2C0_Start(uint8_t addr, uint8_t opRW)
{
    while (SBI0SR_BB == 1);             // bus status 0: free ; 1 :busy.

    SBI0CR1_ACK = 1;                    // *** SBI0CR1<ACK> must be set before transmit data ***.

    DRV_I2C0_SetDevAddr_Direction(addr, opRW); //fill I2C0 databuf -- set slave device address and direction.

    SBI0CR2 = 0xF8;                     // Write "1" to SBI0CR2<MST>, <TRX>, <BB>, <PIN>  and <SBIM>to "1".
                                        // 1111 1000
                                        // set I2C to master mode, transmitter, generating start condition, 
                                        // cancel interrupt service request and serial bus interface mode.

    DRV_I2C0_Confirm_Tx_Completed();    // wait for tranfer completion
    DRV_I2C0_Wait_Slave_ACK();          // wait for Slave return ACK.

    //--------------------------------------------------------------------------
    if (i2c0ErrCode != 0) {
        return i2c0ErrCode;
    }

    return i2c0ErrCode;
}

//******************************************************************************
// name         : DRV_I2C0_ReStart()
// description  : I2C0 generate the restart condition
// input param  : addr : slave device address
// retval       : none
// note         : none
//******************************************************************************
uint8_t DRV_I2C0_ReStart(uint8_t addr)
{
    SBI0CR2 = 0x18;                     // 0001 1000.
                                        // set I2C0 to master mode, transmitter, generating start condition, 
                                        // cancel interrupt service request and serial bus interface mode.

    while (SBI0SR_BB);                  // bus status 0: free ; 1 :busy.
    
    while (SBI0SR_LRB == 0);

    DRV_I2C0_SetDevAddr_Direction(addr, I2C_OP_R);  // set slave device address and direction.
    SBI0CR2 = 0xF8;                     // Write "1" to SBI0CR2<MST>, <TRX>, <BB>, <PIN>  and <SBIM>to "1".
                                        // 1111 1000.

    DRV_I2C0_Confirm_Tx_Completed();    // wait for tranfer completion
    DRV_I2C0_Wait_Slave_ACK();          // wait for Slave return ACK.

    return i2c0ErrCode;
}

//******************************************************************************
// name         : DRV_I2C0_Stop()
// description  : I2C0 generate the stop condition
// input param  : none
// retval       : none
// note         : 
//******************************************************************************
void DRV_I2C0_Stop(void)
{
    SBI0CR2 = 0xD8;                     // Set SBI0CR2<MST>, <TRX> and <PIN> to "1" and SBI0CR2<BB> to "0".
                                        // 1101 1000.
                                        // set I2C0 to master mode, transmitter, generating stop condition, 
                                        // cancel interrupt service request and port mode.

    while (SBI0SR_BB == 1);             // bus status 0: free ; 1 :busy.
}

//******************************************************************************
// name         : DRV_I2C0_Reset()
// description  : software reset I2C0
// input param  : none
// retval       : none
// note         : reset SBI0CR1,I2C0AR,SBI0CR2, not reset SBI0CR2<SBIM>
//******************************************************************************
void DRV_I2C0_Reset()
{
    SBI0CR2 = 0x1A;                     // first write SWRST[1:0] = 0x02.
    SBI0CR2 = 0x19;                     // then write [1:0] = 0x01.
}

//******************************************************************************
// name         : DRV_I2C0_Confirm_Tx_Completed()
// description  : wait for tranfer completion
// input param  : none
// retval       : none
// note         :
//******************************************************************************
void DRV_I2C0_Confirm_Tx_Completed(void)
{
    while (SBI0SR_PIN);                 // after tranfer completion, PIN will chaange to 0
}

//******************************************************************************
// name         : DRV_I2C0_Wait_Slave_ACK()
// description  : wait for Slave return ACK
// input param  : none
// retval       : i2c0ErrCode
// note         :
//******************************************************************************
uint8_t DRV_I2C0_Wait_Slave_ACK()
{
    uint8_t tWait;

    i2c0ErrCode = 0x00;
    tWait = 0;

    while (SBI0SR_LRB == 1) {           // check <LRB> wait for last received bit monitor is "0"    
        tWait++;                        // no response and increase counting.

        if (tWait > waitCountMax) {     // if there is no response for too long, the loop will be broken.
            DRV_I2C0_Stop();            // I2C0 stop -> release SCK.
            i2c0ErrCode = nackTimeoutErr;
            break;
        }
    }

    if (i2c0ErrCode != 0) {
        return i2c0ErrCode;
    }

    // ACK
    i2c0ErrCode = 0x00;                 // if no error, return 0x00

    return i2c0ErrCode;
}

//******************************************************************************
// name         : DRV_I2C0_SetDevAddr_Direction()
// description  : set slave device address and direction
// input param  : addr : slave device address
//                opRW : read or write mode 
//                       write mode (write data to slave)
//                       read mode (read data from slave)
// retval       : 
// note         : 
//******************************************************************************
void DRV_I2C0_SetDevAddr_Direction(uint8_t addr, uint8_t opRW)
{
    SBI0DBR = (addr | opRW);
}

//******************************************************************************
// name         : DRV_I2C0_SetSlaveModeAddr()
// description  : set slave mode address
// input param  : addr: slave address
// retval       : none
// note         : For slave mode only.
//******************************************************************************
void DRV_I2C0_SetSlaveModeAddr(uint8_t addr)
{
    I2C0AR = addr & 0xFE;               // I2C0AR[7:1]: Slave address setting
}

//******************************************************************************
// name         : DRV_I2C0_Error_Handling(void)
// description  : error handling
// input param  : none
// retval       : none
// note         : if one of SDA0 or SCL0 is not H level           
//******************************************************************************
void DRV_I2C0_Error_Handling(void)
{
    if (i2c0ErrCode == initBusLowErr) {
        //Add your error handling error handling......
        while (1) {
            CLR_WDT;                    // clear watchdog timer
            __asm("NOP");
        }
    }
}

//******************************************************************************
// name         : DRV_I2C1_Init()
// description  : I2C1 initialization
// input param  : - i2cReg : I2C initialization data structure
// retval       : none
// note         : set I2C1 pins and transfer mode 
//******************************************************************************
void DRV_I2C1_Init(I2C_REG i2cReg)
{
    I2C1_CallBackFunc = NULL;
    
    //====== I2C1 pin configuration initialization ======
    while (P2DI_P4 != 1 || P2DI_P5 != 1) {  // check SDA1 and SCL1 are both H level
        CLR_WDT;                            // clear watchdog timer.
        i2c1ErrCode = initBusLowErr;
        DRV_I2C1_Error_Handling();              // error handling.
    }
    P2CFGCR = 0x80 | 0x40 | 0x03;           // set P2.4 to SDA1
    P2CFGCR = 0x80 | 0x50 | 0x03;           // set P2.5 to SCL1

    //====== enable function ======
    PCKEN2_I2C1 = ENABLE;               // this must be enabled first, the relevant
                                        // staging settings later to be useful,I2C1 Enable
    SBI1CR2_SBIM = 1;                   // set to serial bus interface mode.

    // SBI1CR1 
    SBI1CR1_ACK = i2cReg.i2cACK;
    SBI1CR1_BC = i2cReg.i2cBC;
    SBI1CR1_SCK = i2cReg.i2cSCK;
    SBI1CR1_NOACK = i2cReg.i2cNOACK;
    
    // SBI1CR2 
    SBI1CR2 = (i2cReg.i2cMST << 7) | 0x18;  // Master/Slave mode. 

    if(i2cReg.i2cMST == MST_SLAVE) {
        DRV_I2C1_SetSlaveModeAddr(i2cReg.i2cSlaveAddr);  //set slave mode address
    }
}

//******************************************************************************
// name         : DRV_I2C1_Int()
// description  : enable / disable I2C1 interrupt
// input param  : config : ENABLE  enable I2C1 interrupt
//                         DISABLE  disable I2C1 interrupt
//                i2cCB  : interrupt CALLBACK function
// retval       : none
// note         : 
//******************************************************************************
void DRV_I2C1_Int(uint8_t config, I2C_CALLBACK i2cCB)
{
    __asm("DI");                        // disable all maskable interrupts.

    IFR_I2C1 = 0;                       // clear interrupt flag.
    IER_I2C1 = config;                  // if config = ENABLE  ,enable I2C1 interrupt
                                        // if config = DISABLE ,disable I2C1 interrupt
    if(config == ENABLE) {
        I2C1_CallBackFunc = i2cCB;
    } else {
        I2C1_CallBackFunc = NULL;
    }
    __asm("EI");                        // enable all maskable interrupts.
}

//******************************************************************************
// name         : DRV_I2C1_Master_Write()
// description  : I2C1 transmits data
// input param  : devAddr : salve device address
//                data    : data buffer to be transmitted
//                len     : write data length
// retval       : errCode
// note         :            
//******************************************************************************
uint8_t DRV_I2C1_Master_Write(uint8_t devAddr, uint8_t *data, uint8_t len)
{
    uint8_t i;

    if (DRV_I2C1_Start(devAddr, I2C_OP_W) < 0) {
        DRV_I2C1_Stop();
        return DRV_RET_BUS_ERROR;
    }

    for (i = 0; i < len; i++) {
        SBI1DBR = *data;
        data++;
        DRV_I2C1_Confirm_Tx_Completed();      // wait for tranfer completion
        if (DRV_I2C1_Wait_Slave_ACK() < 0) {  // if Slave return ACK
            return DRV_RET_BUS_ERROR;
        }
    }

    DRV_I2C1_Stop();

    return DRV_RET_SUCCESS;
}

//******************************************************************************
// name         : DRV_I2C1_Master_Read()
// description  : I2C1 receives data
// input param  : devAddr : start address to read
//                len     : read data length
// output param : data    : read data buffer
// retval       : errCode
// note         :            
//******************************************************************************
uint8_t DRV_I2C1_Master_Read(uint8_t devAddr, uint8_t *data, uint8_t len)
{
    uint8_t i;

    if (DRV_I2C1_Wait_Slave_ACK() < 0) {    // if Slave return ACK
        return DRV_RET_BUS_ERROR;
    }

    if (DRV_I2C1_Start(devAddr, I2C_OP_R) < 0) {
        DRV_I2C1_Stop();
        return DRV_RET_BUS_ERROR;
    }

    // Read (len-1) byte
    for (i = 0; i < len; i++) {
        if(i == len - 1) SBI1CR1_ACK = 0;   // ACK=0
        SBI1DBR   = 0x00;                   // write dummy data to set PIN = 1
        DRV_I2C1_Confirm_Tx_Completed();
        *data = SBI1DBR;
        data++;
    }

    //  Termination of data transfer by master mode
    //  Sent 1 bit negative acknowledge............
    SBI1CR1_BC = 1;                     // BC, bit count = 1
    SBI1DBR    = 0x00;                  // write dummy data to set PIN = 1
    DRV_I2C1_Confirm_Tx_Completed();

    DRV_I2C1_Stop();

    return DRV_RET_SUCCESS;
}

//******************************************************************************
// name         : DRV_I2C1_Start()
// description  : I2C1 generate the start condition
// input param  : addr : slave device address
//                       e.g. slave address of AT24C08 is 0xA0
//                opRW : read or write mode
// retval       :  i2c1ErrCode
// note         :
//******************************************************************************
uint8_t DRV_I2C1_Start(uint8_t addr, uint8_t opRW)
{
    while (SBI1SR_BB == 1);                 // bus status 0: free ; 1 :busy.

    SBI1CR1_ACK = 1;                        // *** SBI1CR1<ACK> must be set before transmit data ***.

    DRV_I2C1_SetDevAddr_Direction(addr, opRW); // fill I2C1 databuf -- set slave device address and direction.

    SBI1CR2 = 0xF8;                         // Write "1" to SBI1CR2<MST>, <TRX>, <BB>, <PIN>  and <SBIM>to "1".
                                            // 1111 1000
                                            // set I2C to master mode, transmitter, generating start condition,
                                            // cancel interrupt service request and serial bus interface mode.

    DRV_I2C1_Confirm_Tx_Completed();        // wait for tranfer completion
    DRV_I2C1_Wait_Slave_ACK();              // wait for Slave return ACK.

    //--------------------------------------------------------------------------
    if (i2c1ErrCode != 0) {
        return i2c1ErrCode;
    }

    return i2c1ErrCode;
}

//******************************************************************************
// name         : DRV_I2C1_ReStart()
// description  : I2C1 generate the restart condition
// input param  : addr : slave device address
// retval       : none
// note         : none
//******************************************************************************
uint8_t DRV_I2C1_ReStart(uint8_t addr)
{
    SBI1CR2 = 0x18;                     // 0001 1000.
                                        // set I2C1 to master mode, transmitter, generating start condition,
                                        // cancel interrupt service request and serial bus interface mode.

    while (SBI1SR_BB);                  // bus status 0: free ; 1 :busy.
    
    while (SBI1SR_LRB == 0);

    DRV_I2C1_SetDevAddr_Direction(addr, I2C_OP_R);  // set slave device address and direction.
    SBI1CR2 = 0xF8;                     // Write "1" to SBI1CR2<MST>, <TRX>, <BB>, <PIN>  and <SBIM>to "1".
                                        // 1111 1000.

    DRV_I2C1_Confirm_Tx_Completed();    // wait for tranfer completion
    DRV_I2C1_Wait_Slave_ACK();          // wait for Slave return ACK.

    return i2c1ErrCode;
}

//******************************************************************************
// name         : DRV_I2C1_Stop()
// description  : I2C1 generate the stop condition
// input param  : none
// retval       : none
// note         : 
//******************************************************************************
void DRV_I2C1_Stop(void)
{
    SBI1CR2 = 0xD8;                     // Set SBI1CR2<MST>, <TRX> and <PIN> to "1" and SBI1CR2<BB> to "0".
                                        // 1101 1000.
                                        // set I2C1 to master mode, transmitter, generating stop condition, 
                                        // cancel interrupt service request and port mode.

    while (SBI1SR_BB == 1);             // bus status 0: free ; 1 :busy.
}

//******************************************************************************
// name         : DRV_I2C1_Reset()
// description  : software reset I2C1
// input param  : none
// retval       : none
// note         : reset SBI1CR1,I2C1AR,SBI1CR2, not reset SBI1CR2<SBIM>
//******************************************************************************
void DRV_I2C1_Reset()
{
    SBI1CR2 = 0x1A;                     // first write SWRST[1:0] = 0x02.
    SBI1CR2 = 0x19;                     // then write [1:0] = 0x01.
}

//******************************************************************************
// name         : DRV_I2C1_Confirm_Tx_Completed()
// description  : wait for tranfer completion
// input param  : none
// retval       : none
// note         :
//******************************************************************************
void DRV_I2C1_Confirm_Tx_Completed(void)
{
    while (SBI1SR_PIN);                 // after tranfer completion, PIN will chaange to 0.
}

//******************************************************************************
// name         : DRV_I2C1_Wait_Slave_ACK()
// description  : wait for Slave return ACK
// input param  : none
// retval       : i2c1ErrCode
// note         :
//******************************************************************************
uint8_t DRV_I2C1_Wait_Slave_ACK()
{
    uint8_t tWait;

    i2c1ErrCode = 0x00;
    tWait = 0;

    while (SBI1SR_LRB == 1) {           // check <LRB> wait for last received bit monitor is "0"    
        tWait++;                        // no response and increase counting.

        if (tWait > waitCountMax) {     // if there is no response for too long, the loop will be broken.
            DRV_I2C1_Stop();            // I2C1 stop -> release SCK.
            i2c1ErrCode = nackTimeoutErr;
            break;
        }
    }

    if (i2c1ErrCode != 0) {
        return i2c1ErrCode;
    }

    // ACK
    i2c1ErrCode = 0x00;                 // if no error, return 0x00

    return i2c1ErrCode;
}

//******************************************************************************
// name         : DRV_I2C1_SetDevAddr_Direction()
// description  : set slave device address and direction
// input param  : addr : slave device address
//                opRW : read or write mode 
//                       write mode (write data to slave)
//                       read mode (read data from slave)
// retval       : 
// note         : 
//******************************************************************************
void DRV_I2C1_SetDevAddr_Direction(uint8_t addr, uint8_t opRW)
{
    SBI1DBR = (addr | opRW);
}

//******************************************************************************
// name         : DRV_I2C1_SetSlaveModeAddr()
// description  : set slave mode address
// input param  : addr : slave address
// retval       : none
// note         : For slave mode only.
//******************************************************************************
void DRV_I2C1_SetSlaveModeAddr(uint8_t addr)
{
    I2C1AR = addr & 0xFE;               // I2C1AR[7:1]: Slave address setting
}

//******************************************************************************
// name         : DRV_I2C1_Error_Handling(void)
// description  : error handling
// input param  : none
// retval       : none
// note         : if one of SDA1 or SCL1 is not H level           
//******************************************************************************
void DRV_I2C1_Error_Handling(void)
{
    if (i2c1ErrCode == initBusLowErr) {
        //Add your error handling error handling......
        while (1) {
            CLR_WDT;                    // clear watchdog timer
            __asm("NOP");
        }
    }
}

//******************************************************************************
// name         : I2C0_IRQ()
// description  : I2C0 interrupt service routine
// input param  : none
// retval       : none
// note         :
//******************************************************************************
void __interrupt I2C0_IRQ(void)
{
    if (I2C0_CallBackFunc != NULL) {
        I2C0_CallBackFunc();
    }
}

//******************************************************************************
// name         : I2C1_IRQ()
// description  : I2C1 interrupt service routine
// input param  : none
// retval       : none
// note         :
//******************************************************************************
void __interrupt I2C1_IRQ(void)
{
    if (I2C1_CallBackFunc != NULL) {
        I2C1_CallBackFunc();
    }
}

