//******************************************************************************
// file         : main.c
// version      : V1.3  2024/05/09
// description  : main program
// note         : this example program is a demostration for I2C1 sending data to I2C0 within a single SQ7705 chip
//                when press button 1, I2C1 sends command 0x09 and data 0x1234, 
//                if I2C0 is received correctly then light on LED1
//                when press button 2, I2C1 sends command 0x0A and data 0x55AA, 
//                if I2C0 is received correctly then light on LED2
//                when press button 3, I2C1 sends command 0x00 and data 0xFFFF, 
//                if I2C0 is received correctly then light on LED3
//                if not yet communicating or receiving error then not light on LEDs
//                
//                1. I2C Master and Slave setting : 
//                    set I2C0 to slave mode and slave address is 0x68
//                    set P7.4 to SDA0
//                    set P7.5 to SCL0
//                    set I2C1 to master mode
//                    set P2.4 to SDA1
//                    set P2.5 to SCL1
// 
//                2. SDA0(P7.4) connects to (P2.4)SDA1
//                   SCL0(P7.5) connects to (P2.5)SCL1
//                    *the above SDA and SCL must be connected with external pull-up resistors(example 2.2K ohm)
// 
//                3. [Button Configuration] 
//                    button 1 :  BTN1_in(P5.0)
//                    button 2 :  BTN2_in(P5.1)
//                    button 3 :  BTN3_in(P5.2)
//                   [LED Configuration] LED 1  (P2.0)
//                   [LED Configuration] LED 2  (P2.1)
//                   [LED Configuration] LED 3  (P2.2)
//******************************************************************************
#ifndef IO_MEM
#define IO_MEM
#endif
#include "ADM_main.h"
#include "DRV_SQ7705_SYSCLK.h"
#include "DRV_SQ7705_I2C.h"

/* --------------------------- global variables ----------------------------- */
uint8_t cmdBuf[4];       // command buffer
uint8_t rdBuf[4];        // read data buffer
I2C_REG i2c0Reg;         // I2C0 Reg structure
I2C_REG i2c1Reg;         // I2C1 Reg structure

uint16_t rx_count;
uint16_t tx_count;
uint8_t rtnLen;
uint8_t rtnBuf[4];
uint8_t rxBuf[4];
uint8_t i2c_SlaveStatus;
uint8_t receiveCMD;
/* -------------------- function prototype declaration ---------------------- */
void I2C0_Slave_CALLBACK(void);

/* ---------------------------- main program -------------------------------- */
void ADM_Main()
{
    DRV_Clock_Init(Clk_Source_SYS, 1);                  // Initial system clock (PLL 48MHz/2 = 24MHz)

    /* ---------------------- GPIO initialization --------------------------- */
    DRV_GPIO_Init(GPIO_P20, GPIO_OUTPUT, GPIO_PULL_UP); // set P2.0 as output pin
                                                        // enable P2.0 built-in pull-up resistor
    DRV_GPIO_Init(GPIO_P21, GPIO_OUTPUT, GPIO_PULL_UP); // set P2.1 as output pin
                                                        // enable P2.1 built-in pull-up resistor
    DRV_GPIO_Init(GPIO_P22, GPIO_OUTPUT, GPIO_PULL_UP); // set P2.2 as output pin
                                                        // enable P2.2 built-in pull-up resistor
    DRV_GPIO_Init(GPIO_P23, GPIO_OUTPUT, GPIO_PULL_UP); // set P2.3 as output pin
                                                        // enable P2.3 built-in pull-up resistor
    LED1_to_3_turn_off;
    
    DRV_GPIO_Init(GPIO_P50, GPIO_INPUT, GPIO_PULL_UP);  // set P5.0 as input pin
                                                        // enable P5.0 built-in pull-up resistor
    DRV_GPIO_Init(GPIO_P51, GPIO_INPUT, GPIO_PULL_UP);  // set P5.1 as input pin
                                                        // enable P5.1 built-in pull-up resistor
    DRV_GPIO_Init(GPIO_P52, GPIO_INPUT, GPIO_PULL_UP);  // set P5.2 as input pin

    /* ----------------------- I2C0 initialization -------------------------- */
    // P7.4 I2C0-SDA
    DRV_GPIO_Init(GPIO_P74, GPIO_INPUT, GPIO_PULL_UP);  // set P7.4 as input pin
                                                        // enable P7.4 built-in pull-up resistor
    // P7.5 I2C0-SCL
    DRV_GPIO_Init(GPIO_P75, GPIO_INPUT, GPIO_PULL_UP);  // set P7.5 as input pin
                                                        // enable P7.5 built-in pull-up resistor
    i2c0Reg.i2cACK = ACK_ACKMODE;                       // set I2C0 to ACK mode
    i2c0Reg.i2cBC = BC_8bit;                            // set I2C0 to 8bit mode
    i2c0Reg.i2cSCK = SCK_HI_135_LO_138;                 // set I2C0 SCK 88KHz(in fsysclk=24MHz)
    i2c0Reg.i2cMST = MST_SLAVE;                         // set I2C0 to slave mode
    i2c0Reg.i2cSlaveAddr = 0x68;                        // set I2C0 slave address is 0x68
    DRV_I2C0_Init(i2c0Reg);                             // I2C0 initial
    DRV_I2C0_Int(ENABLE, I2C0_Slave_CALLBACK);          // Enable I2C0 interrupt and set callback function
    
    /* ---------------------- I2C1 initialization --------------------------- */
    //P2.4 I2C1-SDA 
    DRV_GPIO_Init(GPIO_P24, GPIO_INPUT, GPIO_PULL_UP);  // set P2.4 as input pin
                                                        // enable P2.4 built-in pull-up resistor
    //P2.5 I2C1-SCL
    DRV_GPIO_Init(GPIO_P25, GPIO_INPUT, GPIO_PULL_UP);  // set P2.5 as input pin
                                                        // enable P2.5 built-in pull-up resistor
    i2c1Reg.i2cACK = ACK_ACKMODE;                       // set I2C1 to ACK mode
    i2c1Reg.i2cBC = BC_8bit;                            // set I2C1 to 8bit mode
    i2c1Reg.i2cSCK = SCK_HI_135_LO_138;                 // set I2C1 SCK 88KHz(in fsysclk=24MHz)
    i2c1Reg.i2cMST = MST_MASTER;                        // set I2C1 to master mode
    DRV_I2C1_Init(i2c1Reg);                             // I2C1 initial
    
    /* ----------------- Use button to test I2C write/read ------------------ */
    while(1){
        if(BTN1_in == 0){                              // press button 1
            __ASM("NOP");            
            cmdBuf[0] = 0x09;                          // change command to 0x09
                          
            DRV_I2C1_Master_Write(0x68,(uint8_t*)(&cmdBuf),1);  // I2C1 send address 0x68 and 1-byte data 0x09
            DRV_I2C1_Master_Read(0x68,(uint8_t*)(&rdBuf),2);    // I2C1 read 2-bytes data
            LED1_to_3_turn_off;
            if(rdBuf[0] == 0x12 && rdBuf[1] == 0x34){  // compare read data success
                LED1_on;                               // LED1 On            
            } else{                                    // compare fail 
                LED1_off;                              // LED1 off
            }
             
            while(BTN1_in == 0);

        } else if(BTN2_in == 0){ 
            __ASM("NOP");   
            cmdBuf[0] = 0x0A;                          // change command to 0x0A
                          
            DRV_I2C1_Master_Write(0x68,(uint8_t*)(&cmdBuf),1);  // I2C1 send address 0x68 and 1-byte data 0x0A
            DRV_I2C1_Master_Read(0x68,(uint8_t*)(&rdBuf),2);    // I2C1 read 2-bytes data
            LED1_to_3_turn_off;
            if(rdBuf[0] == 0x55 && rdBuf[1] == 0xAA){  // compare read data success      
                LED2_on;                               // LED2 On             
            } else{                                    // compare fail 
                LED2_off;                              // LED2 Off    
            }
   
            while(BTN2_in == 0);

        } else if(BTN3_in == 0){                       // press button 3
            __ASM("NOP");   
            cmdBuf[0] = 0x00;                          // change command to 0x0A
                          
            DRV_I2C1_Master_Write(0x68,(uint8_t*)(&cmdBuf),1);  // I2C1 send address 0x68 and 1-byte data 0x00
            DRV_I2C1_Master_Read(0x68,(uint8_t*)(&rdBuf),2);
            LED1_to_3_turn_off;
            if(rdBuf[0] == 0xFF && rdBuf[1] == 0xFF){ 
                LED3_on;                               // LED3 On
            } else{                     
                LED3_off;                              // LED3 off
            }
     
            while(BTN3_in == 0);
        }
        LED1_to_3_turn_off;                            // light off all LEDs
    }
}

//******************************************************************************
// name         : I2C0_Slave_CALLBACK()
// description  : I2C0 Slave CALLBACK funnction
// input param  : none
// retval       : none
// note         :
//******************************************************************************
void I2C0_Slave_CALLBACK(void)
{
    uint8_t tmp;
    if(SBI0SR_TRX == 1) {
        if(SBI0SR_AAS == 1) {
            i2c_SlaveStatus = 0;
            if(receiveCMD == 0x09) {
                rtnBuf[0] = 0x12;
                rtnBuf[1] = 0x34;
                rtnLen = 2;
            } else if(receiveCMD == 0x0A) {
                rtnBuf[0] = 0x55;
                rtnBuf[1] = 0xAA;
                rtnLen = 2;             
            } else {
                rtnLen = 0;
            }
            tx_count = 0;
            if(tx_count >= rtnLen) {
                SBI0DBR = 0xFF;
            } else {
                SBI0DBR = rtnBuf[tx_count];
            }
        } else {
            if(SBI0SR_LRB == 1) {
                SBI0DBR = 0xFF;
            } else {
                if(tx_count >= rtnLen) {
                    SBI0DBR = 0xFF;
                    //SBI0CR2 = 0x58;
                    //SBI0CR2 = 0x18;
                } else {
                    SBI0DBR = rtnBuf[tx_count];
                }
            }
        }
        tx_count++;
    } else {
        if(SBI0SR_AAS == 1) {
            i2c_SlaveStatus = 1;
            tmp = SBI0DBR;
            SBI0DBR = 0x00; // transmit dummy data
            __asm("NOP");
        } else {
            if(i2c_SlaveStatus == 1) {
                receiveCMD = SBI0DBR;
                i2c_SlaveStatus = 2;
                SBI0DBR = 0x00;
                rx_count = 0;
                __asm("NOP");
            } else {
                rxBuf[rx_count++] = SBI0DBR;
                SBI0DBR = 0x00;
                __asm("NOP");
            } 
        }
    }
}

/* ----------------------- interrupt service routine ------------------------ */
void __interrupt OnlyReti(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n OnlyRetn(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n WDT_IRQ(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n SWI_IRQ(void)               // software reset interrupt service routine
{
    SWRST = 0x5a;
    SWRST = 0xa5;
    SWRST = 0xc3;
    SWRST = 0x3c;
}

void __interrupt IntADM_Req(void)
{

}

/* ---------------------- end of interrupt service routine ------------------ */


/* ----------------------- SQ7705 interrupt vector -------------------------- */
#pragma section const ADM_INT_VECTOR
void * const IntTbl1[] = {
    ADM_STARTUP,             /* 0x0000 : RESET               */    //non-maskable
    SWI_IRQ,                 /* 0x0002 : SWI_IRQ/INTUNDEF    */    //non-maskable
    (void *)0xffff,          /* 0x0004 : Reserved            */    
    WDT_IRQ,                 /* 0x0006 : WDT_IRQ             */    //non-maskable interrupt source:watchdog timer
    OnlyReti,                /* 0x0008 : TAMPER_IRQ          */    //interrupt source:tamper detection
    OnlyReti,                /* 0x000A : CFD_IRQ             */    //interrupt source:clock faildetection
    OnlyReti,                /* 0x000C : HVD_IRQ             */
    IntADM_Req,              /* 0x000E : U2A_IRQ/IntADM_Req  */    //interrupt source:user mode to admin mode
    OnlyReti,                /* 0x0010 : RTC_IRQ             */    //interrupt source:real-time clock timer
    OnlyReti,                /* 0x0012 : TMRA00_IRQ          */    //interrupt source:TMRA00 8-bit timer
    OnlyReti,                /* 0x0014 : TMRA01_IRQ          */    //interrupt source:TMRA01 8-bit timer
    OnlyReti,                /* 0x0016 : TMRA10_IRQ          */    //interrupt source:TMRA10 8-bit timer
    OnlyReti,                /* 0x0018 : TMRA11_IRQ          */    //interrupt source:TMRA11 8-bit timer
    OnlyReti,                /* 0x001A : TMRA20_IRQ          */
    OnlyReti,                /* 0x001C : TMRA21_IRQ          */
    OnlyReti,                /* 0x001E : TBT_IRQ             */
    OnlyReti,                /* 0x0020 : RX0_IRQ             */    //interrupt source:UART0 RX0
    OnlyReti,                /* 0x0022 : TX0_IRQ             */    //interrupt source:UART0 TX0
    I2C0_IRQ,                /* 0x0024 : I2C0_IRQ            */    //interrupt source:I2C0
    OnlyReti,                /* 0x0026 : SPI0_IRQ            */    //interrupt source:SPI0
    OnlyReti,                /* 0x0028 : EXT0_IRQ            */    //external interrupt0
    OnlyReti,                /* 0x002A : EXT1_IRQ            */    //external interrupt1
    OnlyReti,                /* 0x002C : EXT2_IRQ            */    //external interrupt2
    OnlyReti,                /* 0x002E : EXT3_IRQ            */    //external interrupt3
    OnlyReti,                /* 0x0030 : ADC_IRQ             */    //interrupt source:ADC
    OnlyReti,                /* 0x0032 : LVD0_IRQ            */
    OnlyReti,                /* 0x0034 : LVD1_IRQ            */
    OnlyReti,                /* 0x0036 : Reserved            */
    OnlyReti,                /* 0x0038 : Reserved            */
    OnlyReti,                /* 0x003A : FMC_IRQ             */    //interrupt source:flash memory controller
    OnlyReti,                /* 0x003C : DMACH0_IRQ          */
    OnlyReti,                /* 0x003E : DMACH1_IRQ          */
    OnlyReti,                /* 0x0040 : DMACH2_IRQ          */
    OnlyReti,                /* 0x0042 : DMACH3_IRQ          */
    OnlyReti,                /* 0x0044 : Reserved            */
    OnlyReti,                /* 0x0046 : Reserved            */
    OnlyReti,                /* 0x0048 : Reserved            */
    OnlyReti,                /* 0x004A : Reserved            */
    OnlyReti,                /* 0x004C : Reserved            */
    OnlyReti,                /* 0x004E : DIC_IRQ             */    //interrupt source:DIC
    OnlyReti,                /* 0x0050 : LEUART_RX_IRQ       */
    OnlyReti,                /* 0x0052 : LEUART_TX_IRQ       */
    OnlyReti,                /* 0x0054 : TRNG_IRQ            */
    OnlyReti,                /* 0x0056 : EXT4_IRQ            */    //external interrupt4
    OnlyReti,                /* 0x0058 : EXT5_IRQ            */    //external interrupt5
    OnlyReti,                /* 0x005A : EXT6_IRQ            */    //external interrupt6
    OnlyReti,                /* 0x005C : EXT7_IRQ            */    //external interrupt7
    OnlyReti,                /* 0x005E : EEMC_IRQ            */    //interrupt source:EEPROM controller
    OnlyReti,                /* 0x0060 : TCA0_IRQ            */    //interrupt source:TCA0 16-bit timer
    OnlyReti,                /* 0x0062 : TCA1_IRQ            */    //interrupt source:TCA1 16-bit timer
    OnlyReti,                /* 0x0064 : STM0_IRQ            */
    OnlyReti,                /* 0x0066 : STM1_IRQ            */
    OnlyReti,                /* 0x0068 : AES_IRQ             */
    OnlyReti,                /* 0x006A : SHA_IRQ             */
    OnlyReti,                /* 0x006C : RX1_IRQ             */    //interrupt source:UART1 RX1
    OnlyReti,                /* 0x006E : TX1_IRQ             */    //interrupt source:UART1 TX1
    OnlyReti,                /* 0x0070 : I2C1_IRQ            */    //interrupt source:I2C1
    OnlyReti,                /* 0x0072 : SPI1_IRQ            */    //interrupt source:SPI1
    OnlyReti,                /* 0x0074 : RX2_IRQ             */    //interrupt source:UART2 RX2
    OnlyReti,                /* 0x0076 : TX2_IRQ             */    //interrupt source:UART2 TX2
    OnlyReti                 /* 0x0078 : TCA2_IRQ            */    //interrupt source:TCA2 16-bit timer
};
#pragma section const
/* -------------------- end of SQ7705 interrupt vector ---------------------- */

