//******************************************************************************
// file         : DRV_SQ7705_SHA.c
// version      : V1.3 2025/03/13
// description  : SHA program file
// note         : 
//******************************************************************************
#include "DRV_SQ7705_SHA.h"

#define SHA_CLOCK_ENABLE 1

//-----------------------
#define CONTI_MODE_OFF 0
#define CONTI_MODE_ON 1

#define MUTIL_BLOCK_OFF 0
#define MUTIL_BLOCK_ON  1

#define BIT_MODE_BYTE 0
#define BIT_MODE_BIT 1

#define PADING_OFF 0
#define PADING_ON 1


//-----------------------
 
//-----------------------
// 0...Disable
// 1...Enable
#define AUTOMODE 1 
//-----------------------

//-----------------------
// 0...Disable
// 1...Enable
#define CONTIMODE 0
//-----------------------

//-----------------------
// 0...Disable
// 1...Enable
#define PAD_ENABLE 0
//-----------------------
#define SHA_BLOCK_SIZE 64

uint8_t Crypto_Clock_Enable = 1;
uint8_t shaInterruptEnable = 0;
uint16_t Start_Address, End_Address;
uint16_t EEPROM_Address = 0x3040;
uint16_t Flash_Address = 0x5000;
uint8_t AREA = 0;
uint16_t SHA_Flag = 0;
uint8_t EEPROM_CHECK[262];
uint8_t FLASH_CHECK[512];
uint8_t digest[32];

struct SHA_Device {
    uint8_t          sha_size;     /*!< sha output size is 16 or 32 bytes */
    uint8_t          initialized;  /*!< initialized: true, false */
    uint8_t          sync;         /*!< sync: true, async: false  */
    uint8_t          padding;      /*!< padding: true, no padding: false */
    uint8_t          processed;    /*!< sha process done */
}sha_dev;
uint32_t contiSize = 0; //contiue mode data counter 

//******************************************************************************
// name         : DRV_SHA_Init()
// description  : SHA initialization
// input param  : sync : Sha process in sync or async
//                padding : TRUE: Padding input data, FALSE: No padding
// output param :
// retval       : Process status.
// note         : 
//******************************************************************************
uint8_t DRV_SHA_Init(uint8_t sync, uint8_t padding)
{
    uint8_t result;
    PRSTR5_SHA = 1;            //reset SHA
    //CLKCR1_CYCTCKEN = Crypto_Clock_Enable;
    //_0x0021.bit.b4 = 1;
    if((result = DRV_HAL_SHA_ClockEnable(SHA_CLOCK_ENABLE)) != SHA_OK){
        return result;
    }

    sha_dev.sync = sync;
    sha_dev.padding = padding;
    sha_dev.initialized = TRUE;
    sha_dev.processed = FALSE;
    return SHA_OK;
}

//******************************************************************************
// name         : DRV_SHA_Deinit()
// description  : SHA deinitialization
// input param  :
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_SHA_Deinit(void)
{
    PRSTR5_SHA = 0;
    if(DRV_HAL_SHA_ClockEnable((uint8_t)!SHA_CLOCK_ENABLE) != SHA_OK){
        while(1);
    }

    sha_dev.initialized = FALSE;
    sha_dev.processed = FALSE;
}

//******************************************************************************
// name         : DRV_SHA_Size()
// description  : 
// input param  : shaSize : 16:SHA128, 32:SHA256
// output param :
// retval       :
// note         : Process status.
//******************************************************************************
uint8_t DRV_SHA_Size(uint8_t shaSize)
{
    if(shaSize == 16 || shaSize == 32)
        sha_dev.sha_size = shaSize;
    else
        return SHA_Err_Mem_Wrong_Size;
    return SHA_OK;
}

//******************************************************************************
// name         : DRV_SHA_Process()
// description  : 
// input param  : memType  : Memory type is SRAM or EEPROM or FLASH
//                input    : Buffer holding the input data
//                length   : Byte length of the input data
//                autoMode : 1: auto process multi-block message, 0: non-auto process multi-block message
//                conti    : 1: execute SHA till byte size count = BSZ setting, 0: do nothing
// output param :
// retval       : Process status.
// note         :
//******************************************************************************
uint8_t DRV_SHA_Process(const uint8_t memType, const uint8_t *input, uint16_t length, uint8_t autoMode, uint8_t conti)
{
    uint8_t result;
    sha_dev.processed = FALSE;
    if(sha_dev.initialized){
        DRV_SHA_Message_Memory(memType);
        if((result = DRV_SHA_Message_Load(memType, (uint8_t *)input, length)) != SHA_OK){
            return result;
        }
        if((result = DRV_SHACR0_Check((uint8_t)!sha_dev.sync, autoMode, conti, sha_dev.padding)) != SHA_OK){
            return result;
        }
        if(sha_dev.sync){
            DRV_SHA_Start_with_Polling();
            if((result = DRV_SHA_Pad_Done_Check(sha_dev.sha_size)) != SHA_OK){
                return result;
            }
        } else{
            DRV_SHA_Start_with_Interrupt();

            if((result = DRV_SHA_Pad_Done_Check(sha_dev.sha_size)) != SHA_OK){
                return result;
            }
        }
    } else{
        return SHA_Err_Not_Initialized;
    }
    sha_dev.processed = TRUE;
    return SHA_OK;
}

//******************************************************************************
// name         : DRV_SHA_Process_DataIn()
// description  : 
// input param  : input    : Buffer holding the input data
//                length   : Byte length of the input data
//                autoMode : 1: auto process multi-block message, 0: non-auto process multi-block message
//                conti    : 1: execute SHA till byte size count = BSZ setting, 0: do nothing
// output param :
// retval       : Process status.
// note         :
//******************************************************************************
uint8_t DRV_SHA_Process_DataIn(const uint8_t *input, uint16_t length, uint8_t autoMode, uint8_t conti)
{
    return DRV_SHA_Process((uint8_t)SRAM, (uint8_t *)input, length, autoMode, conti);
}

//******************************************************************************
// name         : DRV_SHA_Process_EEPROM()
// description  : 
// input param  : startAddress : Start address of the input data
//                length   : Byte length of the input data
//                autoMode : 1: auto process multi-block message, 0: non-auto process multi-block message
//                conti    : 1: execute SHA till byte size count = BSZ setting, 0: do nothing
// output param :
// retval       : Process status.
// note         :
//******************************************************************************
uint8_t DRV_SHA_Process_EEPROM(const uint16_t startAddress, uint16_t length, uint8_t autoMode, uint8_t conti)
{
    return DRV_SHA_Process(EEPROM, (uint8_t *)startAddress, length, autoMode, conti);
}

//******************************************************************************
// name         : DRV_SHA_Process_FLASH()
// description  : 
// input param  : startAddress : Start address of the input data
//                length   : Byte length of the input data
//                autoMode : 1: auto process multi-block message, 0: non-auto process multi-block message
//                conti    : 1: execute SHA till byte size count = BSZ setting, 0: do nothing
// output param :
// retval       : Process status.
// note         :
//******************************************************************************
uint8_t DRV_SHA_Process_FLASH(const uint16_t startAddress, uint16_t length, uint8_t autoMode, uint8_t conti)
{
    return DRV_SHA_Process(FLASH, (uint8_t *)startAddress, length, autoMode, conti);
}

//******************************************************************************
// name         : DRV_SHA_Process_End()
// description  : 
// input param  : output : Buffer holding the output data
// output param :
// retval       : Process status.
// note         :
//******************************************************************************
uint8_t DRV_SHA_Process_End(uint8_t *output)
{
    if(sha_dev.processed){
        DRV_SHA_Load_Digest(output);
        sha_dev.processed = FALSE;
        return SHA_OK;
    } else{
        return SHA_Err_Not_Process;
    }
}

//******************************************************************************
// name         : DRV_HAL_SHA_ClockEnable()
// description  : Enable/Disable SHA control
// input param  : enable : 1:enable, 0:disable
// output param :
// retval       : result: SHA_OK, SHA_ERR
// note         :
//******************************************************************************
uint8_t DRV_HAL_SHA_ClockEnable(uint8_t enable)
{
    PCKEN4_SHA = enable;
    if(PCKEN4_SHA != enable){
        return SHA_Err_Enable;
    }
    return SHA_OK;
}

//******************************************************************************
// name         : DRV_SHA_Interrupt_Init()
// description  : Enable/Disable SHA interrupt
// input param  : shaInterruptEnable : 1:enable, 0:disable
// output param :
// retval       : result : SHA_OK, SHA_ERR
// note         :
//******************************************************************************
uint8_t  DRV_SHA_Interrupt_Init(uint8_t shaInterruptEnable)
{
    IER_SHA = 0;
    SHACR0_IE = 0;
    if(shaInterruptEnable){
        __ASM("DI");
         IER_SHA =shaInterruptEnable;
         __ASM("EI");
        SHACR0_IE = shaInterruptEnable;
    }
    if(IER_SHA != shaInterruptEnable){
        return SHA_Err_Interrupt_Init;
    }

    if(SHACR0_IE != shaInterruptEnable){
        return SHA_Err_Interrupt_Init;
    }
    return SHA_OK;
}

//******************************************************************************
// name         : DRV_SHA_Message_Load()
// description  : Load SHA message to memory
// input param  : memType : 0:SRAM, 1:EEPROM, 2:FLASH
//                msg : message hex byte array
//                msgSize : message hex byte count
// output param :
// retval       : result : SHA_OK, SHA_ERR
// note         : 
//******************************************************************************
uint8_t DRV_SHA_Message_Load(uint8_t memType, uint8_t *msg, uint16_t msgSize)
{
    uint8_t result;
    if(memType == 0){
        if((result = DRV_SHA_Load_SRAM_Message((uint8_t *)msg,msgSize)) != SHA_OK){
            return result;
        }
    } else if(memType == 1){
        if((result = DRV_SHA_Load_EEPROM_Message((uint16_t)msg, msgSize)) != SHA_OK){
            return result;
        }
    } else if(memType == 2){
        if((result = DRV_SHA_Load_Flash_Message((uint16_t)msg, msgSize)) != SHA_OK){
            return result;
        }
    }
    return SHA_OK;
}

//******************************************************************************
// name         : DRV_SHA_Load_SRAM_Message()
// description  : Load SHA message to SRAM
// input param  : msg : message hex byte array
//                msgSize : message hex byte count
// output param :
// retval       : result : SHA_OK, SHA_ERR
// note         :
//******************************************************************************
uint8_t DRV_SHA_Load_SRAM_Message(uint8_t *msg, uint16_t msgSize)
{
    uint8_t result;
    if((result = DRV_SHA_Messgage_Address(msg)) != SHA_OK){
        return result;
    }
    if((result = DRV_SHA_Message_Size(msgSize)) != SHA_OK){
        return result;
    }
    return SHA_OK;
}

//******************************************************************************
// name         : DRV_SHA_Load_Flash_Message()
// description  : Load SHA message to FLASH
// input param  : startAddress : message start address
//                msgSize : message hex byte count
// output param :
// retval       : result : SHA_OK, SHA_ERR
// note         :
//******************************************************************************
uint8_t DRV_SHA_Load_Flash_Message(uint16_t startAddress, uint16_t msgSize)
{
    uint8_t result;
    if((result = DRV_SHA_Messgage_Address((uint8_t *)startAddress)) != SHA_OK){
        return result;
    }
    if((result = DRV_SHA_Message_Size(msgSize)) != SHA_OK){
        return result;
    }
    return SHA_OK;
}

//******************************************************************************
// name         : DRV_SHA_Load_EEPROM_Message()
// description  : Load SHA message to EEPROM
// input param  : startAddress : message start address
//                msgSize : message hex byte count
// output param :
// retval       : result : SHA_OK, SHA_ERR
// note         :
//******************************************************************************
uint8_t DRV_SHA_Load_EEPROM_Message(uint16_t startAddress, uint16_t msgSize)
{
    uint8_t result;
    if((result = DRV_SHA_Messgage_Address((uint8_t *)startAddress)) != SHA_OK){
        return result;
    }
    if((result = DRV_SHA_Message_Size(msgSize)) != SHA_OK){
        return result;
    }
    return SHA_OK;
}

//******************************************************************************
// name         : DRV_SHA_Message_Memory()
// description  : Set memory type for SHA
// input param  : memType : 0:SRAM, 1:EEPROM, 2:FLASH
// output param :
// retval       : result : SHA_OK, SHA_ERR
// note         :
//******************************************************************************
uint8_t DRV_SHA_Message_Memory(uint8_t memType)
{
    SHAADR2 = memType;
    if(SHAADR2 != memType){
        return SHA_Err_Mem_Type;
    }
    return SHA_OK;
}

//******************************************************************************
// name         : DRV_SHA_Messgage_Address()
// description  : Set the input message memory 
// input param  : msg : message address
// output param :
// retval       : result : SHA_OK, SHA_ERR
// note         :
//******************************************************************************
uint8_t DRV_SHA_Messgage_Address(uint8_t *msg)
{
    uint8_t SHAADR1_Temp = SHAADR1 = ((uint16_t)msg >> 8);
    uint8_t SHAADR0_Temp = SHAADR0 = (uint16_t)msg & 0xFF;
    //while(SHACR1_FULL);
    if(SHAADR1_Temp != SHAADR1){
        return SHA_Err_Mem_Addr;
    }
    if(SHAADR0_Temp != SHAADR0){
        return SHA_Err_Mem_Addr;
    }
    return SHA_OK;
}

//******************************************************************************
// name         : DRV_SHA_Message_Size()
// description  : Set the input message size
// input param  : size : message size
// output param :
// retval       : result : SHA_OK, SHA_ERR
// note         :
//******************************************************************************
uint8_t DRV_SHA_Message_Size(uint16_t size)
{
    uint8_t SHABSZ0_Temp = SHABSZ0  = (size & 0xFF);
    uint8_t SHABSZ1_Temp = SHABSZ1  = ((size>>8) & 0xFF);
    uint8_t SHABSZ2_Temp = SHABSZ2  = ((size>>16) & 0xFF);
    uint8_t SHABSZ3_Temp = SHABSZ3  = ((size>>24) & 0xFF);
    //while(SHACR1_FULL);
    // SHABSZ0 = 0x00;
    // SHABSZ1 = 0x00;
    // SHABSZ2 = 0x99;
    // SHABSZ3 = 0x99;

    if(SHABSZ0_Temp != SHABSZ0){
        return SHA_Err_Mem_Size;
    }
    if(SHABSZ1_Temp != SHABSZ1){
        return SHA_Err_Mem_Size;
    }
    
    if(SHABSZ2_Temp != SHABSZ2){
        return SHA_Err_Mem_Size;
    }
    
    if(SHABSZ3_Temp != SHABSZ3){
        return SHA_Err_Mem_Size;
    }
    return SHA_OK;
}

//******************************************************************************
// name         : DRV_SHA_Auto_Enable()
// description  : Enable/Disable auto multi-block processing
// input param  : enable : 1:auto multi-block processing, 0:disable auto
// output param :
// retval       : result : SHA_OK, SHA_ERR
// note         :
//******************************************************************************
uint8_t DRV_SHA_Auto_Enable(uint8_t enable)
{
    SHACR0_AUTO = enable;
    if(SHACR0_AUTO != enable){
        return SHA_Err_Auto;
    }
    return SHA_OK;
}

//******************************************************************************
// name         : DRV_SHA_Pad_Done_Check()
// description  : Check padding digest done
// input param  : digestSize : digest size
// output param :
// retval       : result : SHA_OK, SHA_ERR
// note         :
//******************************************************************************
uint8_t DRV_SHA_Pad_Done_Check(uint16_t digestSize)
{
    if(digestSize % 64 != 0){
        if(SHACR1_PAD_DONE){
            return SHA_OK;
        } else {
            return SHA_Err_Pad;
        }
    }
    return SHA_OK;
}

//******************************************************************************
// name         : DRV_SHACR0_Check()
// description  : Check all the initial setting status
// input param  : shaInterruptEnable : SHA interrupt enable flag
//                autoMode  : 1: auto multi-block processing, 0:disable auto
//                contiMode : 1: Execute SHA till byte size count = SHABSZ, 0: no action
//                padEnable : 1: padding, 0: no padding
// output param : 
// retval       : result: SHA_OK, SHA_ERR
// note         :
//******************************************************************************
uint8_t DRV_SHACR0_Check(uint8_t shaInterruptEnable, uint8_t autoMode, uint8_t contiMode, uint8_t padEnable)
{
    uint8_t result;
    if((result = DRV_SHA_Interrupt_Init(shaInterruptEnable)) != SHA_OK){
        return result;
    }

    if((result = DRV_SHA_Auto_Enable(autoMode)) != SHA_OK){
        return result;
    }

    if((result = DRV_SHA_Conti_Enable(contiMode)) != SHA_OK){
        return result;
    }

    if((result = DRV_SHA_Pad_Enable(padEnable)) != SHA_OK){
        return result;
    }
    return SHA_OK;
}

//******************************************************************************
// name         : DRV_SHA_Start_with_Polling()
// description  : Start SHA calculation and polling done status
// input param  :
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_SHA_Start_with_Polling(void)
{
    SHACR0_START= 1;
    while((SHACR1 & 0x80)!=0) {
        __ASM("NOP");
    }
}

//******************************************************************************
// name         : DRV_SHA_Start_with_Interrupt()
// description  : Start SHA calculation and wait for interrupt
// input param  :
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_SHA_Start_with_Interrupt(void)
{
    SHACR0_START= 1;
    while(!SHA_Flag);
    SHA_Flag = 0;
}

//******************************************************************************
// name         : DRV_SHA_Load_Digest()
// description  : Load digest from SHADO
// input param  :
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_SHA_Load_Digest(uint8_t* digest)
{
    int i = 0;
    for(i=0;i<sha_dev.sha_size;i++){
        *(digest+i) = SHADO;
    }
    __ASM("NOP");
}

//******************************************************************************
// name         : DRV_SHA_Conti_Enable()
// description  : Enable/Disable continue mode
// input param  : enable : 1: Execute SHA till byte size count = SHABSZ, 0: no action
// output param :
// retval       : result : SHA_OK, SHA_ERR
// note         :
//******************************************************************************
uint8_t DRV_SHA_Conti_Enable(uint8_t enable)
{
    SHACR0_CONTI = enable;
    if(SHACR0_CONTI != enable){
        return SHA_Err_Conti;
    }
    return SHA_OK;
}

//******************************************************************************
// name         : DRV_SHA_Pad_Enable()
// description  : Enable/Disable padding mode
// input param  : enable : 1: padding, 0: no padding
// output param :
// retval       : result : SHA_OK, SHA_ERR
// note         :
//******************************************************************************
uint8_t DRV_SHA_Pad_Enable(uint8_t enable)
{
    SHACR0_PAD_EN = enable;
    if(SHACR0_PAD_EN != enable){
        return SHA_Err_Pad;
    }
    return SHA_OK;
}

//******************************************************************************
// name         : Check_4_Multiples()
// description  : Check if the message size is multi-block
// input param  : size : message size
// output param :
// retval       : result : SHA_OK, SHA_ERR
// note         :
//******************************************************************************
uint8_t Check_4_Multiples(uint16_t size)
{
    uint16_t temp = size;
    if(size <=0){
        return SHA_Err_Mem_Size_Negative;
    }
    while(size > 0){
        size = size - 4;
        if(temp < size){
            return SHA_Err_Mem_Size_Not_4_Multiple;
        }
    }
    return SHA_OK;
}

//******************************************************************************
// name         : Check_64()
// description  : Check if the size is 64
// input param  : size : left message size
// output param :
// retval       : result : SHA_OK, SHA_ERR
// note         :
//******************************************************************************
uint8_t Check_64(uint16_t size)
{
    if(size <=0){
        return SHA_Err_Mem_Size_Negative;
    }
    size = size - 0x40;
    if(size != 0){
        return SHA_Err_Mem_Size_Not_64;
    }
    return SHA_OK;
}

//******************************************************************************
// name         : DRV_SHADI_Input()
// description  : input multi-block
// input param  : msg : message hex byte array
//                msgSize : message size
// output param :
// retval       : result : SHA_OK, SHA_ERR
// note         :
//******************************************************************************
uint8_t DRV_SHADI_Input(uint8_t* msg, uint16_t msgSize)
{
    uint8_t result;
    uint8_t temp;
    uint16_t i = 0;
    if((result = Check_4_Multiples(msgSize)) != SHA_OK){
        return result;
    }
    
    for(i=0;i<msgSize;i++){
        SHADI= temp = *(msg + i);
        if(SHADI == temp){
            if(temp!=0){
                return SHA_Err_SHADI; // SHADI is not readable
            }
        }

        if(i+1 < 64 ){
            // SHACR1_FULL is set only when i+1 is 64, in the case of i+1 is less than 64. 
            // Here, i+1 is the true iteration times since the for loop starts from i = 0.
            // When the iteration times i+1 supass 64, the message may still not be ready for 
            if(((result = Check_64(i+1)) != SHA_OK) && SHACR1_FULL){
                return result;
            }
        }
    }
    
    return SHA_OK;
}

//******************************************************************************
// name         : DRV_SHA_256_ContiStart()
// description  : 1.Reset SHA function 
//                2.Reset continue mode process data counter to 0
// input param  :
//
// output param :
// retval       : 
// note         :Example for different size data in SHA continue mode processing 
//               Total message size(totalMsgSize) <= 64 example  
//               For example  totalMsgSize 19
//               >>DRV_SHA_Init(TRUE,TRUE);   
//               >>DRV_SHA_ContiStart(); reset continue mode process counter to 0
//               >>DRV_SHA_ContiFinal(memoryType, messageAddress, 19,digestResult); 
//
//               Total message size(totalMsgSize) > 64 example  
//               For example  totalMsgSize 129
//               >>DRV_SHA_Init(TRUE,TRUE);  
//               >>DRV_SHA_ContiStart(); reset continue mode counter to 0
//               >>DRV_SHA_ContiUpdate(SRAM, messageAddress1);                      //process byte 0~63
//               >>DRV_SHA_ContiUpdate(SRAM, messageAddress2);                      //process byte 64~127 
//               >>DRV_SHA_ContiFinal(memoryType, messageAddress3, 1,digestResult); //process byte 128

//******************************************************************************
void DRV_SHA_ContiStart()
{
    PRSTR5_SHA = 1;        
    PCKEN4_SHA = 1;
    contiSize = 0;
}
//******************************************************************************
// name         : DRV_SHA_256_ContiUpdate()
// description  : SHA continue mode update ,
//                Process 64 byte data SHA256 engine from memAddr
// input param  : memType: memory type
//                           SRAM    0
//                           EEPROM  1
//                           FLASH   2
//                memAddr: memory start address
// retval       : Process status. SHA_OK means OK, other failed 
// note         :
//******************************************************************************
uint8_t DRV_SHA_ContiUpdate(uint8_t memType, uint8_t *memAddr)
{
    uint8_t ret = SHA_OK;
     
    contiSize = contiSize + SHA_BLOCK_SIZE;
    //memroy type
    SHAADR2 = memType;                                       
    //memory address 
    SHAADR1 = ((uint16_t)memAddr >> 8);
    SHAADR0 = (uint16_t)memAddr & 0xFF;  
    //memory size 
    SHABSZ0  = (contiSize & 0xFF);
    SHABSZ1  = ((contiSize >> 8) & 0xFF);
    SHABSZ2  = ((contiSize >> 16) & 0xFF);
    SHABSZ3  = ((contiSize >> 24) & 0xFF);      

    IER_SHA = 0;
    //SHACR0 setting
    SHACR0_IE = 0;
    SHACR0_AUTO = MUTIL_BLOCK_ON;
    SHACR0_CONTI = CONTI_MODE_ON;
    SHACR0_PAD_EN = PADING_ON;
    //Start calcualate SHA
    SHACR0_START = 1;
    //Waiting calculate
    while (1) {
        __ASM("NOP");
        if (SHACR1_BUSY == 0)
            break;
    }
    if (!SHACR1_BLOCK_DONE)
        return SHA_Err;      
    return ret;
}

//******************************************************************************
// name         : DRV_SHA_ContiFinal()
// description  : SHA continue mode final function
//                Process 64 byte data SHA256 engine from memAddr
// input param  : memType: memory type
//                           SRAM    0
//                           EEPROM  1
//                           FLASH   2
//                memAddr: memory start address
//                memSize: 1~64
//                digest : 32 byte data array for SHA256 digest     
// retval       : Process status. SHA_OK means OK, other failed 
// note         :
//******************************************************************************
uint8_t DRV_SHA_ContiFinal(uint8_t memType, uint8_t *memAddr, uint8_t memSize, uint8_t *digest)
{
    uint8_t ret = SHA_OK;
    int i = 0;
    if (memSize < 1 || memSize > 64)
        return SHA_Err;

    contiSize = contiSize + memSize;
    //memory type
    SHAADR2 = memType;
    //memory address 
    SHAADR1 = ((uint16_t)memAddr >> 8);
    SHAADR0 = (uint16_t)memAddr & 0xFF;  
    //memory size 
    SHABSZ0  = (contiSize & 0xFF);
    SHABSZ1  = ((contiSize >> 8) & 0xFF);
    SHABSZ2  = ((contiSize >> 16) & 0xFF);
    SHABSZ3  = ((contiSize >> 24) & 0xFF);      
 
    //SHACR0_Check(0, 0, 1, CONTI_MODE_OFF, 1);
    IER_SHA = 0; //
    //SHACR0 setting
    SHACR0_IE = 0;
    SHACR0_AUTO = MUTIL_BLOCK_ON;
    SHACR0_CONTI = CONTI_MODE_OFF;
    SHACR0_PAD_EN = PADING_ON;
    //Start calcualate SHA
    SHACR0_START = 1;
    while (1) {
        __ASM("NOP");
        if (SHACR1_BUSY == 0)
            break;
    }
    //pad done check only in message size % 64 !=0
    if ((contiSize % 64 != 0) && (SHACR1_PAD_DONE != 1))
        return SHA_Err_Pad;

    //Load digest
    for (i = 0; i < 32; i++) {
        *(digest + i) = SHADO;
    }
    return ret;
}
