//******************************************************************************
// file         : DRV_SQ7705_HASHDRBG.c
// version      : V1.2 2023/10/11
// description  : HashDRBG related functions
// note         : HashDRBG related functions are gathered in this subroutine
//                Must include DRV_SQ7705_SHA and DRV_SQ7705_TRNG
//******************************************************************************
#include "DRV_SQ7705_HASHDRBG.h"

uint8_t hash_buf_V[HASHDRBG_SEED_LEN];
uint8_t hash_buf_CKey[HASHDRBG_SEED_LEN];
uint32_t hash_reseedCounter = 0;

uint8_t DRBG_AddByte(uint8_t *b1, const uint8_t b2)
{
    union{uint16_t word; uint8_t byte[2];} res;

    res.word = *b1 + b2;
    *b1 = res.byte[0];
    return res.byte[1];
}

void DRBG_IncrData(uint8_t *data, uint16_t dataLen)
{
    int16_t i;
    uint8_t carry;
    carry = DRBG_AddByte(data + dataLen - 1, (uint8_t)1);

    for(i = dataLen - 2; i >= 0; i--) {
        if(carry == 0) {
            break;
        } else {
            carry = DRBG_AddByte(data + i, carry);
        }
    }
}

uint8_t DRBG_AddByteWithCarry(uint8_t *b1, uint8_t b2, uint8_t carry)
{
    uint16_t res = *b1 + b2 + carry;
    *b1 = (uint8_t) res;
    return (((uint8_t)(res >> 8))!=0);
}

void DRBG_AddData(uint8_t *data, uint16_t dataLen, uint8_t *data1, uint16_t data1Len)
{
    int16_t i;
    uint8_t carry;
    uint16_t diff = dataLen - data1Len;

    carry = DRBG_AddByte(data + dataLen - 1, data1[data1Len - 1]);

    for(i = data1Len - 2; i >= 0; i--) {
         carry = DRBG_AddByteWithCarry(data + i + diff, data1[i], carry);
    }
    if((carry == 1) && (diff > 0)) {
        DRBG_IncrData(data, diff);
    }
}

void DRBG_AddInt(uint8_t* data, uint16_t dataLen, uint32_t n)
{
    int16_t i;
    uint8_t carry;
    uint16_t diff = dataLen - 8;

    carry = DRBG_AddByte(data+ dataLen - 1, (uint8_t)n);

    for(i = 6; i >= 0; i--) {
         carry = DRBG_AddByteWithCarry(data + i + diff, (uint8_t)(n >> (8 * (7-i))), carry);
    }
    if((carry == 1) && (diff > 0)) {
        DRBG_IncrData(data + diff -1, diff); 
    }
}

void SHA256_Digest_Cal(uint8_t *message, uint8_t *digest, uint16_t msgLen)
{
    uint8_t i;
    uint8_t lastPackageLen = (msgLen>0 && (msgLen%64)==0) ? 64 :(msgLen%64);

    DRV_SHA_Init(TRUE, TRUE);
    DRV_SHA_ContiStart();

    if(msgLen>64) {
        for (i = 1; (i*64) < msgLen; i++) {
            DRV_SHA_ContiUpdate(SRAM, message);
            message+=64;
        }
    }
    DRV_SHA_ContiFinal(SRAM, message, lastPackageLen, digest);
}

void Hash_DF(uint8_t* inputString, uint16_t inputStrLen, uint8_t* requestedBytes, uint8_t requestedLen)
{
    union{uint32_t dword; uint8_t byte[4];} requestedLenDword;
    uint8_t i, len;
    uint8_t counter = 1;
    uint8_t* ptr;
    uint8_t tmp1[64];
    uint8_t buf[190];    //buffer >= (5+input string length)
     
    len = requestedLen/HASHDRBG_OUT_LEN;
    if(requestedLen%HASHDRBG_OUT_LEN)
        len++;
 
    requestedLenDword.dword = (uint32_t)requestedLen * 8;
   
    buf[1] = requestedLenDword.byte[3];
    buf[2] = requestedLenDword.byte[2];
    buf[3] = requestedLenDword.byte[1];
    buf[4] = requestedLenDword.byte[0];
    memcpy(buf+5, inputString, inputStrLen);
    ptr = tmp1;
    for(i=0; i<len; i++) {
       buf[0] = counter;  
       SHA256_Digest_Cal(buf, ptr, 5+inputStrLen);
       counter++;
       ptr += 32;
    }

    memcpy(requestedBytes, tmp1, requestedLen);
}

void Hash_Gen(uint8_t *returnedBytes, uint8_t requestedLength)
{
    uint8_t m,i;    
    uint8_t W[128];
    uint8_t data[HASHDRBG_SEED_LEN];
    uint8_t *ptr;

    /* m = ceil(requestedLength / outlen) */
    m = requestedLength/HASHDRBG_OUT_LEN; 
    if(requestedLength%HASHDRBG_OUT_LEN)
        m++;
    
    /* data = V */
    memcpy(data, hash_buf_V, HASHDRBG_SEED_LEN);
    ptr = W;
    for(i = 1; i <= m; i++) {
        /* W = W || Hash(data) */
        SHA256_Digest_Cal(data, ptr, HASHDRBG_SEED_LEN);
        ptr+=32;
        DRBG_IncrData(data, HASHDRBG_SEED_LEN);
    }

    /* returnedBytes = Leftmost(W, requestedLength) */
    memcpy(returnedBytes, W, requestedLength);
}

void TRNG_GetRandomNumber(uint8_t* outputTRNG, uint8_t len)
{
    uint8_t i, m = len / 4;
    uint8_t trngNumer[4];

    DRV_TRNG_Init();                       // TRNG initialization

    for(i = 0; i < m; i++) {
        DRV_TRNG_Generate(trngNumer);      // generate a 32-bit true random number
        memcpy(outputTRNG, trngNumer, 4);
        outputTRNG+=4;
    }

    if(len % 4 != 0) {                     // If length is not a multiple of 4
        DRV_TRNG_Generate(trngNumer);      // generate a 32-bit true random number
        memcpy(outputTRNG, trngNumer, len % 4);
    }
}

//******************************************************************************
// name         : DRV_HASHDRBG_Instantiate()
// description  : Acquire entropy input, nonce, personalization string
//                Create seed and initial internal states
// input param  : - entropyInput : entropy data
//                - entropyLength : entropy data length
//                - nonce : nonce data
//                - nonceLength : nonce data length
//                - persString : personalization String
//                - persLength : personalization String length
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_HASHDRBG_Instantiate(uint8_t* entropyInput, uint8_t entropyLength,
uint8_t* nonce, uint8_t nonceLength, uint8_t* persString, uint8_t persLength)
{
    uint8_t seed_material[SEED_MATERIAL_LEN];
    uint8_t seed[HASHDRBG_SEED_LEN];

    /* seed_material = entropyInput || Nonce || persString */
    memcpy(seed_material, entropyInput, entropyLength);  
    if((nonce != NULL) && (nonceLength > 0)) {
        memcpy(seed_material + entropyLength, nonce, nonceLength);
    }
    
    if((persString)!=NULL && (persLength>0)) {
        memcpy(seed_material + entropyLength + nonceLength, persString, persLength);     
    }

    /* seed = Hash_DF(seed_material, seedlen) */
    Hash_DF(seed_material, entropyLength + nonceLength + persLength, seed, HASHDRBG_SEED_LEN);
  
    /* V = seed */
    memcpy(hash_buf_V, seed, HASHDRBG_SEED_LEN); 

    /* C = Hash_DF((0x00 || V), seedlen) */
    seed_material[0] = 0x00;
    memcpy(seed_material+1, hash_buf_V, HASHDRBG_SEED_LEN);
    Hash_DF(seed_material, HASHDRBG_SEED_LEN+1, hash_buf_CKey, HASHDRBG_SEED_LEN);

    /* reseedCounter = 1 */
    hash_reseedCounter = 1;
}

//******************************************************************************
// name         : DRV_HASHDRBG_Reseed()
// description  : Acquire new entropy input, Create new seed and internal states
// input param  : - entropyInput : entropy data
//                - entropyLength : entropy data length
//                - additionalInput : additional data
//                - addLength : additional data length
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_HASHDRBG_Reseed(uint8_t* entropyInput, uint8_t entropyLength, 
uint8_t* additionalInput, uint8_t addLength)
{
    uint8_t seed_material[1+HASHDRBG_SEED_LEN+64+64];  // 0x01||V||entropyInput||additionalInput
    uint8_t seed[HASHDRBG_SEED_LEN];
    uint8_t Vt[HASHDRBG_SEED_LEN+1];

    /* seed_material = 0x01 || V || entropyInput || additionalInput */
    seed_material[0] = 0x01;
    memcpy(seed_material+1, hash_buf_V, HASHDRBG_SEED_LEN);
    memcpy(seed_material+1+HASHDRBG_SEED_LEN, entropyInput, entropyLength);
    if((additionalInput != NULL) && (addLength > 0)) {
        memcpy(seed_material + 1 + HASHDRBG_SEED_LEN + entropyLength, additionalInput, addLength);
    }

    /* seed = Hash_DF(seed_material, seedlen) */
    Hash_DF(seed_material, 1+HASHDRBG_SEED_LEN+entropyLength+addLength, seed, HASHDRBG_SEED_LEN);

    /* V = seed */
    memcpy(hash_buf_V, seed, HASHDRBG_SEED_LEN);

    /* C = Hash_DF((0x00 || V), seedlen)*/
    Vt[0] = 0x00;
    memcpy(Vt+1, hash_buf_V, HASHDRBG_SEED_LEN);
    Hash_DF(Vt, HASHDRBG_SEED_LEN+1, hash_buf_CKey, HASHDRBG_SEED_LEN);

    /* reseedCounter = 1 */
    hash_reseedCounter = 1;
}

//******************************************************************************
// name         : DRV_HASHDRBG_Generate()
// description  : Produce pseudorandom bits, Update Internal states
// input param  : - additionalInput : additional data
//                - addLength : additional data length
//                - entropyPRInput : entropy prediction data
//                - entropyPRLength : entropy prediction data length
//                - returnedBytes : returned data
//                - requestedLength : returned data length
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_HASHDRBG_Generate(uint8_t* additionalInput, uint8_t addLength, 
uint8_t* entropyPRInput, uint8_t entropyPRLength, uint8_t* returnedBytes, uint8_t requestedLength)
{
    uint8_t w[HASHDRBG_OUT_LEN];
    uint8_t H[HASHDRBG_OUT_LEN];
    uint8_t tmp1[1+HASHDRBG_SEED_LEN];
    uint8_t *V = hash_buf_V;
    uint8_t *C = hash_buf_CKey;
          
    if((entropyPRInput != NULL) && (entropyPRLength > 0)) {
        DRV_HASHDRBG_Reseed(entropyPRInput,entropyPRLength,additionalInput,addLength);
        additionalInput = NULL;
    }
    
    if( (additionalInput != NULL) && (addLength > 0)) {
         /* w = Hash(0x02 || V || additionalInput) */
         uint8_t tmp[1+HASHDRBG_SEED_LEN+64];  

         tmp[0] = 0x02;
         memcpy(tmp+1, V, HASHDRBG_SEED_LEN);
         memcpy(tmp+1+HASHDRBG_SEED_LEN, additionalInput, addLength);
         SHA256_Digest_Cal(tmp,w,1+HASHDRBG_SEED_LEN+addLength);

         /* V = (V + w) mod 2^seedlen */
         DRBG_AddData(V, HASHDRBG_SEED_LEN, w, HASHDRBG_OUT_LEN);
    }
    /* (returnedBytes) = Hash_Gen(requestedLength, V) */
    Hash_Gen(returnedBytes, requestedLength);

    /* H = Hash(0x03 || V) */
    tmp1[0] = 0x03;
    memcpy(tmp1+1, V, HASHDRBG_SEED_LEN);
    SHA256_Digest_Cal(tmp1, H, 1+HASHDRBG_SEED_LEN);

    /* V = (V + H + C + reseedCounter ) mod 2^seedlen */
    DRBG_AddData(V, HASHDRBG_SEED_LEN, H, HASHDRBG_OUT_LEN);
    DRBG_AddData(V, HASHDRBG_SEED_LEN, C, HASHDRBG_SEED_LEN);
    DRBG_AddInt(V, HASHDRBG_SEED_LEN, hash_reseedCounter);

    /* reseedCounter = reseedCounter + 1 */
    hash_reseedCounter += 1;
}

//******************************************************************************
// name         : DRV_HASHDRBG_GetRandomNumber()
// description  : generate 32-byte random numbers using HashDRBG 
// output param : - outputDRBG : 32-byte random numbers
// retval       : 
// note         : 
//******************************************************************************
void DRV_HASHDRBG_GetRandomNumber(uint8_t* outputDRBG)
{
    uint8_t DRBG_Entropy[32];
    uint8_t DRBG_Nonce[16];

    TRNG_GetRandomNumber(DRBG_Entropy, 32);  // Get 32 bytes random number for entropy from TRNG
    TRNG_GetRandomNumber(DRBG_Nonce, 16);    // Get 16 bytes random number for nonce from TRNG
    // DRBG Instantiate  , Entropy = 32 bytes , Nonce = 16 bytes . Personal data = 0 byte
    DRV_HASHDRBG_Instantiate(DRBG_Entropy, 32, DRBG_Nonce, 16, NULL, 0);
    TRNG_GetRandomNumber(DRBG_Entropy, 32);  // Get 32 bytes random number for entropyPR from TRNG
    // DRBG Generate  , EntropyPR = 32 bytes
    DRV_HASHDRBG_Generate(NULL, 0, DRBG_Entropy, 32, outputDRBG, 32);
}
