//******************************************************************************
// file         : DRV_SQ7705_HMAC.c
// version      : V1.2 202310/04
// description  : HMAC with SHA256 related functions
// note         :
//HMAC=H(K XOR opad, H(K XOR ipad, text))
//H SHA256 hash function
//K for secret key. Key Length 32 byte
//B=64 the byte-length of blocks ,
//ipad = the byte 0x36 repeated B times
//opad = the byte 0x5C repeated B times.
//text input message
//******************************************************************************
#include "DRV_SQ7705_HMAC.h"
#include "DRV_SQ7705_SHA.h"
#define OUTER_PAD 0x5C
#define INNER_PAD 0x36
//******************************************************************************
// name         : DRV_HMAC_Init
// description  : initial K XOR opad , K XOR ipad
// input param  :
// output param :
// retval       :
// note         :
// HMAC=H(K XOR opad, H(K XOR ipad, text))
// H SHA256 hash function
// K for secret key. Key Length 32 byte
// B=64 the byte-length of blocks ,
// ipad = the byte 0x36 repeated B times
// opad = the byte 0x5C repeated B times.
// text input message
// Example for different size data in HMAC processing 
// Total message size(totalMsgSize) <= 64 example  
// For example  totalMsgSize 18               
// DRV_HMAC_Init(hmacInfo,keyAddress) intial hmac 
// DRV_HMAC_Final(hmacInfo,memoryType, messageAddress, 18); process byte 0~17
// Total message size(totalMsgSize) > 64 example  
// For example  totalMsgSize 129
// DRV_HMAC_Init(hmacInfo,keyAddress);  
// DRV_HMAC_Update(hmacInfo, memType ,msgAddress1) 
// DRV_HMAC_Update(hmacInfo, memType ,msgAddress2)
// DRV_HMAC_Final(hmacInfo,memoryType, messageAddress3, 1);  
//******************************************************************************
void DRV_HMAC_Init(HMAC_SHA256_INFO *hmac,uint8_t *keyAddress)
{
    int i = 0;
    
    memset(hmac->keyInpad, 0, SHA256_BLOCK_LEN);
    memset(hmac->keyOpad, 0, SHA256_BLOCK_LEN);   
    memcpy(hmac->keyInpad, keyAddress, 32);    
    memcpy(hmac->keyOpad, keyAddress, 32);
    
    for (i = 0; i < SHA256_BLOCK_LEN; i++) {
        hmac->keyInpad[ i ] = hmac->keyInpad[ i ] ^ INNER_PAD;
        hmac->keyOpad[ i ] = hmac->keyOpad[ i ] ^ OUTER_PAD;
    }
    DRV_SHA_Init(1, 1);
    //Start inner hash  H(K XOR ipad, text)
    DRV_SHA_ContiStart();    
    DRV_SHA_ContiUpdate(SRAM, hmac->keyInpad);

}
//******************************************************************************
// name         : DRV_HMAC_Update
// description  : process 64 byte message from message address
// input param  :-hmac HMAC  information
//               -memType    memory type SRAM,EEPROM,FLASH                          
//               -msgAddress memory address
//               
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_HMAC_Update(HMAC_SHA256_INFO *hmac, uint8_t memType ,uint8_t *msgAddress)
{    
    DRV_SHA_ContiUpdate(memType, msgAddress);
}
//******************************************************************************
// name         : DRV_HMAC_Final
// description  : final step of HMAC to get MAC
// input param  :-hmac HMAC information
//               -msgAddress message address
//               -msgLen message length 1~64 byte
// output param : hmac->digest 32 byte message digest
// retval       :
// note         :
// HMAC=H(K XOR opad, H(K XOR ipad, text))
//******************************************************************************
uint8_t DRV_HMAC_Final(HMAC_SHA256_INFO *hmac, uint8_t memType, uint8_t *msgAddress, uint8_t msgLen)
{
    if(msgLen<1||msgLen>64)
        return HMAC_LEN_ERR;
    //Finish inner hash  H(K XOR ipad, text)     
    DRV_SHA_ContiFinal(memType, msgAddress, msgLen, hmac->digestInnerHash);
    //Process H(K XOR opad, inner hash result)
    DRV_SHA_ContiStart();
    DRV_SHA_ContiUpdate(memType, hmac->keyOpad);
    DRV_SHA_ContiFinal(memType, hmac->digestInnerHash, 32, hmac->mac);
    return HMAC_OK;
}