//******************************************************************************
// file         : DRV_SQ7705_FLASH.c
// version      : V1.2 2023/08/31
// description  : FLASH Momory related functions
// note         : FLASH Momory related functions are gathered in this subroutine
//******************************************************************************
#include "DRV_SQ7705_FLASH.h" 

//******************************************************************************
// name         : DRV_FLASH_Init()
// description  : initialize flash clock(FCKDIV)
// input param  : 
// output param : 
// retval       : 
// note         : sysclk : System clock frequency in MHz
//******************************************************************************
void DRV_FLASH_Init(uint8_t sysclk)
{
    FCKDIV = sysclk - 1;
}

//******************************************************************************
// name         : DRV_FLASH_EraseSector()
// description  : sector erase of flash memory
// input param  : - address : start address to erase
// output param : 
// retval       : - 0 : erase successful
//                - 1 : erase fail
// note         : erase 32 bytes
//******************************************************************************
uint8_t DRV_FLASH_EraseSector(uint16_t address)
{
    uint8_t status;                             // status variable
    DRV_FERR_CLEAR();
    
    FADDR0 = (uint8_t)address & 0x00FF;         // set start address
    FADDR1 = (uint8_t)(address >> 8) & 0x00FF;
    
    FCR = 0x18;                                 // sector erase command

    FDATA0 = 0xA5;                              // A5, 5A 
    FDATA0 = 0x5A;                              // A5, 5A 
    
    do{                                         // wait for sector erase completed
        status = FSR;
        status &= 0x80;
    }while(status);                             // when status is busy(FSR_BUSY=1)

    return(FSR_FERR);                           // return error message
}

//******************************************************************************
// name         : DRV_FLASH_WriteByte()
// description  : program one byte data in the flash memory
// input param  : - address : program address
//                - data : program data
// output param : 
// retval       : - 0 : program successful
//                - 1 : program fail
// note         : 
//******************************************************************************
uint8_t DRV_FLASH_WriteByte(uint16_t address, uint8_t data)
{
    uint8_t status;                             // status variable
    DRV_FERR_CLEAR();
    
    FDATA0 = data;                              // write data to flash data register  
    
    FADDR0 = (uint8_t)address & 0x00FF;         // set address
    FADDR1 = (uint8_t)(address >> 8) & 0x00FF;

    FCR = 0x04;                                 // program command  
    FDATA0 = 0xA5;                              // A5, 5A 
    FDATA0 = 0x5A;                              // A5, 5A 
    
    do{                                         // wait for programming completed
        status = FSR;
        status &= 0x80;
    }while(status);                             // when status is busy(FSR_BUSY=1)
    
    return(FSR_FERR);                           // return error message
}

//******************************************************************************
// name         : DRV_FLASH_ReadByte()
// description  : read one byte data in the flash memory
// input param  : - address : read address
// output param : - data : data in the flash memory
// retval       : 
// note         : 
//******************************************************************************
void DRV_FLASH_ReadByte(uint16_t address, uint8_t *data)
{
    DRV_FERR_CLEAR();
    FADDR0 = (unsigned char)address & 0x00FF;         // set address
    FADDR1 = (unsigned char)(address >> 8) & 0x00FF;
   
    FCR = 0x00;                                       // read command
    *data = FDATA0; 
}

//******************************************************************************
// name         : DRV_FERR_CLEAR()
// description  : clear error status
// input param  : 
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_FERR_CLEAR()
{
    if(FSR_FERR != 0){
        FSR_FERR = 0x07;
    }
}