//******************************************************************************
// file         : DRV_SQ7705_EINT.h
// version      : V1.2 2023/08/28
// description  : EINT header file
// note         : 
//******************************************************************************
#ifndef _DRV_SQ7705_EINT_H__
#define _DRV_SQ7705_EINT_H__

#include "SQ7705_SFR.h"
#include "TypeDefs.h"

extern uint8_t EXT0_Int_Flag;    // external interrupt flag
extern uint8_t EXT1_Int_Flag;    // external interrupt flag
extern uint8_t EXT2_Int_Flag;    // external interrupt flag
extern uint8_t EXT3_Int_Flag;    // external interrupt flag
extern uint8_t EXT4_Int_Flag;    // external interrupt flag
extern uint8_t EXT5_Int_Flag;    // external interrupt flag
extern uint8_t EXT6_Int_Flag;    // external interrupt flag
extern uint8_t EXT7_Int_Flag;    // external interrupt flag

// EINTCRX INTSEL[2:0]  bit 5~7
// (EINTCR_Address,EINTCRX INTSEL[2:0]<<5)
//EINT 0
#define EINT0_P00  0, 0x0
#define EINT0_P10  0, 0x1
#define EINT0_P20  0, 0x2
#define EINT0_P50  0, 0x5
#define EINT0_P60  0, 0x6
#define EINT0_P70  0, 0x7
//EINT 1
#define EINT1_P01  1, 0x0
#define EINT1_P11  1, 0x1
#define EINT1_P21  1, 0x2
#define EINT1_P51  1, 0x5
#define EINT1_P61  1, 0x6
#define EINT1_P71  1, 0x7
//EINT 2
#define EINT2_P02  2, 0x0
#define EINT2_P12  2, 0x1
#define EINT2_P22  2, 0x2
#define EINT2_P32  2, 0x3
#define EINT2_P42  2, 0x4
#define EINT2_P52  2, 0x5
#define EINT2_P62  2, 0x6
#define EINT2_P72  2, 0x7
//EINT 3
#define EINT3_P03  3, 0x0
#define EINT3_P13  3, 0x1
#define EINT3_P23  3, 0x2
#define EINT3_P33  3, 0x3
#define EINT3_P43  3, 0x4
#define EINT3_P53  3, 0x5
#define EINT3_P63  3, 0x6
//EINT 4
#define EINT4_P04  4, 0x0
#define EINT4_P14  4, 0x1
#define EINT4_P24  4, 0x2
#define EINT4_P34  4, 0x3
#define EINT4_P44  4, 0x4
#define EINT4_P54  4, 0x5
#define EINT4_P64  4, 0x6
#define EINT4_P74  4, 0x7
//EINT 5
#define EINT5_P05  5, 0x0
#define EINT5_P15  5, 0x1
#define EINT5_P25  5, 0x2
#define EINT5_P35  5, 0x3
#define EINT5_P45  5, 0x4
#define EINT5_P55  5, 0x5
#define EINT5_P65  5, 0x6
#define EINT5_P75  5, 0x7
#define EINT6_P06  6, 0x0
//EINT 6
#define EINT6_P16  6, 0x1
#define EINT6_P26  6, 0x2
#define EINT6_P36  6, 0x3
#define EINT6_P46  6, 0x4
#define EINT6_P56  6, 0x5
#define EINT6_P66  6, 0x6
//EINT 7
#define EINT7_P07  7, 0x0
#define EINT7_P17  7, 0x1
#define EINT7_P37  7, 0x3
#define EINT7_P47  7, 0x4
#define EINT7_P57  7, 0x5

//EINTCRX INTxVL       bit 4      noise canceller pass signal level when the interrupt request signal is generated
typedef enum {
    INT_VL_LOW = 0,
    INT_VL_HIGH
} INT_VL;
//EINTCRX INTxES[1:0]  bit 2~3    selects the interrupt request generating condition
typedef enum {
    INTES_RISING = 0,
    INTES_FALLING,
    INTES_RIS_FAL
} TRIGGER_TYPE;
//EINTCRX INTxINC[1:0] bit 1~0    sets the noise canceller sampling interval
typedef enum {
    FSYSCLK_DIV_1 = 0x0,
    FSYSCLK_DIV_4 = 0x01,
    FSYSCLK_DIV_8 = 0x02,
    FSYSCLK_DIV_16 = 0x03
} NOISE_SAMPLE;

//******************************************************************************
// name         : DRV_EINT_Init()
// description  : external interrupt initialization
// input param  : - eintNum : EINT number
//                - intSel  : selects the external interrupt pin
//                - intLvl  : noise canceller pass signal level when the interrupt request signal is generated
//                - intEs   : selects the interrupt request generating condition
//                            INTES_RISING = 0,  an interrupt request is generated at the rising edge
//                            INTES_FALLING = 1, an interrupt request is generated at the falling edge
//                            INTES_RIS_FAL = 2   an interrupt request is generated at the both edges
//                - sampleClock  : sets the noise canceller sampling interval
// retval       : none (all parameters can only use the defined values, otherwise the function will be disabled)
// note         :
//   EINTCRX INTSEL[2:0]  bit 5~7   selects the external interrupt pin
//   EINTCRX INTxVL       bit 4     noise canceller pass signal level when the interrupt request signal is generated
//   EINTCRX INTxES[1:0]  bit 2~3   selects the interrupt request generating condition
//   EINTCRX INTxINC[1:0] bit 1~0   sets the noise canceller sampling interval
//******************************************************************************
void DRV_EINT_Init(uint8_t eintNum, uint8_t intSel, INT_VL intLvl, TRIGGER_TYPE intEs, NOISE_SAMPLE sampleClock);

//******************************************************************************
// name         : DRV_EINT_DeInit()
// description  : external interrupt deinitialization
// input param  : - eintNum : EINT number 0~7
// retval       : none (the parameters can only use the value of 0~7, otherwise the function will be invalid)
// note         :
//******************************************************************************
void DRV_EINT_DeInit(uint8_t eintNum);

//******************************************************************************
// name         : DRV_EINT_Enable()
// description  : enable external interrupt
// input param  : - eintNum : EINT number 0~7
// retval       : none (the parameters can only use the value of 0~7, otherwise the function will be invalid)
// note         :
// interrupt enable registerg
// #define     IER2                      _0x01B2.byte
// #define     IER_EXT3                  _0x01B2.bit.b7
// #define     IER_EXT2                  _0x01B2.bit.b6
// #define     IER_EXT1                  _0x01B2.bit.b5
// #define     IER_EXT0                  _0x01B2.bit.b4
// #define     IER5                      _0x01B5.byte
// #define     IER_EXT7                  _0x01B5.bit.b6
// #define     IER_EXT6                  _0x01B5.bit.b5
// #define     IER_EXT5                  _0x01B5.bit.b4
// #define     IER_EXT4                  _0x01B5.bit.b3
// interrupt flag register
// #define     IFR2                      _0x01A2.byte
// #define     IFR_EXT3                  _0x01A2.bit.b7
// #define     IFR_EXT2                  _0x01A2.bit.b6
// #define     IFR_EXT1                  _0x01A2.bit.b5
// #define     IFR_EXT0                  _0x01A2.bit.b4
// #define     IFR5                      _0x01A5.byte
// #define     IFR_EXT7                  _0x01A5.bit.b6
// #define     IFR_EXT6                  _0x01A5.bit.b5
// #define     IFR_EXT5                  _0x01A5.bit.b4
// #define     IFR_EXT4                  _0x01A5.bit.b3
//******************************************************************************
void DRV_EINT_Enable(uint8_t eintNum);

//******************************************************************************
// name         : DRV_EINT_Disable()
// description  : disable external interrupt
// input param  : - eintNum : EINT number 0~7
// retval       : none (the parameters can only use the value of 0~7, otherwise the function will be invalid)
// note         :
// interrupt enable register
// #define     IER2                      _0x01B2.byte
// #define     IER_EXT3                  _0x01B2.bit.b7
// #define     IER_EXT2                  _0x01B2.bit.b6
// #define     IER_EXT1                  _0x01B2.bit.b5
// #define     IER_EXT0                  _0x01B2.bit.b4
// #define     IER5                      _0x01B5.byte
// #define     IER_EXT7                  _0x01B5.bit.b6
// #define     IER_EXT6                  _0x01B5.bit.b5
// #define     IER_EXT5                  _0x01B5.bit.b4
// #define     IER_EXT4                  _0x01B5.bit.b3
// interrupt flag register
// #define     IFR2                      _0x01A2.byte
// #define     IFR_EXT3                  _0x01A2.bit.b7
// #define     IFR_EXT2                  _0x01A2.bit.b6
// #define     IFR_EXT1                  _0x01A2.bit.b5
// #define     IFR_EXT0                  _0x01A2.bit.b4
// #define     IFR5                      _0x01A5.byte
// #define     IFR_EXT7                  _0x01A5.bit.b6
// #define     IFR_EXT6                  _0x01A5.bit.b5
// #define     IFR_EXT5                  _0x01A5.bit.b4
// #define     IFR_EXT4                  _0x01A5.bit.b3
//******************************************************************************
void DRV_EINT_Disable(uint8_t eintNum);

//******************************************************************************
// name         : DRV_EXT0_IRQ()
// description  : EINT0 interrupt service routine
// input param  :
// retval       : none
// note         :
//******************************************************************************
void __interrupt DRV_EXT0_IRQ(void);

//******************************************************************************
// name         : DRV_EXT1_IRQ()
// description  : EINT1 interrupt service routine
// input param  :
// retval       : none
// note         :
//******************************************************************************
void __interrupt DRV_EXT1_IRQ(void);

//******************************************************************************
// name         : DRV_EXT2_IRQ()
// description  : EINT2 interrupt service routine
// input param  :
// retval       : none
// note         :
//******************************************************************************
void __interrupt DRV_EXT2_IRQ(void);

//******************************************************************************
// name         : DRV_EXT3_IRQ()
// description  : EINT3 interrupt service routine
// input param  :
// retval       : none
// note         :
//******************************************************************************
void __interrupt DRV_EXT3_IRQ(void);

//******************************************************************************
// name         : DRV_EXT4_IRQ()
// description  : EINT4 interrupt service routine
// input param  :
// retval       : none
// note         :
//******************************************************************************
void __interrupt DRV_EXT4_IRQ(void);

//******************************************************************************
// name         : DRV_EXT5_IRQ()
// description  : EINT5 interrupt service routine
// input param  :
// retval       : none
// note         :
//******************************************************************************
void __interrupt DRV_EXT5_IRQ(void);

//******************************************************************************
// name         : DRV_EXT6_IRQ()
// description  : EINT6 interrupt service routine
// input param  :
// retval       : none
// note         :
//******************************************************************************
void __interrupt DRV_EXT6_IRQ(void);

//******************************************************************************
// name         : DRV_EXT7_IRQ()
// description  : EINT7 interrupt service routine
// input param  :
// retval       : none
// note         :
//******************************************************************************
void __interrupt DRV_EXT7_IRQ(void);

#endif
