//******************************************************************************
// file         : DRV_SQ7705_EINT.c
// version      : V1.2 2023/08/28
// description  : EINT related functions
// note         : EINT related functions are gathered in this subroutine
//******************************************************************************
#include "DRV_SQ7705_EINT.h"

#define  PCKEN3_ADDR 0x017B       // peripheral clock enable register for EINT,bit0 EINT0,bit1 EINT1,....,bit7 EINT7
#define  EINTCR_BASE_ADDR 0x0190  // EINT control SFR base address 0x0190~0x0197, EINT0~EINT8
#define  IFR2_ADDR        0x01A2  // interrupt flag register EINT 0~3
#define  IFR5_ADDR        0x01A5  // interrupt flag register EINT 4~7
#define  IER2_ADDR        0x01B2  // interrupt enable register EINT0~3
#define  IER5_ADDR        0x01B5  // interrupt enable register EINT4~7

uint8_t EXT0_Int_Flag;            // external interrupt flag
uint8_t EXT1_Int_Flag;            // external interrupt flag
uint8_t EXT2_Int_Flag;            // external interrupt flag
uint8_t EXT3_Int_Flag;            // external interrupt flag
uint8_t EXT4_Int_Flag;            // external interrupt flag
uint8_t EXT5_Int_Flag;            // external interrupt flag
uint8_t EXT6_Int_Flag;            // external interrupt flag
uint8_t EXT7_Int_Flag;            // external interrupt flag
uint8_t *eintEnableAddr;          // pointer to peripheral clock enable register
uint8_t *eintCRAddr;              // pointer to EINT control register
uint8_t *eintIFRAddr;             // pointer to interrupt enable register
uint8_t *eintFuncEnableAddr;      // pointer to interrupt enable register

//******************************************************************************
// name         : DRV_EINT_Init()
// description  : external interrupt initialization
// input param  : - eintNum : EINT number
//                - intSel  : selects the external interrupt pin
//                - intLvl  : noise canceller pass signal level when the interrupt request signal is generated
//                - intEs   : selects the interrupt request generating condition
//                            INTES_RISING = 0,  an interrupt request is generated at the rising edge
//                            INTES_FALLING = 1, an interrupt request is generated at the falling edge
//                            INTES_RIS_FAL = 2   an interrupt request is generated at the both edges
//                - sampleClock  : sets the noise canceller sampling interval
// retval       : none (all parameters can only use the defined values, otherwise the function will be disabled)
// note         :
//   EINTCRX INTSEL[2:0]  bit 5~7   selects the external interrupt pin
//   EINTCRX INTxVL       bit 4     noise canceller pass signal level when the interrupt request signal is generated
//   EINTCRX INTxES[1:0]  bit 2~3   selects the interrupt request generating condition
//   EINTCRX INTxINC[1:0] bit 1~0   sets the noise canceller sampling interval
//******************************************************************************
void DRV_EINT_Init(uint8_t eintNum, uint8_t intSel, INT_VL intLvl, TRIGGER_TYPE intEs, NOISE_SAMPLE sampleClock)
{
    // enable EINTx  *this must be enabled first, the relevant staging settings later to be useful
    eintEnableAddr = (uint8_t *)PCKEN3_ADDR;
    *eintEnableAddr |= 0x1 << eintNum;
    // pointer to EINT control register
    eintCRAddr = (uint8_t *)(EINTCR_BASE_ADDR + eintNum);
    *eintCRAddr = intSel << 5 | intLvl << 4 | intEs << 2 | sampleClock;
}

//******************************************************************************
// name         : DRV_EINT_DeInit()
// description  : external interrupt deinitialization
// input param  : - eintNum : EINT number 0~7
// retval       : none (the parameters can only use the value of 0~7, otherwise the function will be invalid)
// note         :
//******************************************************************************
void DRV_EINT_DeInit(uint8_t eintNum)
{
    //external interrupt deinitialization
    eintEnableAddr = (uint8_t *) PCKEN3_ADDR ;
    *eintEnableAddr &= ~(0x1 << eintNum);
}

//******************************************************************************
// name         : DRV_EINT_Enable()
// description  : enable external interrupt
// input param  : - eintNum : EINT number 0~7
// retval       : none (the parameters can only use the value of 0~7, otherwise the function will be invalid)
// note         :
// interrupt enable registerg
// #define     IER2                      _0x01B2.byte
// #define     IER_EXT3                  _0x01B2.bit.b7
// #define     IER_EXT2                  _0x01B2.bit.b6
// #define     IER_EXT1                  _0x01B2.bit.b5
// #define     IER_EXT0                  _0x01B2.bit.b4
// #define     IER5                      _0x01B5.byte
// #define     IER_EXT7                  _0x01B5.bit.b6
// #define     IER_EXT6                  _0x01B5.bit.b5
// #define     IER_EXT5                  _0x01B5.bit.b4
// #define     IER_EXT4                  _0x01B5.bit.b3
// interrupt flag register
// #define     IFR2                      _0x01A2.byte
// #define     IFR_EXT3                  _0x01A2.bit.b7
// #define     IFR_EXT2                  _0x01A2.bit.b6
// #define     IFR_EXT1                  _0x01A2.bit.b5
// #define     IFR_EXT0                  _0x01A2.bit.b4
// #define     IFR5                      _0x01A5.byte
// #define     IFR_EXT7                  _0x01A5.bit.b6
// #define     IFR_EXT6                  _0x01A5.bit.b5
// #define     IFR_EXT5                  _0x01A5.bit.b4
// #define     IFR_EXT4                  _0x01A5.bit.b3
//******************************************************************************
void DRV_EINT_Enable(uint8_t eintNum)
{
    __ASM("DI"); // disable all maskable interrupts

    if (eintNum < 4 ) {
        //EINT0~3
        // clear EINT interrupt flag
        eintIFRAddr = (uint8_t *) IFR2_ADDR;
        *(eintIFRAddr) &= ~(1 << (eintNum + 4));
        // set EINT enable register to enable EINT
        eintFuncEnableAddr = (uint8_t *)IER2_ADDR;
        *eintFuncEnableAddr |= (1 << eintNum + 4);
    } else { //EINT4~7
        // clear EINT interrupt flag
        eintIFRAddr = (uint8_t *)IFR5_ADDR;
        *(eintIFRAddr) &= ~(1 << (eintNum - 1));
        // set EINT enable register to enable EINT
        eintFuncEnableAddr = (uint8_t *)IER5_ADDR;
        *eintFuncEnableAddr |= (1 << (eintNum - 1));
    }

    // IFR_EXT0 = 0; // clear EINT0 interrupt flag
    // IER_EXT0 = 1; // set EINT0 enable register to enable EINT0
    __ASM("EI"); // enable all maskable interrupts
}

//******************************************************************************
// name         : DRV_EINT_Disable()
// description  : disable external interrupt
// input param  : - eintNum : EINT number 0~7
// retval       : none (the parameters can only use the value of 0~7, otherwise the function will be invalid)
// note         :
// interrupt enable register
// #define     IER2                      _0x01B2.byte
// #define     IER_EXT3                  _0x01B2.bit.b7
// #define     IER_EXT2                  _0x01B2.bit.b6
// #define     IER_EXT1                  _0x01B2.bit.b5
// #define     IER_EXT0                  _0x01B2.bit.b4
// #define     IER5                      _0x01B5.byte
// #define     IER_EXT7                  _0x01B5.bit.b6
// #define     IER_EXT6                  _0x01B5.bit.b5
// #define     IER_EXT5                  _0x01B5.bit.b4
// #define     IER_EXT4                  _0x01B5.bit.b3
// interrupt flag register
// #define     IFR2                      _0x01A2.byte
// #define     IFR_EXT3                  _0x01A2.bit.b7
// #define     IFR_EXT2                  _0x01A2.bit.b6
// #define     IFR_EXT1                  _0x01A2.bit.b5
// #define     IFR_EXT0                  _0x01A2.bit.b4
// #define     IFR5                      _0x01A5.byte
// #define     IFR_EXT7                  _0x01A5.bit.b6
// #define     IFR_EXT6                  _0x01A5.bit.b5
// #define     IFR_EXT5                  _0x01A5.bit.b4
// #define     IFR_EXT4                  _0x01A5.bit.b3
//******************************************************************************
void DRV_EINT_Disable(uint8_t eintNum)
{
    if (eintNum < 4 ) { //EINT0~3
        //disable EINT
        eintFuncEnableAddr = (uint8_t *) IER2_ADDR;
        *eintFuncEnableAddr &= ~(1 << eintNum + 4);
    } else { //EINT4~7
        //disable EINT
        eintFuncEnableAddr = (uint8_t *) IER5_ADDR;
        *eintFuncEnableAddr &= ~(1 << (eintNum - 1));
    }
}

//******************************************************************************
// name         : DRV_EXT0_IRQ()
// description  : EINT0 interrupt service routine
// input param  :
// retval       : none
// note         :
//******************************************************************************
void __interrupt DRV_EXT0_IRQ(void)
{
    EXT0_Int_Flag = 1;  // Set interrupt flag to 1
}

//******************************************************************************
// name         : DRV_EXT1_IRQ()
// description  : EINT1 interrupt service routine
// input param  :
// retval       : none
// note         :
//******************************************************************************
void __interrupt DRV_EXT1_IRQ(void)
{
    EXT1_Int_Flag = 1;  // Set interrupt flag to 1
}

//******************************************************************************
// name         : DRV_EXT2_IRQ()
// description  : EINT2 interrupt service routine
// input param  :
// retval       : none
// note         :
//******************************************************************************
void __interrupt DRV_EXT2_IRQ(void)
{
    EXT2_Int_Flag = 1;  // Set interrupt flag to 1
}

//******************************************************************************
// name         : DRV_EXT3_IRQ()
// description  : EINT3 interrupt service routine
// input param  :
// retval       : none
// note         :
//******************************************************************************
void __interrupt DRV_EXT3_IRQ(void)
{
    EXT3_Int_Flag = 1;  // Set interrupt flag to 1
}

//******************************************************************************
// name         : DRV_EXT4_IRQ()
// description  : EINT4 interrupt service routine
// input param  :
// retval       : none
// note         :
//******************************************************************************
void __interrupt DRV_EXT4_IRQ(void)
{
    EXT4_Int_Flag = 1;  // Set interrupt flag to 1
}

//******************************************************************************
// name         : DRV_EXT5_IRQ()
// description  : EINT5 interrupt service routine
// input param  :
// retval       : none
// note         :
//******************************************************************************
void __interrupt DRV_EXT5_IRQ(void)
{
    EXT5_Int_Flag = 1;  // Set interrupt flag to 1
}

//******************************************************************************
// name         : DRV_EXT6_IRQ()
// description  : EINT6 interrupt service routine
// input param  :
// retval       : none
// note         :
//******************************************************************************
void __interrupt DRV_EXT6_IRQ(void)
{
    EXT6_Int_Flag = 1;  // Set interrupt flag to 1
}

//******************************************************************************
// name         : DRV_EXT7_IRQ()
// description  : EINT7 interrupt service routine
// input param  :
// retval       : none
// note         :
//******************************************************************************
void __interrupt DRV_EXT7_IRQ(void)
{
    EXT7_Int_Flag = 1;  // Set interrupt flag to 1
}
