//******************************************************************************
// file         : DRV_SQ7705_EERPOM.c
// version      : V1.2 2023/09/04
// description  : EERPOM related functions
// note         : EERPOM related functions are gathered in this subroutine
//******************************************************************************
#include "DRV_SQ7705_EEPROM.h" 

//******************************************************************************
// name         : DRV_EEPROM_Init()
// description  : initialize eeprom clock(FCKDIV)
// input param  : 
// output param : 
// retval       : 
// note         : sysclk : System clock frequency in MHz
//******************************************************************************
void DRV_EEPROM_Init(uint8_t sysclk)
{
    FCKDIV = sysclk - 1;
}

//******************************************************************************
// name         : DRV_EEPROM_EraseSector()
// description  : sector erase of EEPROM
// input param  : - address : start address to erase
// output param : 
// retval       : - 0 : erase successful
//                - 1 : erase fail
// note         : erase 32 bytes
//******************************************************************************
uint8_t DRV_EEPROM_EraseSector(uint16_t address)
{
    uint8_t status;                              // status variable
    DRV_EEERR_CLEAR();
    
    EEADDR0 = (uint8_t)address & 0x00FF;         // set start address
    EEADDR1 = (uint8_t)(address >> 8) & 0x00FF;
    
    EECR = 0x18;                                 // sector erase command
    EEDATA0 = 0xA5;                              // A5, 5A 
    EEDATA0 = 0x5A;                              // A5, 5A 
    
    do{                                          // wait for sector erase completed
        status = EESR;
        status &= 0x80;
    }while(status);                              // when status is busy(EESR_BUSY=1)

    return(EESR_EEERR);                          // return error message
}

//******************************************************************************
// name         : DRV_EEPROM_EraseBlock()
// description  : block erase of EEPROM
// input param  : - address : start address to erase
// output param : 
// retval       : - 0 : erase successful
//                - 1 : erase fail
// note         : erase 512 bytes
//******************************************************************************
uint8_t DRV_EEPROM_EraseBlock(uint16_t address)
{
    uint8_t status;                              // status variable
    DRV_EEERR_CLEAR();
    
    EEADDR0 = (uint8_t)address & 0x00FF;         // set start address
    EEADDR1 = (uint8_t)(address >> 8) & 0x00FF;
    
    EECR = 0x1B;                                 // block erase command
    EEDATA0 = 0xA5;                              // A5, 5A 
    EEDATA0 = 0x5A;                              // A5, 5A 
    
    do{                                          // wait for block erase completed
        status = EESR;
        status &= 0x80;
    }while(status);                              // when status is busy(EESR_BUSY=1)

    return(EESR_EEERR);                          // return error message
}

//******************************************************************************
// name         : DRV_EEPROM_EraseAll()
// description  : device erase of EEPROM
// input param  : 
// output param : 
// retval       : - 0 : erase successful
//                - 1 : erase fail
// note         : 
//******************************************************************************
uint8_t DRV_EEPROM_EraseAll(void)
{
    uint8_t status;                              // status variable
    DRV_EEERR_CLEAR();
        
    EECR = 0x19;                                 // device erase command
    EEDATA0 = 0xA5;                              // A5, 5A 
    EEDATA0 = 0x5A;                              // A5, 5A 
    
    do{                                          // wait for device erase completed
        status = EESR;
        status &= 0x80;
    }while(status);                              // when status is busy(EESR_BUSY=1)

    return(EESR_EEERR);                          // return error message
}

//******************************************************************************
// name         : DRV_EEPROM_WriteByte()
// description  : program one byte data in the eeprom
// input param  : - address : program address
//                - data : program data
// output param : 
// retval       : - 0 : program successful
//                - 1 : program fail
// note         : 
//******************************************************************************
uint8_t DRV_EEPROM_WriteByte(uint16_t address, uint8_t data)
{
    uint8_t status;                              // status variable
    DRV_EEERR_CLEAR();
    
    EEDATA0 = data;                              // write data to eerpom data register 

    EEADDR0 = (uint8_t)address & 0x00FF;         // set address
    EEADDR1 = (uint8_t)(address >> 8) & 0x00FF;

    EECR = 0x04;                                 // program command  
    EEDATA0 = 0xA5;                              // A5, 5A 
    EEDATA0 = 0x5A;                              // A5, 5A 

    do{                                          // wait for programming completed
        status = EESR;
        status &= 0x80;
    }while(status);                              // when status is busy(EESR_BUSY=1)

    return(EESR_EEERR);                          // return error message
}

//******************************************************************************
// name         : DRV_EEPROM_ReadByte()
// description  : read one byte data in the eeprom
// input param  : - address : read address
// output param : - data : data in the eeprom
// retval       : 
// note         : 
//******************************************************************************
void DRV_EEPROM_ReadByte(uint16_t address, uint8_t *data)
{
    DRV_EEERR_CLEAR();
    EEADDR0 = (unsigned char)address & 0x00FF;         // set address
    EEADDR1 = (unsigned char)(address >> 8) & 0x00FF;
   
    EECR = 0x00;                                       // read command
    *data = EEDATA0; 
}

//******************************************************************************
// name         : DRV_EEERR_CLEAR()
// description  : clear error status
// input param  : 
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_EEERR_CLEAR()
{
    if(EESR_EEERR !=0){
       EESR_EEERR = 0x07;
    }
}