//******************************************************************************
// file         : DRV_SQ7705_DelayLoop.c
// version      : V1.2 2023/08/31
// description  : DelayLoop related functions
// note         : 1. DelayLoop related functions are collected in this subroutine
//                2. delay_ms() is only applicable when the system clock source is PLL 
//                   and the system main frequency is 24MHz, 12MHz and 6MHz
//                3. delay_100us() is only applicable when the system clock source is PLL
//                   and the system main frequency is 24MHz
//                4. there is an error because the while loop used to delay the loop,
//                   for precise delay application, please use timer for delay function
//******************************************************************************
#include "DRV_SQ7705_DelayLoop.h"

volatile  uint16_t  delay_base_ms;
volatile  uint16_t  delay_base_100us;

//******************************************************************************
// name         : DRV_Delay_Loop_Init()
// description  : initialize the delay loop
// input param  : 
// output param : 
// retval       : 
// note         : the number of times required to switch the minimum unit of the 
//                Delay Loop function based on the system frequency
//******************************************************************************
void DRV_Delay_Loop_Init()
{
    /* Tips:
       // This function example is only applicable to the system clock source for 
       // PLL and the system main frequency is 24MHz, 12MHz and 6MHz
       // otherwise you must add or adjust in DRV_Delay_Loop_Init() by yourself
    */
    
    if(CLKCR0_SCKSEL != PLL){
        return;
    }

    switch(CLKCR0_SCKPSC){
        case PLL_24M:                            // when system frequecy is PLL 24MHz
            delay_base_ms = f24m_1ms_t;
            delay_base_100us = f24m_100us_t;
        break;

        case PLL_12M:                            // when system frequecy is PLL 12MHz
            delay_base_ms = f12m_1ms_t;
        break;

        case PLL_6M:                             // when system frequecy is PLL 6MHz
            delay_base_ms = f06m_1ms_t;
        break;

        default:
            delay_base_ms = f24m_1ms_t;
            delay_base_100us = f24m_100us_t;
        break;
      }
}

//******************************************************************************
// name         : DRV_Delay_ms()
// description  : delay function in unit of ms
// input param  : x is a multiplier, x > 0
// output param : 
// retval       : 
// note         : it is only applicable when the system clock source is PLL and 
//                the system main frequency is 24MHz, 12MHz and 6MHz
//                for example,
//                delay_ms(1);           // delay 1 ms
//                delay_ms(30);          // delay 30 ms
//                delay_ms(500);         // delay 500 ms
//******************************************************************************
void DRV_Delay_ms(uint16_t x)
{
    volatile uint16_t  i = 0;

    if(!x) return;
    
    for(; x>0; x--){
        for(i=0; i< delay_base_ms; i++){
            __asm("NOP");
            CLR_WDT;                        // clear the watchdog timer
        }
        CLR_WDT;                            // clear the watchdog timer
    }
}

//******************************************************************************
// name         : DRV_Delay_100us()
// description  : delay function in unit of 100 us
// input param  : y is a multiplier, y > 0, y is suggested to be 1 to 10
// output param : 
// retval       : 
// note         : it is only applicable when the system clock source is PLL and the system main frequency is 24MHz
//                there is an error because the while loop used to delay the loop,
//                for precise delay application, please use timer for delay function
//                for example,
//                delay_100us(1);        // delay 100 us = 0.1 ms
//                delay_100us(10);       // delay 1000 us = 1 ms
//******************************************************************************
void DRV_Delay_100us(uint8_t y)
{
    volatile uint16_t  i = 0;

    if(!y) return;

    for(; y>0; y--){
        for(i=0; i< delay_base_100us; i++){
            __asm("NOP");
            CLR_WDT;                        // clear the watchdog timer
        }
        CLR_WDT;                            // clear the watchdog timer
    }
}
