//******************************************************************************
// file         : main.c
// version      : V1.3  2023/11/20
// description  : in this example, DVOB output pin is P5.6 and default DVOB output 
//                frequency is 5.86 KHz and LED1 (P2.0) light on
//                1. press button 1 : BTN1_in(P1.0), stop DVOB output and light off all LEDs
//                2. press button 2 : BTN2_in(P1.1), set next output frequency and start DVOB output
//                3. press button 3 : BTN3_in(P1.2), set last output frequency and start DVOB output
//
//                there are 4 kinds of DVOB output frequency, this example:
//                Freq_Out_Case = 1, DVOB output frequency is 5.86 KHz   LED1 (P2.0) light on
//                Freq_Out_Case = 2, DVOB output frequency is 11.72 KHz  LED2 (P2.1) light on
//                Freq_Out_Case = 3, DVOB output frequency is 23.44 KHz  LED3 (P2.2) light on
//                Freq_Out_Case = 4, DVOB output frequency is 46.875 KHz LED4 (P2.4) light on
//
// note         : [Button Configuration]
//                    button 1 :  BTN1_in(P1.0)
//                    button 2 :  BTN2_in(P1.1)
//                    button 3 :  BTN3_in(P1.2)
//                [LED Configuration]
//                    LED1  (P2.0)
//                    LED2  (P2.1)
//                    LED3  (P2.2)
//                    LED4  (P2.4)
//******************************************************************************
#ifndef IO_MEM
#define IO_MEM
#endif
#include "ADM_main.h"
#include "DRV_SQ7705_EINT.h"
#include "DRV_SQ7705_DVO.h"
#include "DRV_SQ7705_GPIO.h"
#include "DRV_SQ7705_SYSCLK.h"

typedef enum{
    STATE_WAIT_KEY=0,
    STATE_SET_FREQ,
    STATE_WAIT_KEY_RELEASE
}MAIN_STATE;

typedef enum{
    DVO_DIV_4096 = 1,
    DVO_DIV_2048,
    DVO_DIV_1024,
    DVO_DIV_512
}DVO_Freq_Mode;

/* --------------------------- global variables ----------------------------- */
static uint8_t Main_State;                   // button state
uint8_t Index;
uint8_t Freq_Out_Case;                       // frequency out case index

/* ------------------ function prototype declaration ------------------------ */
void Freq_Out_Case_Satus_LED();

/* ---------------------------- main program -------------------------------- */
void ADM_Main()
{
    DRV_Clock_Init(Clk_Source_SYS, 1);                  // PLL 24MHz Initial system clock.
    
    DRV_GPIO_Init(GPIO_P20, GPIO_OUTPUT, GPIO_PULL_UP); // set P2.0 as output pin
                                                        // enable P2.0 built-in pull-up resistor
    DRV_GPIO_Init(GPIO_P21, GPIO_OUTPUT, GPIO_PULL_UP); // set P2.1 to output pin
                                                        // enable P2.1 built-in pull-up resistor
    DRV_GPIO_Init(GPIO_P22, GPIO_OUTPUT, GPIO_PULL_UP); // set P2.2 as output pin
                                                        // enable P2.2 built-in pull-up resistor
    DRV_GPIO_Init(GPIO_P23, GPIO_OUTPUT, GPIO_PULL_UP); // set P2.3 as output pin
                                                        // enable P2.3 built-in pull-up resistor
    DRV_GPIO_WritePin(GPIO_P20, GPIO_PIN_HIGH);         // set LED1(P2.0) to light off
    DRV_GPIO_WritePin(GPIO_P21, GPIO_PIN_HIGH);         // set LED2(P2.1) to light off
    DRV_GPIO_WritePin(GPIO_P22, GPIO_PIN_HIGH);         // set LED3(P2.2) to light off
    DRV_GPIO_WritePin(GPIO_P23, GPIO_PIN_HIGH);         // set LED4(P2.3) to light off
    
    /* ---------------------- EINT initialization ------------------------ */
    DRV_GPIO_Init(GPIO_P10, GPIO_INPUT, GPIO_PULL_UP);
    DRV_GPIO_Init(GPIO_P11, GPIO_INPUT, GPIO_PULL_UP);
    DRV_GPIO_Init(GPIO_P12, GPIO_INPUT, GPIO_PULL_UP);
    /* EINT0 P10 an interrupt request is generated at the falling edge for button 1*/
    DRV_EINT_Init(EINT0_P10, INT_VL_HIGH,INTES_FALLING,FSYSCLK_DIV_1);
    DRV_EINT_Enable(0);
    /* EINT1 P11 an interrupt request is generated at the falling edge for button 2*/
    DRV_EINT_Init(EINT1_P11, INT_VL_HIGH,INTES_FALLING,FSYSCLK_DIV_1);
    DRV_EINT_Enable(1);
    /* EINT2 P12 an interrupt request is generated at the falling edge for button 3*/
    DRV_EINT_Init(EINT2_P12, INT_VL_HIGH,INTES_FALLING,FSYSCLK_DIV_1);
    DRV_EINT_Enable(2);

    /* ---------------------- DVOB initialization amd start ----------------- */
    // if fsysclk=24MHz, set the DVOB output frequency to fsysclk/4096, which is about 5.86 KHz
    DRV_DVO_Init(OUTPUT_CLOCK_FSYSCLK_DIV_2_POW_12,DVO_OUTPUT_P56);
    DRV_DVO_Start();

    //----------------------------------------------------------
    Main_State = STATE_WAIT_KEY;                      // default state to wait for key press
    //----------------------------------------------------------
    while(1){
        switch(Main_State){
            case STATE_WAIT_KEY:             // wait for key press
                if(EXT0_Int_Flag){          // if button 1 is pressed, light off all LEDs
                    EXT0_Int_Flag = 0;
                    DRV_DVO_Stop();
                    DRV_GPIO_WritePin(GPIO_P20, GPIO_PIN_HIGH);  // light off LED 1
                    DRV_GPIO_WritePin(GPIO_P21, GPIO_PIN_HIGH);  // light off LED 2
                    DRV_GPIO_WritePin(GPIO_P22, GPIO_PIN_HIGH);  // light off LED 3
                    DRV_GPIO_WritePin(GPIO_P23, GPIO_PIN_HIGH);  // light off LED 4
                    Main_State = STATE_WAIT_KEY;
                } else if(EXT1_Int_Flag){   // if button 2 is pressed
                    EXT1_Int_Flag = 0;
                    DRV_DVO_Stop();
                    Index++;
                    if(Index > DVO_DIV_512){
                        Index = DVO_DIV_4096;
                    }
                    Freq_Out_Case = Index;
                    Main_State = STATE_SET_FREQ;
                    DRV_DVO_Start();
                } else if(EXT2_Int_Flag){   // if button 3 is pressed
                    EXT2_Int_Flag = 0;
                    DRV_DVO_Stop();
                    if (Index!=0){
                        Index--;
                    } else {
                        Index =1;
                    }
                    if(Index < DVO_DIV_4096){
                        Index = DVO_DIV_512;
                    }
                    Freq_Out_Case = Index;
                    Main_State = STATE_SET_FREQ;
                    DRV_DVO_Start();
                }
                break;

            case STATE_SET_FREQ:             // select output frequency
                switch(Freq_Out_Case){
                    // set DVOB output frequency to fsysclk/4096, it's about 5.86 KHz
                    case DVO_DIV_4096:
                        DRV_DVO_Init(OUTPUT_CLOCK_FSYSCLK_DIV_2_POW_12,DVO_OUTPUT_P56);
                        break;
                    // set DVOB output frequency to fsysclk/2048, it's about 11.72 KHz
                    case DVO_DIV_2048:
                        DRV_DVO_Init(OUTPUT_CLOCK_FSYSCLK_DIV_2_POW_11,DVO_OUTPUT_P56);
                        break;
                    // set DVOB output frequency to fsysclk/1024, it's about 23.44 KHz
                    case DVO_DIV_1024:
                        DRV_DVO_Init(OUTPUT_CLOCK_FSYSCLK_DIV_2_POW_10,DVO_OUTPUT_P56);
                        break;
                    // set DVOB output frequency to fsysclk/512, it's about 46.875 KHz
                    case DVO_DIV_512:
                        DRV_DVO_Init(OUTPUT_CLOCK_FSYSCLK_DIV_2_POW_9,DVO_OUTPUT_P56);
                        break;
                    // set DVOB output frequency to fsysclk/4096, it's about 5.86 KHz
                    default:
                        DRV_DVO_Init(OUTPUT_CLOCK_FSYSCLK_DIV_2_POW_12,DVO_OUTPUT_P56);
                        break;
                }
                // according to the DVOB output frequency case, light on the corresponding case LED light number
                Freq_Out_Case_Satus_LED();
                Main_State = STATE_WAIT_KEY_RELEASE;
                break;

            case STATE_WAIT_KEY_RELEASE:              // button released
                if(DRV_GPIO_ReadPin(GPIO_P10) == 1 
                || DRV_GPIO_ReadPin(GPIO_P11) == 1 
                || DRV_GPIO_ReadPin(GPIO_P12) == 1){  // wait for button released
                    Main_State = STATE_WAIT_KEY;      // return to wait for key press    
                }
                break;

            default:
                break;
        }
    }
}

//******************************************************************************
// name         : Freq_Out_Case_Satus_LED()
// description  : according to the DVOB output frequency case, light on the corresponding case LED light number
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void Freq_Out_Case_Satus_LED()
{
    if(Freq_Out_Case==1){
        DRV_GPIO_WritePin(GPIO_P20, GPIO_PIN_LOW);            // light on LED1
        DRV_GPIO_WritePin(GPIO_P21, GPIO_PIN_HIGH);           // light off LED2
        DRV_GPIO_WritePin(GPIO_P22, GPIO_PIN_HIGH);           // light off LED3
        DRV_GPIO_WritePin(GPIO_P23, GPIO_PIN_HIGH);           // light off LED4
    } else if(Freq_Out_Case==2){
        DRV_GPIO_WritePin(GPIO_P20, GPIO_PIN_HIGH);           // light off LED1
        DRV_GPIO_WritePin(GPIO_P21, GPIO_PIN_LOW);            // light on LED2
        DRV_GPIO_WritePin(GPIO_P22, GPIO_PIN_HIGH);           // light off LED3
        DRV_GPIO_WritePin(GPIO_P23, GPIO_PIN_HIGH);           // light off LED4
    } else if(Freq_Out_Case==3){
        DRV_GPIO_WritePin(GPIO_P20, GPIO_PIN_HIGH);           // light off LED1
        DRV_GPIO_WritePin(GPIO_P21, GPIO_PIN_HIGH);           // light off LED2
        DRV_GPIO_WritePin(GPIO_P22, GPIO_PIN_LOW);            // light on LED3
        DRV_GPIO_WritePin(GPIO_P23, GPIO_PIN_HIGH);           // light off LED4
    } else if(Freq_Out_Case==4){
        DRV_GPIO_WritePin(GPIO_P20, GPIO_PIN_HIGH);           // light off LED1
        DRV_GPIO_WritePin(GPIO_P21, GPIO_PIN_HIGH);           // light off LED2
        DRV_GPIO_WritePin(GPIO_P22, GPIO_PIN_HIGH);           // light off LED3
        DRV_GPIO_WritePin(GPIO_P23, GPIO_PIN_LOW);            // light on LED4
    }
}

/* ----------------------- interrupt service routine ------------------------ */
void __interrupt OnlyReti(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n OnlyRetn(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n WDT_IRQ(void)
{
    __ASM("NOP");                              // no operation
}

void __interrupt_n SWI_IRQ(void)               // software reset interrupt service routine
{
    SWRST = 0x5a;
    SWRST = 0xa5;
    SWRST = 0xc3;
    SWRST = 0x3c;
}

void __interrupt IntADM_Req(void)
{

}

/* ---------------------- end of interrupt service routine ------------------ */


/* ----------------------- SQ7705 interrupt vector -------------------------- */
#pragma section const ADM_INT_VECTOR
void * const IntTbl1[] = {
    ADM_STARTUP,             /* 0x0000 : RESET               */    //non-maskable
    SWI_IRQ,                 /* 0x0002 : SWI_IRQ/INTUNDEF    */    //non-maskable
    (void *)0xffff,          /* 0x0004 : Reserved            */    
    WDT_IRQ,                 /* 0x0006 : WDT_IRQ             */    //non-maskable interrupt source:watchdog timer
    OnlyReti,                /* 0x0008 : TAMPER_IRQ          */    //interrupt source:tamper detection
    OnlyReti,                /* 0x000A : CFD_IRQ             */    //interrupt source:clock faildetection
    OnlyReti,                /* 0x000C : HVD_IRQ             */
    IntADM_Req,              /* 0x000E : U2A_IRQ/IntADM_Req  */    //interrupt source:user mode to admin mode
    OnlyReti,                /* 0x0010 : RTC_IRQ             */    //interrupt source:real-time clock timer
    OnlyReti,                /* 0x0012 : TMRA00_IRQ          */    //interrupt source:TMRA00 8-bit timer
    OnlyReti,                /* 0x0014 : TMRA01_IRQ          */    //interrupt source:TMRA01 8-bit timer
    OnlyReti,                /* 0x0016 : TMRA10_IRQ          */    //interrupt source:TMRA10 8-bit timer
    OnlyReti,                /* 0x0018 : TMRA11_IRQ          */    //interrupt source:TMRA11 8-bit timer
    OnlyReti,                /* 0x001A : TMRA20_IRQ          */
    OnlyReti,                /* 0x001C : TMRA21_IRQ          */
    OnlyReti,                /* 0x001E : TBT_IRQ             */
    OnlyReti,                /* 0x0020 : RX0_IRQ             */    //interrupt source:UART0 RX0
    OnlyReti,                /* 0x0022 : TX0_IRQ             */    //interrupt source:UART0 TX0
    OnlyReti,                /* 0x0024 : I2C0_IRQ            */    //interrupt source:I2C0
    OnlyReti,                /* 0x0026 : SPI0_IRQ            */    //interrupt source:SPI0
    DRV_EXT0_IRQ,            /* 0x0028 : EXT0_IRQ            */    //external interrupt0
    DRV_EXT1_IRQ,            /* 0x002A : EXT1_IRQ            */    //external interrupt1
    DRV_EXT2_IRQ,            /* 0x002C : EXT2_IRQ            */    //external interrupt2
    OnlyReti,                /* 0x002E : EXT3_IRQ            */    //external interrupt3
    OnlyReti,                /* 0x0030 : ADC_IRQ             */    //interrupt source:ADC
    OnlyReti,                /* 0x0032 : LVD0_IRQ            */
    OnlyReti,                /* 0x0034 : LVD1_IRQ            */
    OnlyReti,                /* 0x0036 : Reserved            */
    OnlyReti,                /* 0x0038 : Reserved            */
    OnlyReti,                /* 0x003A : FMC_IRQ             */    //interrupt source:flash memory controller
    OnlyReti,                /* 0x003C : DMACH0_IRQ          */
    OnlyReti,                /* 0x003E : DMACH1_IRQ          */
    OnlyReti,                /* 0x0040 : DMACH2_IRQ          */
    OnlyReti,                /* 0x0042 : DMACH3_IRQ          */
    OnlyReti,                /* 0x0044 : Reserved            */
    OnlyReti,                /* 0x0046 : Reserved            */
    OnlyReti,                /* 0x0048 : Reserved            */
    OnlyReti,                /* 0x004A : Reserved            */
    OnlyReti,                /* 0x004C : Reserved            */
    OnlyReti,                /* 0x004E : DIC_IRQ             */    //interrupt source:DIC
    OnlyReti,                /* 0x0050 : LEUART_RX_IRQ       */
    OnlyReti,                /* 0x0052 : LEUART_TX_IRQ       */
    OnlyReti,                /* 0x0054 : TRNG_IRQ            */
    OnlyReti,                /* 0x0056 : EXT4_IRQ            */    //external interrupt4
    OnlyReti,                /* 0x0058 : EXT5_IRQ            */    //external interrupt5
    OnlyReti,                /* 0x005A : EXT6_IRQ            */    //external interrupt6
    OnlyReti,                /* 0x005C : EXT7_IRQ            */    //external interrupt7
    OnlyReti,                /* 0x005E : EEMC_IRQ            */    //interrupt source:EEPROM controller
    OnlyReti,                /* 0x0060 : TCA0_IRQ            */    //interrupt source:TCA0 16-bit timer
    OnlyReti,                /* 0x0062 : TCA1_IRQ            */    //interrupt source:TCA1 16-bit timer
    OnlyReti,                /* 0x0064 : STM0_IRQ            */
    OnlyReti,                /* 0x0066 : STM1_IRQ            */
    OnlyReti,                /* 0x0068 : AES_IRQ             */
    OnlyReti,                /* 0x006A : SHA_IRQ             */
    OnlyReti,                /* 0x006C : RX1_IRQ             */    //interrupt source:UART1 RX1
    OnlyReti,                /* 0x006E : TX1_IRQ             */    //interrupt source:UART1 TX1
    OnlyReti,                /* 0x0070 : I2C1_IRQ            */    //interrupt source:I2C1
    OnlyReti,                /* 0x0072 : SPI1_IRQ            */    //interrupt source:SPI1
    OnlyReti,                /* 0x0074 : RX2_IRQ             */    //interrupt source:UART2 RX2
    OnlyReti,                /* 0x0076 : TX2_IRQ             */    //interrupt source:UART2 TX2
    OnlyReti                 /* 0x0078 : TCA2_IRQ            */    //interrupt source:TCA2 16-bit timer
};
#pragma section const
/* -------------------- end of SQ7705 interrupt vector ---------------------- */

