//******************************************************************************
// file         : DRV_SQ7705_DMA.c
// version      : V1.2 (2023/08/31)
// description  : DMA related functions
// note         : DMA related functions are gathered in this subroutine
//******************************************************************************
#define __DRV_SQ7705_DMA_C__

#include "DRV_SQ7705_DMA.h"

unsigned char IntDMA0_CH_Flag;
unsigned char IntDMA1_CH_Flag;
unsigned char IntDMA2_CH_Flag;
unsigned char IntDMA3_CH_Flag;

//******************************************************************************
// name         : DRV_DMA_Init()
// description  : DMA initialization
// input param  : dmaChannel:    DMA_CH0 ~ DMA_CH3
//                sourceSelect:  DMA_MEMORY ~ DMA_AESDST
//                sourceIncrement: source address increment mode
//                destinationSelect: source/destination Select
//                destinationIncrement: destination address increment mode
//                sourceAddr: DMA source address
//                destinationAddr : DMA destination address
// retval       : none
// note         : DMA related parameters
// example:
// DRV_DMA_Init(DMA_CH0, DMA_MEMORY, TRUE, DMA_MEMORY, TRUE, 0x1230, 0x1340);
//******************************************************************************
void DRV_DMA_Init(unsigned char     dmaChannel,
                  unsigned char     sourceSelect,
                  unsigned char     sourceIncrement,
                  unsigned char     destinationSelect,
                  unsigned char     destinationIncrement,
                  unsigned short    sourceAddr,
                  unsigned short    destinationAddr)
{
    if(sourceIncrement == INCREMENT){       // if resoure address increment ?
        sourceSelect |= 0x20;
    }

    if(destinationIncrement == INCREMENT){  // if destination address increment ?
        destinationSelect |= 0x20;
    }

    switch(dmaChannel) {

        case DMA_CH0:
            PCKEN5_DMA = ENABLE;        // Enable DMA block clock.
            CH0SCFG = sourceSelect;
            CH0SA = sourceAddr;
            CH0SA2 = DMA_SFR_RAM;       // Memory selection: 0 (SFR, RAM).
            CH0DCFG = destinationSelect;
            CH0DA = destinationAddr;
            CH0DA2 = DMA_SFR_RAM;       // Memory selection: 0 (SFR, RAM).
            break;

        case DMA_CH1:
            PCKEN5_DMA = ENABLE;        // Enable DMA block clock.
            CH1SCFG_SRCSEL = sourceSelect;
            CH1SA = sourceAddr;
            CH1SA2 = DMA_SFR_RAM;       // Memory selection: 0 (SFR, RAM).
            CH1DCFG_DSTSEL = destinationSelect;
            CH1DA = destinationAddr;
            CH1DA2 = DMA_SFR_RAM;       // Memory selection: 0 (SFR, RAM).
            break;
            
        case DMA_CH2:
            PCKEN5_DMA = ENABLE;        // Enable DMA block clock.
            CH2SCFG_SRCSEL = sourceSelect;           
            CH2SA = sourceAddr;
            CH2SA2 = DMA_SFR_RAM;       // Memory selection: 0 (SFR, RAM).
            CH2DCFG_DSTSEL = destinationSelect;
            CH2DA = destinationAddr;
            CH2DA2 = DMA_SFR_RAM;       // Memory selection: 0 (SFR, RAM).
            break;  

        case DMA_CH3:
            PCKEN5_DMA = ENABLE;        // Enable DMA block clock.
            CH3SCFG_SRCSEL = sourceSelect;
            CH3SA = sourceAddr;
            CH3SA2 = DMA_SFR_RAM;       // Memory selection: 0 (SFR, RAM).
            CH3DCFG_DSTSEL = destinationSelect;
            CH3DA = destinationAddr;
            CH3DA2 = DMA_SFR_RAM;       // Memory selection: 0 (SFR, RAM).
            break;  
    }
}

//******************************************************************************
// name         : DRV_DMA_Size()
// description  : DMA channel size
// input param  : dmaChannel  : DMA_CHANNEL_0 ~ DMA_CHANNEL_3
//                sourceSize  :   DMA_BYTE
//                                DMA_HALF_WORD 
//                                DMA_WORD
//                destinationSize : DMA_BYTE
//                                  DMA_HALF_WORD
//                                  DMA_WORD
// retval       : none
// note         : 
//******************************************************************************
void DRV_DMA_Size(unsigned char dmaChannel,
                  unsigned char sourceSize,
                  unsigned char destinationSize)
{
    switch(dmaChannel) {
        case DMA_CH0:
            SDCSSZ_CH0SSZ = sourceSize;      // DMA_CH0 source size
            SDCDSZ_CH0DSZ = destinationSize; // DMA_CH0 destination size
            break;

        case DMA_CH1:
            SDCSSZ_CH1SSZ = sourceSize;      // DMA_CH1 source size
            SDCDSZ_CH1DSZ = destinationSize; // DMA_CH1 destination size
            break;

        case DMA_CH2:
            SDCSSZ_CH2SSZ = sourceSize;      // DMA_CH2 source size
            SDCDSZ_CH2DSZ = destinationSize; // DMA_CH2 destination size
            break;

        case DMA_CH3:
            SDCSSZ_CH3SSZ = sourceSize;      // DMA_CH3 source size
            SDCDSZ_CH3DSZ = destinationSize; // DMA_CH3 destination size
            break;
    } 
}

//******************************************************************************
// name         : DRV_DMA_Int()
// description  : enable/disable DMA interrupt 
// input param  : dmaChannel      : DMA_CH0 ~ DMA_CH3
//                enable          : ENABLE  = 1  enable interrupt
//                                  DISABLE = 0  disable interrupt
// retval       : none
// note         : 
//******************************************************************************
void DRV_DMA_Int(unsigned char dmaChannel , unsigned char enable)
{ 
    __ASM("DI");

    switch (dmaChannel) {
        case DMA_CH0:
            SDCINT_CH0IE = enable;
            IFR_DMACH0 = 0;
            IER_DMACH0 = enable;
            __ASM("NOP");
            break;

        case DMA_CH1:
            SDCINT_CH1IE = enable;
            IFR_DMACH1 = 0;
            IER_DMACH1 = enable;
            __ASM("NOP");
            break;

        case DMA_CH2:
            SDCINT_CH2IE = enable;
            IFR_DMACH2 = 0;
            IER_DMACH2 = enable;
            __ASM("NOP");
            break;

        case DMA_CH3:
            SDCINT_CH3IE = enable;
            IFR_DMACH3= 0;
            IER_DMACH3 = enable;
            __ASM("NOP");
            break;
    }
    
    __ASM("EI");
}

//******************************************************************************
// name         : DRV_DMA_ReadActive()
// description  : DMA 
// input param  : dmaChannel : DMA_CH0 ~ DMA_CH3
// retval       : 0: inactive 1: active 0xFF: parameter error
// note         : 
//******************************************************************************
unsigned char DRV_DMA_ReadActive(unsigned char dmaChannel)
{

    switch (dmaChannel) {
        case DMA_CH0:
            return SDCSR0_CH0ACT;

        case DMA_CH1:
            return SDCSR0_CH1ACT;

        case DMA_CH2:
            return SDCSR0_CH2ACT;

        case DMA_CH3:
            return SDCSR0_CH3ACT;
    }

    return 0xFF;            // parameter error
}

//******************************************************************************
// name         : DRV_DMA_ReadState()
// description  : DMA 
// input param  : dmaChannel  : DMA_CH0 ~ DMA_CH3
// retval       :   0x00: DMA_NoError,      no error
//                  0x01: ---,              reserved
//                  0x02: DMA_SW_Error,     software active error
//                  0X03: DMA_BUS_Error,    bus error
//                  0xFF: ---,              reserved
// note         : 
//******************************************************************************
unsigned char DRV_DMA_ReadState(unsigned char dmaChannel)
{
    switch (dmaChannel) {
        case DMA_CH0:
            return SDCSR1_CH0ST;

        case DMA_CH1:
            return SDCSR1_CH1ST;

        case DMA_CH2:
            return SDCSR1_CH2ST;

        case DMA_CH3:
            return SDCSR1_CH3ST;
    }

    return 0xFF;            // parameter error
}

//******************************************************************************
// name         : DRV_DMA_Start()
// description  : start DMA
// input param  : sourceSelect      : source DMA channel selection
//                destinationSelect : destination DMA channel selection
// retval       : none
// note         : 
//******************************************************************************
void DRV_DMA_Start(unsigned char dmaChannel, unsigned short dmacount)
{
    switch (dmaChannel) {
        case DMA_CH0:
            CH0CNT = dmacount;
            __ASM("NOP");
        
            SDCR_CH0EN = 1;

            //-----------------------------------------------------------------
            // both source and destination are MEMORY
            //-----------------------------------------------------------------
            if((CH0SCFG_SRCSEL == DMA_MEMORY) && (CH0DCFG_DSTSEL == DMA_MEMORY)) {
                SDCSR0_CH0ACT = 1;
                __ASM("NOP");
            }

            break;

        case DMA_CH1:
            CH1CNT = dmacount;
            __ASM("NOP");
        
            SDCR_CH1EN = 1;

            //-----------------------------------------------------------------
            // both source and destination are MEMORY
            //-----------------------------------------------------------------
            if((CH1SCFG_SRCSEL == DMA_MEMORY) && (CH1DCFG_DSTSEL == DMA_MEMORY)) {
                SDCSR0_CH1ACT = 1;
                __ASM("NOP");
            }
        
            break;

        case DMA_CH2:
            CH2CNT = dmacount;
            __ASM("NOP");
        
            SDCR_CH2EN = 1;

            //-----------------------------------------------------------------
            // both source and destination are MEMORY
            //-----------------------------------------------------------------
            if((CH2SCFG_SRCSEL == DMA_MEMORY) && (CH2DCFG_DSTSEL == DMA_MEMORY)) {
                SDCSR0_CH2ACT = 1;
                __ASM("NOP");
            }
        
            break;

        case DMA_CH3:
            CH3CNT = dmacount;
            __ASM("NOP");
        
            SDCR_CH3EN = 1;

            //-----------------------------------------------------------------
            // both source and destination are MEMORY
            //-----------------------------------------------------------------
            if((CH3SCFG_SRCSEL == DMA_MEMORY) && (CH3DCFG_DSTSEL == DMA_MEMORY)) {
                SDCSR0_CH3ACT = 1;
                __ASM("NOP");
            }
        
            break;
    }
}

//******************************************************************************
// name         : DRV_DMA_Stop()
// description  : stop DMA
// input param  : dmaChannel  : DMA_CHANNEL_0 ~ DMA_CHANNEL_3
// retval       : none
// note         : 
//******************************************************************************
void DRV_DMA_Stop(unsigned char dmaChannel)
{
    switch (dmaChannel) {
        case DMA_CH0:
            SDCR_CH0EN = 0;
            break;

        case DMA_CH1:
            SDCR_CH1EN = 0;
            break;

        case DMA_CH2:
            SDCR_CH2EN = 0;
            break;

        case DMA_CH3:
            SDCR_CH3EN = 0;
            break;
    }
}

//******************************************************************************
// name         : DRV_DMA_WaitComplet()
// description  : 
// input param  : dmaChannel : DMA_CH0 ~ DMA_CH3
// retval       : none
// note         : 
//******************************************************************************
void DRV_DMA_WaitComplet(unsigned char dmaChannel)
{
    switch (dmaChannel) {
        case DMA_CH0:
            while(SDCSR0_CH0ACT == 1){
                __ASM("NOP");
            }
            break;

        case DMA_CH1:
            while(SDCSR0_CH1ACT == 1){
                __ASM("NOP");
            }
            break;

        case DMA_CH2:
            while(SDCSR0_CH2ACT == 1){
                __ASM("NOP");
            }
            break;

        case DMA_CH3:
            while(SDCSR0_CH3ACT == 1){
                __ASM("NOP");
            }
            break;
    }
}

//******************************************************************************
// name         : __interrupt IntDMA0_CH()
// description  : IntDMA0_CH interrupt service routine.
// note         :
//******************************************************************************
void __interrupt IntDMA0_CH(void)
{
    SDCR_CH0EN = 0;                     // Disable DMA_CH0.
    SDCINT = ((SDCINT & 0x0E) | 0xE0);  // Clear DMA0 interrupt flag & Disable DMA0 interrupt.
    IER_DMACH0 = DISABLE;              // Disable DMA_CH0 interrupt.
    IFR_DMACH0 = 0;                    // Clear DMA_CH0 interrupt flag.

    IntDMA0_CH_Flag = 1;
}

//******************************************************************************
// name         : __interrupt IntDMA1_CH()
// description  : IntDMA1_CH interrupt service routine.
// note         :
//******************************************************************************
void __interrupt IntDMA1_CH(void)
{
    SDCR_CH1EN = DISABLE;               // Disable DMA_CH1.
    SDCINT = ((SDCINT & 0x0D) | 0xD0);  // Clear DMA1 interrupt flag & Disable DMA1 interrupt.
    IER_DMACH1 = DISABLE;              // Disable DMA_CH1 interrupt.
    IFR_DMACH1 = 0;                    // Clear DMA_CH1 interrupt flag.

    IntDMA1_CH_Flag = 1;
}

//******************************************************************************
// name         : __interrupt IntDMA2_CH()
// description  : IntDMA2_CH interrupt service routine.
// note         :
//******************************************************************************
void __interrupt IntDMA2_CH(void)
{
    SDCR_CH2EN = DISABLE;               // Disable DMA_CH2.
    SDCINT = ((SDCINT & 0x0B) | 0xB0);  // Clear DMA2 interrupt flag & Disable DMA2 interrupt.
    IER_DMACH2 = DISABLE;              // Disable DMA_CH2 interrupt.
    IFR_DMACH2 = 0;                    // Clear DMA_CH2 interrupt flag.

    IntDMA2_CH_Flag = 1;
}

//******************************************************************************
// name         : __interrupt IntDMA3_CH()
// description  : IntDMA3_CH interrupt service routine.
// note         :
//******************************************************************************
void __interrupt IntDMA3_CH(void)
{
    SDCR_CH3EN = DISABLE;               // Disable DMA_CH3.
    SDCINT = ((SDCINT & 0x07) | 0x70);  // Clear DMA3 interrupt flag & Disable DMA3 interrupt.
    IER_DMACH3 = DISABLE;              // Disable DMA_CH3 interrupt.
    IFR_DMACH3= 0;                    // Clear DMA_CH3 interrupt flag.

    IntDMA3_CH_Flag = 1;
}

