//******************************************************************************
// file         : DRV_SQ7705_CTRDRBG.c
// version      : V1.2 2023/10/11
// description  : CtrDRBG related functions
// note         : CtrDRBG related functions are gathered in this subroutine
//                Must include DRV_SQ7705_AES and DRV_SQ7705_TRNG
//******************************************************************************
#include "DRV_SQ7705_CTRDRBG.h"

const uint8_t constantDF[32] = 
{
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B,
    0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
    0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F
};

uint8_t ctr_buf_V[32];
uint8_t ctr_buf_CKey[32];
uint8_t ctr_seedlen = 0, ctr_type = 0 , ctr_df = 0;
uint32_t ctr_reseedCounter = 0;

void XorData(uint8_t* res, uint8_t* data1, uint8_t* data2, uint16_t dataLength)
{
    uint16_t i;
    
    for(i=0; i<dataLength; i++)
    {
        res[i] = data1[i] ^ data2[i];
    }
}

uint8_t DRBG_AddByte(uint8_t *b1, const uint8_t b2)
{
    union{uint16_t word; uint8_t byte[2];} res; 

    res.word = *b1 + b2;
    *b1 = res.byte[0];
    return res.byte[1];
}

void DRBG_IncrData(uint8_t *data, uint16_t dataLen)
{
    int16_t i;
    uint8_t carry;
    carry = DRBG_AddByte(data + dataLen - 1, (uint8_t)1);

    for(i = dataLen - 2; i >= 0; i--) {
        if(carry == 0) {
            break;
        } else {
            carry = DRBG_AddByte(data + i, carry);
        }
    }
}

uint8_t DRBG_AddByteWithCarry(uint8_t *b1, uint8_t b2, uint8_t carry)
{
    uint16_t res = *b1 + b2 + carry;
    *b1 = (uint8_t) res;
    return (((uint8_t)(res >> 8))!=0);
}

void DRBG_AddData(uint8_t *data, uint16_t dataLen, uint8_t *data1, uint16_t data1Len)
{
    int16_t i;
    uint8_t carry;
    uint16_t diff = dataLen - data1Len;

    carry = DRBG_AddByte(data + dataLen - 1, data1[data1Len - 1]);

    for(i = data1Len - 2; i >= 0; i--) {
        carry = DRBG_AddByteWithCarry(data + i + diff, data1[i], carry);
    }
    if((carry == 1) && (diff > 0)) {
        DRBG_IncrData(data, diff);
    }
}

void DRBG_AddInt(uint8_t* data, uint16_t dataLen, uint32_t n)
{
    int16_t i;
    uint8_t carry;
    uint16_t diff = dataLen - 8;

    carry = DRBG_AddByte(data+ dataLen - 1, (uint8_t)n);

    for(i = 6; i >= 0; i--) {
        carry = DRBG_AddByteWithCarry(data + i + diff, (uint8_t)(n >> (8 * (7-i))), carry);
    }
    if((carry == 1) && (diff > 0)) {
        DRBG_IncrData(data + diff -1, diff); 
    }
}

//******************************************************************************
// name         : BCC()
// description  : 
// input param  : - AESkey : AES key
//                - data : Source data to compress
//                - dataLen : Length of data , must be a multiple of CTRDRBG_OUT_LEN
//                - type : DRBG algorithm to use (AES128 or AES256)
// output param : - res : Destination array of bytes
// retval       : 
// note         : 
//******************************************************************************
void BCC(uint8_t* res, uint8_t* AESkey, uint8_t* data, uint16_t dataLen, uint8_t type)
{
    uint8_t chainingValue[CTRDRBG_OUT_LEN];
    uint8_t inputBlock[CTRDRBG_OUT_LEN];
    uint8_t blocki[CTRDRBG_OUT_LEN];
    uint16_t i, n = 0; 
    uint8_t KeySize;
    
    memset(chainingValue, 0, CTRDRBG_OUT_LEN);
    memset(inputBlock, 0, CTRDRBG_OUT_LEN);
    memset(blocki, 0, CTRDRBG_OUT_LEN);

    /* chainingValue = 0^CTRDRBG_OUT_LEN */ 
    memset(chainingValue, 0, CTRDRBG_OUT_LEN);
    
    /* n = len(data) / CTRDRBG_OUT_LEN (len(data) is a multiple of CTRDRBG_OUT_LEN) */
    n = dataLen / CTRDRBG_OUT_LEN;

    if(type == CTRDRBG_TYPE_AES128) {
        DRV_AES_Init(AES_KEYSZ_128);
        DRV_AES_Load_Program_Key(AES_IN, AESkey, AES_KEYSZ_128, 0);
    } else {
        DRV_AES_Init(AES_KEYSZ_256);
        DRV_AES_Load_Program_Key(AES_IN, AESkey, AES_KEYSZ_256, 0);
    }

    for(i = 0; i < n; i++) {
        /* blocki = ith first block of CTRDRBG_OUT_LEN bytes of data */
        memcpy(blocki, data + (i*CTRDRBG_OUT_LEN), CTRDRBG_OUT_LEN);

        /* inputBlock = chainingValue ^ blocki */
        XorData(inputBlock, chainingValue, blocki, CTRDRBG_OUT_LEN);        
        DRV_AES_ECB_Crypt(AES_Action_Encryption, inputBlock, chainingValue, CTRDRBG_OUT_LEN);
    }
    /* output_block = chainingValue */
    memcpy(res, chainingValue, CTRDRBG_OUT_LEN);
}

//******************************************************************************
// name         : Block_Cipher_df()
// description  : 
// input param  : - inputString : Entropy byte to compress
//                - inputStrLen : Length of inputString(in bytes)
//                - requestedLen : Size (in bytes) of requestedBytes, must be <= 64
//                                 AES128: 32 bytes
//                                 AES256: 48 bytes
//                - type : DRBG algorithm to use (AES128 or AES256)
// output param : - requestedBytes : Destination array of bytes
// retval       : 
// note         : 
//******************************************************************************
void Block_Cipher_df(uint8_t* inputString, uint32_t inputStrLen, uint8_t* requestedBytes, uint8_t requestedLen, uint8_t type)
{
    union{uint32_t dword; uint8_t byte[4];}i;
    union{uint32_t dword; uint8_t byte[4];}L;
    union{uint32_t dword; uint8_t byte[4];}N;
    uint16_t lenS, diff=0;
    uint8_t data[192];//8+64+64+32+8+16;
    uint8_t* S = data + 16;
    uint8_t K[32];
    uint8_t iv[CTRDRBG_OUT_LEN];
    uint8_t temp[32+CTRDRBG_OUT_LEN];
    uint8_t X[CTRDRBG_OUT_LEN];
    uint8_t KeySize, keylen, lentemp=0;

    if(requestedLen > 64) {
        // requestedLen > 64 , return error 
        return;
    }

    if(type == CTRDRBG_TYPE_AES128)
        keylen = 16;
    else
        keylen = 32;

    /* L = len(inputString) (as we work with bytes it's not necessary to divide by 8 */
    L.dword = inputStrLen;

    /* N = requestedLen */ 
    N.dword = requestedLen;

    /* S = L || N || inputString || 0x80 */
    lenS = 8 + inputStrLen + 1;

    /* while(len(S) mod CTRDRBG_OUT_LEN != 0, S = S || 0x00)*/
    while((lenS % CTRDRBG_OUT_LEN) != 0) {
        lenS += 1;
        diff += 1;
    }  

    S[0] = L.byte[3];
    S[1] = L.byte[2];
    S[2] = L.byte[1];
    S[3] = L.byte[0];
    S[4] = N.byte[3];
    S[5] = N.byte[2];
    S[6] = N.byte[1];
    S[7] = N.byte[0];
    
    memcpy(S + 8, inputString, inputStrLen); 
    S[8 + inputStrLen] = 0x80;
    memset(S + 8 + inputStrLen + 1, 0, diff);

    /* K = leftmost keylen bytes of 0x00010203...1D1E1F */ 
    memcpy(K, constantDF, keylen);

    i.dword = 0;

    while(lentemp < (keylen + CTRDRBG_OUT_LEN)) {
        /* iv = i || 0^(CTRDRBG_OUT_LEN - len(i)) */
        iv[0] = i.byte[3];
        iv[1] = i.byte[2];
        iv[2] = i.byte[1]; 
        iv[3] = i.byte[0];
        
        memset(iv + 4, 0, CTRDRBG_OUT_LEN - 4);

        /* temp = temp || BCC(K, (IV || S) */
        memcpy(data, iv, CTRDRBG_OUT_LEN);
        BCC(temp + lentemp, K, data, CTRDRBG_OUT_LEN + lenS, type);
        lentemp += CTRDRBG_OUT_LEN;

        /* i = i + 1 */
        i.dword +=1; 
    }

    /* K = leftmost(temp, keylen) */
    memcpy(K, temp, keylen);

    /* X = select(temp, keylen + 1, keylen + CTRDRBG_OUT_LEN) */   
    memcpy(X, temp + keylen, CTRDRBG_OUT_LEN);  

    /* temp = the null string */
    lentemp = 0;

    if(type==CTRDRBG_TYPE_AES128) {
        DRV_AES_Init(AES_KEYSZ_128);
        DRV_AES_Load_Program_Key(AES_IN, K, AES_KEYSZ_128, 0);
    } else {
        DRV_AES_Init(AES_KEYSZ_256);
        DRV_AES_Load_Program_Key(AES_IN, K, AES_KEYSZ_256, 0);
    }

    while(lentemp < requestedLen) {
        //AES_EncDec( X, X, AES_ENCRYPT, AES_MODE_ECB);
        DRV_AES_ECB_Crypt(AES_Action_Encryption, X, X, CTRDRBG_OUT_LEN);

        /* temp = temp || X */
        memcpy(temp + lentemp, X, CTRDRBG_OUT_LEN);
        lentemp += CTRDRBG_OUT_LEN;
    }

    /* requestedBytes = leftmost number_of_bytes_to_return of temp */
    memcpy(requestedBytes, temp, requestedLen);   
}

void DRV_CTRDRBG_Update(uint8_t *providedData, const uint8_t type)
{
    uint8_t keylen, KeySize;
    uint8_t temp[CTRDRBG_SEED_LEN];
    uint8_t output_block[CTRDRBG_OUT_LEN];
    uint8_t lentemp=0;

    memset(temp, 0, ctr_seedlen);           /* Avoid a lot of valgrind error */
    memset(output_block, 0, CTRDRBG_OUT_LEN);    /* Avoid a lot of valgrind error */
    
    if(type==CTRDRBG_TYPE_AES128) {
        keylen = 16;
        DRV_AES_Init(AES_KEYSZ_128);
        DRV_AES_Load_Program_Key(AES_IN, ctr_buf_CKey, AES_KEYSZ_128, 0);
    } else {
        keylen = 32;
        DRV_AES_Init(AES_KEYSZ_256);
        DRV_AES_Load_Program_Key(AES_IN, ctr_buf_CKey, AES_KEYSZ_256, 0);
    }

    while(lentemp < ctr_seedlen) {
        /* V = (V + 1) mod 2^CTRDRBG_OUT_LEN */
        DRBG_IncrData(ctr_buf_V, CTRDRBG_OUT_LEN);
    
        /* output_block = Block_Encrypt(Key, V) */
        DRV_AES_ECB_Crypt(AES_Action_Encryption, ctr_buf_V, output_block, CTRDRBG_OUT_LEN);  
        
        /* tmp = tmp || output_block */
        /* we guarantee that we do not overflow on tmp */
        if(lentemp + CTRDRBG_OUT_LEN > ctr_seedlen) {
            uint16_t diff = ctr_seedlen - lentemp;
            memcpy(temp + lentemp, output_block , diff);
            lentemp += diff;
        } else {
            memcpy(temp + lentemp, output_block, CTRDRBG_OUT_LEN);
            lentemp += CTRDRBG_OUT_LEN;
        }
    }

    /* temp = temp ^ providedData */
    XorData(temp, temp, providedData, ctr_seedlen);

    /* Key = Leftmost keylen bytes of temp , Key = leftmost(temp,keylen) */
    memcpy(ctr_buf_CKey, temp, keylen);

    /* V = Rightmost bytes of temp , V = rightmost(tmp,blocklen)*/
    memcpy(ctr_buf_V, temp + keylen, CTRDRBG_OUT_LEN);
}

void TRNG_GetRandomNumber(uint8_t* outputTRNG, uint8_t len)
{
    uint8_t i, m = len / 4;
    uint8_t trngNumer[4];

    DRV_TRNG_Init();                       // TRNG initialization

    for(i = 0; i < m; i++) {
        DRV_TRNG_Generate(trngNumer);      // generate a 32-bit true random number
        memcpy(outputTRNG, trngNumer, 4);
        outputTRNG+=4;
    }

    if(len % 4 != 0) {                     // If length is not a multiple of 4
        DRV_TRNG_Generate(trngNumer);      // generate a 32-bit true random number
        memcpy(outputTRNG, trngNumer, len % 4);
    }
}

//******************************************************************************
// name         : DRV_CTRDRBG_Instantiate()
// description  : Acquire entropy input, nonce, personalization string
//                Create seed and initial internal states
// input param  : - entropyInput : entropy data
//                - entropyLength : entropy data length
//                - nonce : nonce data
//                - nonceLength : nonce data length
//                - persString : personalization String
//                - persLength : personalization String length
//                - type : DRBG algorithm to use (AES128 or AES256)
//                - df : use derivation function or not
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_CTRDRBG_Instantiate(uint8_t* entropyInput, uint8_t entropyLength, uint8_t* nonce,
uint8_t nonceLength, uint8_t* persString, uint8_t persLength, uint8_t type, uint8_t df)
{
    uint8_t seed_material[64+64+32];
    uint8_t seed_material2[48];
    uint8_t keylen = 0; 

    if(type == CTRDRBG_TYPE_AES128) {
        ctr_seedlen = 32;
        keylen = 16;
    } else if(type == CTRDRBG_TYPE_AES256) {
        ctr_seedlen = 48;
        keylen = 32;
    }

    if(df) {
        /*seed_material = entropyInput || nonce || personalization string  */
        memcpy(seed_material, entropyInput, entropyLength);  
        if((nonce != NULL) && (nonceLength > 0)) {
            memcpy(seed_material + entropyLength, nonce, nonceLength);
        }
        
        if((persString)!=NULL && (persLength>0)) {
            memcpy(seed_material + entropyLength + nonceLength, persString, persLength);
        }

        //memcpy(buf,seed_material,entropyLength + nonceLength + persLength);
        Block_Cipher_df(seed_material, entropyLength + nonceLength + persLength, seed_material2, ctr_seedlen, type);
    } else {
        memcpy(seed_material2,entropyInput,entropyLength);
    }
    
    ctr_type = type;
    ctr_df = df;

    /* Key = 0^keylen */
    memset(ctr_buf_CKey,0,keylen);

    /* V = 0^CTRDRBG_OUT_LEN */
    memset(ctr_buf_V,0,CTRDRBG_OUT_LEN);

    /* (Key, V) = DRV_CTRDRBG_Update(seed_material, Key, V) */
    DRV_CTRDRBG_Update(seed_material2, type); 

    /* reseedCounter = 1 */
    ctr_reseedCounter = 1;
}

//******************************************************************************
// name         : DRV_CTRDRBG_Reseed()
// description  : Acquire new entropy input, Create new seed and internal states
// input param  : - entropyInput : entropy data
//                - entropyLength : entropy data length
//                - additionalInput : additional data
//                - addLength : additional data length
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_CTRDRBG_Reseed(uint8_t* entropyInput, uint8_t entropyLength, uint8_t* additionalInput, uint8_t addLength)
{
    uint8_t seed_material[128];
    uint8_t seed_material2[CTRDRBG_SEED_LEN];
    
    memcpy(seed_material, entropyInput, entropyLength); 

    if((additionalInput != NULL) && (addLength > 0)) {
        memcpy(seed_material + entropyLength, additionalInput, addLength);
    }
    
    /* seed_material = Block_Cipher_df(seed_material, seedlen) */
    Block_Cipher_df(seed_material, entropyLength + addLength, seed_material2, ctr_seedlen, ctr_type);

    DRV_CTRDRBG_Update(seed_material2, ctr_type);

    /* reseedCounter = 1 */
    ctr_reseedCounter = 1;
}

//******************************************************************************
// name         : DRV_CTRDRBG_Generate()
// description  : Produce pseudorandom bits, Update Internal states
// input param  : - additionalInput : additional data
//                - addLength : additional data length
//                - entropyPRInput : entropy prediction data
//                - entropyPRLength : entropy prediction data length
//                - returnedBytes : returned data
//                - requestedLength : returned data length
// output param : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_CTRDRBG_Generate(uint8_t* additionalInput, uint8_t addLength, uint8_t* entropyPRInput,
uint8_t entropyPRLength, uint8_t* returnedBytes, uint8_t requestedLength)
{
    uint8_t lentemp=0;
    uint8_t add_input[CTRDRBG_SEED_LEN];
    uint8_t output_block[CTRDRBG_OUT_LEN];
    uint8_t temp[128];
    uint8_t KeySize;

    if((entropyPRInput != NULL) && (entropyPRLength > 0)) {
        DRV_CTRDRBG_Reseed(entropyPRInput,entropyPRLength,additionalInput,addLength);
        additionalInput = NULL;
    }

    if((additionalInput != NULL) && (addLength > 0)) {
        if(ctr_df) {
            /* additionalInput = Block_Cipher_df(additionnal_input, seedlen) */
            Block_Cipher_df(additionalInput, addLength, add_input, ctr_seedlen, ctr_type);

            DRV_CTRDRBG_Update(add_input, ctr_type);
        } else if(addLength < ctr_seedlen) {
            memset(add_input,0,CTRDRBG_SEED_LEN);
            memcpy(add_input,additionalInput,addLength);
            /* (Key, V) = DRV_CTRDRBG_Update(additionalInput, Key, V) */
            DRV_CTRDRBG_Update(add_input, ctr_type);
        }
    } else {
        /* additionalInput = 0 ^seedlen */
        memset(add_input, 0, ctr_seedlen);
    }

    if(ctr_type==CTRDRBG_TYPE_AES128) {
        DRV_AES_Init(AES_KEYSZ_128);
        DRV_AES_Load_Program_Key(AES_IN, ctr_buf_CKey, AES_KEYSZ_128, 0);
    } else {
        DRV_AES_Init(AES_KEYSZ_256);
        DRV_AES_Load_Program_Key(AES_IN, ctr_buf_CKey, AES_KEYSZ_256, 0);
    }

    while(lentemp < requestedLength) {
        /* V = (V + 1) mod 2^CTRDRBG_OUT_LEN */
        DRBG_IncrData(ctr_buf_V, CTRDRBG_OUT_LEN);

        /* output_block = Block_Encrypt(Key, V) */
        // AES_EncDec( ctr_buf_V, output_block, AES_ENCRYPT, AES_MODE_ECB);
        DRV_AES_ECB_Crypt(AES_Action_Encryption, ctr_buf_V, output_block, CTRDRBG_OUT_LEN);         

        /* temp = temp || output_block */
        /* we guarantee that we do noyt overflow on temp */
        if((lentemp + CTRDRBG_OUT_LEN) > requestedLength) {
            uint32_t diff = requestedLength - lentemp;
            memcpy(temp + lentemp, output_block, diff);
            lentemp += diff;
        } else {
            memcpy(temp + lentemp, output_block, CTRDRBG_OUT_LEN);
            lentemp += CTRDRBG_OUT_LEN;
        }
    }

    /* returnedBytes = lestmost requestedLength of temp */ 
    memcpy(returnedBytes, temp, requestedLength);
    
    /* (Key, V) = DRV_CTRDRBG_Update(additionalInput, Key, V) */
    DRV_CTRDRBG_Update(add_input, ctr_type);
    
    /* reseed_coutner = reseedCounter + 1 */
    ctr_reseedCounter += 1;
}

//******************************************************************************
// name         : DRV_CTRDRBG_GetRandomNumber()
// description  : generate 32-byte random numbers using CtrDRBG 
// output param : - outputDRBG : 32-byte random numbers
// retval       : 
// note         : 
//******************************************************************************
void DRV_CTRDRBG_GetRandomNumber(uint8_t* outputDRBG)
{
    uint8_t DRBG_Entropy[32];
    uint8_t DRBG_Nonce[16];

    TRNG_GetRandomNumber(DRBG_Entropy, 32);  // Get 32 bytes random number for entropy from TRNG
    TRNG_GetRandomNumber(DRBG_Nonce, 16);    // Get 16 bytes random number for nonce from TRNG
    // DRBG Instantiate  , Entropy = 32 bytes , Nonce = 16 bytes . Personal data = 0 byte
    DRV_CTRDRBG_Instantiate(DRBG_Entropy, 32, DRBG_Nonce, 16, NULL, 0, CTRDRBG_TYPE_AES128, 1);
    TRNG_GetRandomNumber(DRBG_Entropy, 32);  // Get 32 bytes random number for entropyPR from TRNG
    // DRBG Generate  , EntropyPR = 32 bytes
    DRV_CTRDRBG_Generate(NULL, 0, DRBG_Entropy, 32, outputDRBG, 32);
}