//******************************************************************************
// file         : DRV_SQ7705_AES.c
// version      : V1.2 2024/01/23
// description  : AES related functions
// note         : AES related functions are gathered in this subroutine
//
//                AES Block Diagram
//
//                +-------------------+     +-----------+
//            +-->|Key Buffer(32-byte)|---->|           |
//            |   +-------------------+     |           |     +----------------------+
// +-----+    |   +---------------------+   |    AES    |     |                      |   +------+
// |AESIN|----+-->|Input Buffer(16-byte)|-->|           |---->|Output Buffer(16-byte)|-->|AESOUT|
// +-----+    |   +---------------------+   |Computation| +-->|                      |   +------+
//            |   +-----------+             |           | |   +----------------------+
//            +-->|IV(16-byte)|------------>|           | |
//            |   +-----------+             +-----------+ |
//            +-------------------------------------------+
//******************************************************************************
#include "DRV_SQ7705_AES.h"

struct AES_Device {
    uint8_t             iv_loaded;      /*!< iv_loaded: true, false */
    uint8_t             key_loaded;     /*!< key_loaded: true, false */
    uint8_t             initialized;    /*!< initialized: true, false */
    uint8_t             mode_action;    /*!< initialized: true, false */
} aes_dev;

uint16_t adm_IntAES_Flag;
uint8_t CMD_Tmp;
uint8_t adm_casediv = AES_CLKDIV_CRYPTO_DIV8;
uint8_t aesOutput[16];

//******************************************************************************
// name         : DRV_AES_Init()
// description  : AES initialization
// input param  : aesKeySize : AES Key Size  0:AES-128, 2:AES-256
// output param :
// retval       : none
// note         :
//******************************************************************************
void DRV_AES_Init(uint8_t aesKeySize)
{
    //PRSTR4_AES = 1;        //reset AES
    PCKEN4_AES = 1;
    // AESCR[7:6]:CLKDIV [5]:CLKSEL [3]:AESIE [2:1]:KEYSZ [0]:AESEN
    AESCR = (adm_casediv | AES_CLKSEL_CRYPTO | AES_IE_EN | aesKeySize | AES_EN);

    __ASM("DI");
    IFR_AES = 0;
    IER_AES = 1;
    __ASM("EI");

    aes_dev.initialized = TRUE;
}

//******************************************************************************
// name         : DRV_AES_Deinit()
// description  : Deinitialize AES function
// input param  :
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_AES_Deinit(void)
{
    PCKEN4_AES = 0;

    __ASM("DI");
    IFR_AES = 0;
    IER_AES = 0;
    __ASM("EI");

    aes_dev.initialized = FALSE;
}

//******************************************************************************
// name         : DRV_AES_Key_Size()
// description  : Setting AES Key Size
// input param  : aesKeySize : AES Key Size  0:AES-128, 2:AES-256
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_AES_Key_Size(uint8_t aesKeySize)
{
    while (((uint8_t)(AESSR)) & 0x80);   //Check AES Busy
    AESCR_KEYSZ = (uint8_t)aesKeySize >> 1;
}

//******************************************************************************
// name         : DRV_AES_Load_Key_Fm_AESIN()
// description  : AES Load Key from input key data to key buffer
// input param  : keyData : input key data
//                keyLength : input key length
// output param :
// retval       : AES_Status
// note         :
//******************************************************************************
uint8_t DRV_AES_Load_Key_Fm_AESIN(const uint8_t *keyData, const uint8_t keyLength)
{
    if (aes_dev.initialized) {
        uint8_t i;
        if (keyLength == 16) {
            DRV_AES_Key_Size(AES_KEYSZ_128);
        } else {
            DRV_AES_Key_Size(AES_KEYSZ_256);
        }
        AESMD = (Load_Key | AES_IN);
        for (i = 0; i < keyLength; i++) {
            AESIN = keyData[i];
            while (((uint8_t)(AESSR)) & 0x80);
        }
        aes_dev.key_loaded = TRUE;
        return AESSR;
    } else {
        return AES_Err_Key_Not_Loaded;
    }
}

//******************************************************************************
// name         : DRV_AES_Load_Key_Fm_InternalKey()
// description  : AES Load Key from internal key slot to key buffer
// input param  : keyNum : AES key number
// output param :
// retval       : AES_Status
// note         :
//******************************************************************************
uint8_t DRV_AES_Load_Key_Fm_InternalKey(const uint8_t keyNum)
{
    if (aes_dev.initialized) {
        AESMD = (Load_Key | Internal_Key);
        AESIN = keyNum;

        while (((uint8_t)(AESSR)) & 0x80);

        aes_dev.key_loaded = TRUE;
        return AESSR;
    } else {
        return AES_Err_Key_Not_Loaded;
    }
}

//******************************************************************************
// name         : DRV_AES_Load_Key_Fm_RandomNumber()
// description  : AES Load Key from TRNG to key buffer
// input param  :
// output param :
// retval       : AES_Status
// note         :
//******************************************************************************
uint8_t DRV_AES_Load_Key_Fm_RandomNumber(void)
{
    DRV_AES_Key_Size(AES_KEYSZ_128);
    return DRV_AES_Load_Key(Random_Number);
}

//******************************************************************************
// name         : DRV_AES_Load_Key_Fm_InputBuffer()
// description  : AES Load Key from InputBuffer to key buffer
// input param  :
// output param :
// retval       : AES_Status
// note         :
//******************************************************************************
uint8_t DRV_AES_Load_Key_Fm_InputBuffer(void)
{
    DRV_AES_Key_Size(AES_KEYSZ_128);
    return DRV_AES_Load_Key(Input_Buffer);
}

//******************************************************************************
// name         : DRV_AES_Load_Key_Fm_OutputBuffer()
// description  : AES Load Key from OutputBuffer to key buffer
// input param  :
// output param :
// retval       : AES_Status
// note         :
//******************************************************************************
uint8_t DRV_AES_Load_Key_Fm_OutputBuffer(void)
{
    DRV_AES_Key_Size(AES_KEYSZ_128);
    return DRV_AES_Load_Key(Output_Buffer);
}

//******************************************************************************
// name         : DRV_AES_Load_Key_Fm_IV()
// description  : AES Load Key from Initialization-vector to key buffer
// input param  :
// output param :
// retval       : AES_Status
// note         :
//******************************************************************************
uint8_t DRV_AES_Load_Key_Fm_IV(void)
{
    DRV_AES_Key_Size(AES_KEYSZ_128);
    return DRV_AES_Load_Key(IV);
}

//******************************************************************************
// name         : DRV_AES_Program_Key()
// description  : Program key buffer data to internal key slot
// input param  : keyNumber : AES key number
// output param :
// retval       : AES_Status
// note         :
//******************************************************************************
uint8_t DRV_AES_Program_Key(const uint8_t keyNumber)
{
    if (aes_dev.key_loaded) {
        AESMD = Program_Key;               // Program Key
        AESIN = keyNumber;                 // Key Number

        while (((uint8_t)(AESSR)) & 0x80); // Check AES Busy
        return AESSR;
    } else {
        return AES_Err_Key_Not_Programmed;
    }
}

//******************************************************************************
// name         : DRV_AES_Load_Program_Key()
// description  : 1. Setting AES Key Size
//                2. AES Load Key from AESIN/Input Buffer/Output Buffer/IV/Random Number to key buffer
//                3. Program key buffer data to internal key slot
// input param  : aesLoadKeyMode : 0: AESIN, 1: Input_Buffer, 2: Output_Buffer,
//                                 3: IV, 4: N/A, 5: Random_Number
//                keyData : input key data (Only for AESIN mode)
//                aesKeySize : AES Key Size  0:AES-128, 2:AES-256 (Only for AESIN mode)
//                keyNumber : AES key number
// output param :
// retval       : AES_Status
// note         :
//******************************************************************************
uint8_t DRV_AES_Load_Program_Key(const uint8_t aesLoadKeyMode, const uint8_t *keyData, const uint8_t aesKeySize, const uint8_t keyNumber)
{
    uint8_t result = AES_Err_Not_Initialized;
    switch (aesLoadKeyMode) {
    case AES_IN:
        DRV_AES_Key_Size(aesKeySize);
        if (aesKeySize == AES_KEYSZ_128)
            result = DRV_AES_Load_Key_Fm_AESIN(keyData, Key_128bit);
        else if (aesKeySize == AES_KEYSZ_256)
            result = DRV_AES_Load_Key_Fm_AESIN(keyData, Key_256bit);
        if (result == AES_OK)
            result = DRV_AES_Program_Key(keyNumber);
        break;
    case Input_Buffer:
        DRV_AES_Key_Size(AES_KEYSZ_128);
        result = DRV_AES_Load_Key_Fm_InputBuffer();
        if (result == AES_OK)
            result = DRV_AES_Program_Key(keyNumber);
        break;
    case Output_Buffer:
        DRV_AES_Key_Size(AES_KEYSZ_128);
        result = DRV_AES_Load_Key_Fm_OutputBuffer();
        if (result == AES_OK)
            result = DRV_AES_Program_Key(keyNumber);
        break;
    case IV:
        DRV_AES_Key_Size(AES_KEYSZ_128);
        result = DRV_AES_Load_Key_Fm_IV();
        if (result == AES_OK)
            result = DRV_AES_Program_Key(keyNumber);
        break;
    case Random_Number:
        DRV_AES_Key_Size(AES_KEYSZ_128);
        result = DRV_AES_Load_Key_Fm_RandomNumber();
        if (result == AES_OK)
            result = DRV_AES_Program_Key(keyNumber);
        break;
    }
    return result;
}
//******************************************************************************
// name         : DRV_AES_Mode_Action
// description  : set AES mode and action
// input param  : mode  : ECB/CFB/CFB/CBC/CTR
//                enc   : AES_Action_Encryption/AES_Action_Decryption
// retval       : AES_OK means OK other failed
// note         :
//******************************************************************************
uint8_t DRV_AES_Mode_Action(const uint8_t mode, const AES_Action enc)
{
    if (aes_dev.key_loaded) {
        if (mode == ECB || (mode >= CBC && mode <= CTR && aes_dev.iv_loaded == TRUE)) {
            if (enc == AES_Action_Encryption) {
                DRV_AES_Encryption(mode);
            } else if (enc == AES_Action_Decryption) {
                DRV_AES_Decryption(mode);
            }
            aes_dev.mode_action = 1;
            return AES_OK;
        } else if (aes_dev.iv_loaded == TRUE) {
            return AES_Err_Wrong_Mode;
        } else {
            return AES_Err_IV_Not_Loaded;
        }
    } else {
        return AES_Err_Key_Not_Loaded;
    }
}
//******************************************************************************
// name         : DRV_AES_Process_Data()
// description  : process data with AES engine ,AES mode and action should been set 
//                by  API DRV_AES_Mode_Action before use this API
// input param  : input  : 16 byte data
// output param : output : 16 byte data 
// retval       : AES_OK means OK other failed
// note         :
//******************************************************************************
uint8_t DRV_AES_Process_Data(const uint8_t *input, uint8_t *output)
{

    if (aes_dev.mode_action == 1) {
        DRV_AES_Load_Input_Buffer(AES_IN, input, AES_BLOCK_SIZE);  //input size = 16-byte
        DRV_AES_Out_Data(output, AES_BLOCK_SIZE);                      //output size = 16
        return AES_OK;
    } else
        return AES_Err_Mode_Action_NotInit;
}
//******************************************************************************
// name         : DRV_AES_Crypt_Length
// description  : aes encrypt/decrypt with length for ECB/CBC/CFB/OFB/CTR
// input param  : mode   : ECB, CBC, CFB, OFB, CTR
//                enc    : Encryption or Decryption
//                input  : input data buffer 
//                output : ouput data buffer 
//                inputLength : input data buffer length , if should be mutiple of 16 (AES Block size)
//                iv: 16 byte initial vector for CBC,CFB,OFB,CTR only                
// output param :
// retval       : AES_OK means OK , other are failed 
//******************************************************************************

uint8_t DRV_AES_Crypt(const uint8_t mode, const AES_Action aesAction, const uint8_t *input, uint8_t *output, uint16_t inputLength, uint8_t *iv)
{
    uint16_t index = 0;
    if(inputLength%AES_BLOCK_SIZE!=0) 
        return AES_Err_Invalid_Length;
       
    if (mode != ECB)
        DRV_AES_Load_IV(AES_IN, iv, 16);
    DRV_AES_Mode_Action(mode, aesAction);
    while (1) {
        DRV_AES_Process_Data(&input[index], &output[index]);
        index = index + AES_BLOCK_SIZE;
        if (index >= inputLength)
            break;

    }
    return AES_OK;
}

//******************************************************************************
// name         : DRV_AES_ECB_Crypt()
// description  :AES ECB process 
// input param  : enc    : Encryption or Decryption
//                input  : input data buffer 
//                output : ouput data buffer 
//                inputLength : input data buffer length , if should be mutiple of 16(AES Block size)
// output param : output : process result , data length is inputLength
// retval       : AES_OK means OK , other are failed 
// note         :
//******************************************************************************
uint8_t DRV_AES_ECB_Crypt(const AES_Action enc, const uint8_t *input, uint8_t *output,uint16_t inputLength)
{
    uint8_t *iv;
    return  DRV_AES_Crypt(ECB,enc, input,  output, inputLength, iv);    
}

//******************************************************************************
// name         : DRV_AES_CBC_Crypt()
// description  :AES CBC process 
// input param  : enc    : Encryption or Decryption
//                input  : input data buffer 
//                output : ouput data buffer 
//                inputLength : input data buffer length , if should be mutiple of 16 (AES Block size)
//                iv: 16 byte initial vector 
// output param : output : process result, data length is inputLength
// retval       : AES_OK means OK , other are failed
// note         :
//******************************************************************************
uint8_t DRV_AES_CBC_Crypt(const AES_Action enc, const uint8_t *input, uint8_t *output,uint16_t inputLength,uint8_t *iv)
{
    return  DRV_AES_Crypt(CBC,enc, input,  output, inputLength, iv);  
}
//******************************************************************************
// name         : DRV_AES_CFB_Crypt()
// description  :AES CFB process 
// input param  : enc    : Encryption or Decryption
//                input  : input data buffer 
//                output : ouput data buffer 
//                inputLength : input data buffer length , if should be mutiple of 16 (AES Block size)
//                iv: 16 byte initial vector 
// output param : output : process result, data length is inputLength
// retval       : AES_OK means OK , other are failed
// note         :
//******************************************************************************
uint8_t DRV_AES_CFB_Crypt(const AES_Action enc, const uint8_t *input, uint8_t *output,uint16_t inputLength,uint8_t *iv)
{
    return  DRV_AES_Crypt(CFB,enc, input,  output, inputLength, iv);  
}

//******************************************************************************
// name         : DRV_AES_OFB_Crypt()
// description  :AES OFB process 
// input param  : enc    : Encryption or Decryption
//                input  : input data buffer 
//                output : ouput data buffer 
//                inputLength : input data buffer length , if should be mutiple of 16 (AES Block size)
//                iv: 16 byte initial vector 
// output param : output : process result, data length is inputLength
// retval       : AES_OK means OK , other are failed
// note         :
//******************************************************************************
uint8_t DRV_AES_OFB_Crypt(const AES_Action enc, const uint8_t *input, uint8_t *output,uint16_t inputLength,uint8_t *iv)
{
    return  DRV_AES_Crypt(OFB,enc, input,  output, inputLength, iv);  
}

//******************************************************************************
// name         : DRV_AES_CTR_Crypt()
// description  :AES CTR process 
// input param  : enc    : Encryption or Decryption
//                input  : input data buffer 
//                output : ouput data buffer 
//                inputLength : input data buffer length , if should be mutiple of 16 (AES Block size)
//                iv: 16 byte initial vector 
// output param : output : process result, data length is inputLength
// retval       : AES_OK means OK , other are failed
// note         :
//******************************************************************************
uint8_t DRV_AES_CTR_Crypt(const AES_Action enc, const uint8_t *input, uint8_t *output,uint16_t inputLength,uint8_t *iv)
{
    return  DRV_AES_Crypt(CTR,enc, input,  output, inputLength, iv);  
}


//******************************************************************************
// name         : DRV_AES_Encryption()
// description  : AES encryption
// input param  : aesEncryptionMode : 0:ECB, 1:CBC, 2:CFB, 3:OFB, 4:CTR
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_AES_Encryption(uint8_t aesEncryptionMode)
{
    aesEncryptionMode &= 0x0F;

    CMD_Tmp = (Encryption | aesEncryptionMode);
    AESMD = CMD_Tmp;    //AESMD = 0x0000 Setting ECB encryption
}

//******************************************************************************
// name         : DRV_AES_Decryption()
// description  : AES decryption
// input param  : aesDecryptionMode : 0:ECB, 1:CBC, 2:CFB, 3:OFB, 4:CTR
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_AES_Decryption(uint8_t aesDecryptionMode)
{
    aesDecryptionMode &= 0x0F;

    CMD_Tmp = (Decryption | aesDecryptionMode);
    AESMD = CMD_Tmp;
}

//******************************************************************************
// name         : DRV_AES_Load_Key()
// description  : AES Load Key to key buffer
// input param  : aesLoadKeyMode : 0: AESIN, 1: InputBuffer, 2: OutputBuffer,
//                                 3: Initialization Vector, 4: InternalKey, 5: Random
// output param :
// retval       : AES_Status
// note         :
//******************************************************************************
uint8_t DRV_AES_Load_Key(const uint8_t aesLoadKeyMode)
{
    if (aes_dev.initialized) {
        AESMD = (Load_Key | aesLoadKeyMode);

        while (((uint8_t)(AESSR)) & 0x80);

        aes_dev.key_loaded = TRUE;
        return AESSR;
    } else {
        return AES_Err_Key_Not_Loaded;
    }
}

//******************************************************************************
// name         : DRV_AES_Load_IV()
// description  : load AES initial value
// input param  : aesLoadIV_Mode : 0: AES_IN, 1: Input_Buffer, 2: Output_Buffer, 3,4,5: N/A
//                IV_Data : Initial vector data
//                IV_length : Initial vector byte length
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_AES_Load_IV(const uint8_t aesLoadIV_Mode, uint8_t *IV_Data, const uint8_t IV_length)
{
    uint8_t i;

    if (aesLoadIV_Mode == AES_IN) {
        AESMD = (Load_IV | aesLoadIV_Mode);        
        for (i = 0; i < IV_length; i++) {
            AESIN = IV_Data[i];
            while (((uint8_t)(AESSR)) & 0x80);
        }
        aes_dev.iv_loaded = TRUE;
    } else {
        AESMD = (Load_IV | aesLoadIV_Mode);
        while (((uint8_t)(AESSR)) & 0x80);
        aes_dev.iv_loaded = TRUE;
    }
}

//******************************************************************************
// name         : DRV_AES_Load_Input_Buffer()
// description  : AES load plaintext / ciphertext to Input_Buffer
// input param  : aesLoadInputBufferMode : 0: AES_IN, 1: N/A, 2: Output_Buffer, 3: IV, 4,5: N/A
//                inputBufferData : input buffer
//                inputBufferLength : input buffer length
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_AES_Load_Input_Buffer(const uint8_t aesLoadInputBufferMode, const uint8_t *inputBufferData, const uint8_t inputBufferLength)
{
    uint8_t i;

    if (aesLoadInputBufferMode == AES_IN) {
        AESMD = (Load_Input_Buffer | aesLoadInputBufferMode);
        for (i = 0; i < inputBufferLength; i++) {
            AESIN = inputBufferData[i];
            while (((uint8_t)(AESSR)) & 0x80) {
                __ASM("NOP");
            }
        }
    } else {
        AESMD = (Load_Input_Buffer | aesLoadInputBufferMode);
        while (((uint8_t)(AESSR)) & 0x80) {
            __ASM("NOP");
        }
    }
}

//******************************************************************************
// name         : DRV_AES_Load_Output_Buffer()
// description  : AES load Output_Buffer
// input param  : aesLoadOutputBufferMode : 0: AES_IN, 1: Input_Buffer, 2: N/A, 3: IV, 4,5: N/A
//                outputBufferLength : output buffer length
// output param : outputBufferData : output buffer
// retval       :
// note         :
//******************************************************************************
void DRV_AES_Load_Output_Buffer(const uint8_t aesLoadOutputBufferMode, const uint8_t *outputBufferData, const uint8_t outputBufferLength)
{
    uint8_t i;

    if (aesLoadOutputBufferMode == AES_IN) {
        AESMD = (Load_Output_Buffer | aesLoadOutputBufferMode);
        for (i = 0; i < outputBufferLength; i++) {
            AESIN = outputBufferData[i];

            while (((uint8_t)(AESSR)) & 0x80) {
                __ASM("NOP");
            }
        }
    } else {
        AESMD = (Load_Output_Buffer | aesLoadOutputBufferMode);
        while (((uint8_t)(AESSR)) & 0x80) {
            __ASM("NOP");
        }
    }
}

//******************************************************************************
// name         : DRV_AES_Peripheral_circuit_Reset()
// description  : reset AES related circuit
// input param  :
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_AES_Peripheral_circuit_Reset(void)
{
    PRSTR4_AES = 1;        //reset AES related circui
}

//******************************************************************************
// name         : DRV_AES_Reset()
// description  : reset AES
// input param  :
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_AES_Reset(void)
{
    AESMD = Reset;
    while (((uint8_t)(AESSR)) & 0x80) {
        __ASM("NOP");
    }
}

//******************************************************************************
// name         : DRV_AES_Abort()
// description  : abort previous command
// input param  :
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_AES_Abort(void)
{
    AESMD = Abort;
    while (((uint8_t)(AESSR)) & 0x80) {
        __ASM("NOP");
    }
}

//******************************************************************************
// name         : DRV_AES_Out_Data()
// description  : AES out ciphertext / plaintext / ....
// input param  : length  : Output data length
// output param : outData : AES output data
// retval       :
// note         : DRV_AES_Out_Data(CIPHERTEXT,16);
//******************************************************************************
void DRV_AES_Out_Data(uint8_t *outData, const uint8_t length)
{
    uint8_t i;
    for (i = 0; i < length; i++) {
        while (((uint8_t)(AESSR)) & 0x80) {
            __ASM("NOP");
        }
        outData[i] = AESOUT;
    }
}

//******************************************************************************
// name         : DRV_AES_XOR()
// description  : AES Load XOR data
// input param  : aesLoadXOR_Mode : AES_IN: 0, Input_Buffer: 1, Output_Buffer: 2,
//                                  IV: 3, Internal_Key: 4, Random_Number: 5
//                inputXOR_Data : input buffer for XOR data
//                inputXOR_Length : length of XOR data
// output param :
// retval       :
// note         :
//******************************************************************************
void DRV_AES_XOR(const uint8_t aesLoadXOR_Mode, const uint8_t *inputXOR_Data, const uint8_t inputXOR_Length)
{
    uint8_t i;

    if (aesLoadXOR_Mode == AES_IN) {
        AESMD = (XOR | aesLoadXOR_Mode);
        for (i = 0; i < inputXOR_Length; i++) {
            AESIN = inputXOR_Data[i];

            while (((uint8_t)(AESSR)) & 0x80) {
                __ASM("NOP");
            }
        }
    } else {
        AESMD = (XOR | aesLoadXOR_Mode);
        while (((uint8_t)(AESSR)) & 0x80) {
            __ASM("NOP");
        }
    }
}

//******************************************************************************
// name         : IntAES()
// description  : AES interrupt service routine
// input param  :
// output param :
// retval       :
// note         :
//******************************************************************************
void __interrupt IntAES(void)
{
    __ASM("NOP");                              // no operation
    adm_IntAES_Flag ++;
}
