//******************************************************************************
// file         : DRV_SQ7705_GPIO.c
// version      : V1.2 2023/08/17
// description  : GPIO related functions
// note         : GPIO related functions are gathered in this subroutine
//******************************************************************************
#include "DRV_SQ7705_GPIO.h"


//******************************************************************************
// name         : DRV_GPIO_Init()
// description  : GPIO initialization
// input param  : - GPIOx  : pin number
//                - inputOrOutput : set pin to input or output            
//                - defPUPD  : enable built-in pull-up or pull-down resistor
// output param :
// retval       : 
// note         : 
//******************************************************************************
void DRV_GPIO_Init(uint8_t GPIOx, uint8_t inputOrOutput, uint8_t defPUPD)
{
    uint8_t GPIONum = GPIOx >> 4;
    uint8_t pinNum = GPIOx & 0x0F;
    uint8_t* PxCFGCR = (uint8_t*)(0x0140 + (GPIONum * 2)); //PxCFGCR
    uint8_t* GPIOAdr = (uint8_t*)(0x0100 + GPIONum);       //PxOE

    *PxCFGCR = 0x80 | ((pinNum & 0x7) << 4);      //Set pin to GPIO mode

    if(inputOrOutput == GPIO_OUTPUT){
        SET_BIT(*GPIOAdr, pinNum);
    } else if(inputOrOutput == GPIO_INPUT){
        CLR_BIT(*GPIOAdr, pinNum);
    }

    if(defPUPD == GPIO_PULL_UP){
        GPIOAdr = (uint8_t*)(0x0110 + GPIONum);   //PxPU
        SET_BIT(*GPIOAdr, pinNum);
        GPIOAdr = (uint8_t*)(0x0120 + GPIONum);   //PxPD
        CLR_BIT(*GPIOAdr, pinNum);
    } else if(defPUPD == GPIO_PULL_DOWN){
        GPIOAdr = (uint8_t*)(0x0110 + GPIONum);   //PxPU
        CLR_BIT(*GPIOAdr, pinNum);
        GPIOAdr = (uint8_t*)(0x0120 + GPIONum);   //PxPD
        SET_BIT(*GPIOAdr, pinNum);
    }
}

//******************************************************************************
// name         : DRV_GPIO_ReadPin()
// description  : read pin voltage level
// input param  : - GPIOx  : pin number
// output param :
// retval       : pin voltage level : 0 : L level
//                                    1 : H level
// note         :
//******************************************************************************
uint8_t DRV_GPIO_ReadPin(uint8_t GPIOx)
{
    uint8_t GPIONum = GPIOx >> 4;
    uint8_t pinNum = GPIOx & 0x0F;
    uint8_t* GPIOAdr = (uint8_t*)(0x00F0 + GPIONum);
    return GET_BIT(*GPIOAdr, pinNum);
}

//******************************************************************************
// name         : DRV_GPIO_WritePin
// description  : set pin voltage level
// input param  : - GPIOx  : select specific GPIO          
//                - highOrLow  : set pin voltage level to high or low
// output param :
// retval       : 
// note         :
//******************************************************************************
void DRV_GPIO_WritePin(uint8_t GPIOx, uint8_t highOrLow)
{
    uint8_t GPIONum = GPIOx >> 4;
    uint8_t pinNum = GPIOx & 0x0F;
    uint8_t* GPIOAdr = (uint8_t*)(0x00E0 + GPIONum);
    if(highOrLow == GPIO_PIN_HIGH){
        SET_BIT(*GPIOAdr, pinNum);
    } else if(highOrLow == GPIO_PIN_LOW){
        CLR_BIT(*GPIOAdr, pinNum);
    }
}

//******************************************************************************
// name         : DRV_GPIO_TogglePin()
// description  : reverse pin voltage level
// input param  : - GPIOx  : pin number
// output param :
// retval       : 
// note         :
//******************************************************************************
void DRV_GPIO_TogglePin(uint8_t GPIOx)
{
    uint8_t GPIONum = GPIOx >> 4;
    uint8_t pinNum = GPIOx & 0x0F;
    uint8_t* GPIOAdr = (uint8_t*)(0x00E0 + GPIONum);
    if(GET_BIT(*GPIOAdr, pinNum) == 1){
        CLR_BIT(*GPIOAdr, pinNum);
    } else if(GET_BIT(*GPIOAdr, pinNum) == 0){
        SET_BIT(*GPIOAdr, pinNum);
    }
}