//******************************************************************************
// file         : DRV_SQ7705_CRC.c
// version      : V1.3 2025/3/13
// description  : CRC related functions
// note         : CRC polynomial CRC-16-CCITT : "X^16+X^12+X^5+1"
//******************************************************************************
#include "DRV_SQ7705_CRC.h"

//DICADR2 Memomry start address based on DMCFG = 0x40
uint8_t crcAnswer[2] = {0, 0}; // CRC result

//******************************************************************************
// name         : DRV_CRC_SrcInputData_Init()
// description  : CRC initialization
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_CRC_SrcInputData_Init(void)
{
    PCKEN4_CRC = 1;  // enable CRC peripheral clock
    CRCCR_SETONE = 0;
}

//******************************************************************************
// name         : DRV_CRC_SrcInputData_DeInit()
// description  : deinitialize CRC
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_CRC_SrcInputData_DeInit(void)
{
    PCKEN4_CRC = 0;  // disable CRC peripheral clock
}

//******************************************************************************
// name         : DRV_CRC_SrcInputData_Start()
// description  : start CRC
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_CRC_SrcInputData_Start(void)
{
    CRCCR_CRCEN = 0;
    CRCCR_CRCEN = 1;  // enable CRC
}

//******************************************************************************
// name         : DRV_CRC_SrcInputData_Stop()
// description  : stop CRC and clear result
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_CRC_SrcInputData_Stop(void)
{
    CRCCR_CRCEN = 0;  // disable CRC
}

//******************************************************************************
// name         : DRV_CRC_SrcInputData_Calculate()
// description  : calculate CRC, data from user-defined array
// input param  : - inputLen : the length of input data
//                - inputData : input data
// output param : - crcResult : 2-byte CRC result
// retval       : 
// note         : 
//******************************************************************************
void DRV_CRC_SrcInputData_Calculate(uint8_t inputLen, uint8_t *inputData, uint8_t *crcResult)
{
    uint8_t i = 0;

    for (i = 0; i < inputLen; i++) {
        CRCDI = inputData[i];      // write data to CRC datain register
    }

    crcResult[0] = CRCDO0;         // read CRCDO0 first, then read CRCDO1
    crcResult[1] = CRCDO1;
    crcAnswer[0] = crcResult[0];
    crcAnswer[1] = crcResult[1];
}

//******************************************************************************
// name         : DRV_CRC_DIC_Init()
// description  : DIC initialization
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_CRC_DIC_Init(void)
{
    PCKEN4_CRC = 1;                // enable CRC peripheral clock
    CRCCR_SETONE = 0;
    PCKEN4_DIC = 1;                // initialization DIC
    CRCCR_CRCEN = 1;
}

//******************************************************************************
// name         : DRV_CRC_DIC_Calculate()
// description  : calculate CRC, data from sram, flash or eeprom
// input param  : - dataSrc : data source
//                      typedef enum{
//                         SRC_FLAH_ARRAY=0,
//                         SRC_SRAM=8,          //0x1000~0x2FFF
//                         SRC_EEPROM_ARRAY=16  //0x3000~0x4DFF
//                      }CRC_DIC_SRC;
//                - address :
//                      #define SRC_FLASH_ARRAY_ADDR 0x0000
//                      #define SRC_SRAM_START_ADDR 0x1000
//                      #define SRC_EEPROM_ARRAY_ADDR 0x3000
//                - size : data size , if size = 0, euqal to 65536 bytes
// output param : - crcResult : 2-byte CRC result
// retval       : 
// note         : source, start address and size of data must consistent with system settings, or it won't response anything
//******************************************************************************
void DRV_CRC_DIC_Calculate(CRC_DIC_SRC dataSrc, uint16_t address, uint16_t size, uint8_t *crcResult)
{
    DICADR0 = (uint8_t)(address & 0X00FF);  // set DIC Address
    DICADR1 = (uint8_t)(address >> 8);
    DICADR2_AREA = (uint8_t)dataSrc;
    DICCNT0 = (uint8_t)(size & 0X00FF);
    DICCNT1 = (uint8_t)(size >> 8);
    DICCR_DICEN = 1;                        // enable DIC

    while (DICCR_DICEN);                    // wait for DIC to be ready

    crcAnswer[0] = CRCDO0;                  // read CRCDO0 first, then read CRCDO1
    crcAnswer[1] = CRCDO1;
    crcResult[0] = crcAnswer[0];
    crcResult[1] = crcAnswer[1];

}

//******************************************************************************
// name         : DRV_CRC_DIC_DeInit()
// description  : deinitialize DIC
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_CRC_DIC_DeInit(void)
{
    PCKEN4_DIC = 0;                         // initialization DIC
}

//******************************************************************************
// name         : DRV_CRC_Check()
// description  : check if the calculated CRC value is correct
// input param  : - crc_ref : CRC comparison value
// retval       : 0 correct result ,1 fail result
// note         : 
//******************************************************************************
uint8_t DRV_CRC_Check(uint8_t *crcRef)
{
    if ((crcRef[0] == crcAnswer[0]) && (crcRef[1] == crcAnswer[1]) )
        return 1;
    else
        return 0;
}

