//******************************************************************************
// file         : DRV_SQ7705_ADC.c
// version      : V1.2 2023/09/01
// description  : ADC related functions
// note         : ADC related functions are gathered in this subroutine
//******************************************************************************
#include "DRV_SQ7705_ADC.h"

//******************************************************************************
// name         : DRV_ADC_Channel_Init()
// description  : set IO as ADC channel
// input param  : - AINx : ADC channel
// retval       : 
// note         : 
//******************************************************************************
void DRV_ADC_Channel_Init(ADC_AIN_CHANNEL_LIST AINx)
{
    if (AINx < 8) {
        P1CFGCR = 0x87 + (0x10 * AINx);
    } else {
        P5CFGCR = 0x87 + (0x10 * (AINx - 8));
    }
}

//******************************************************************************
// name         : DRV_ADC_ChannelScan_Init()
// description  : initialize ADC scan channel
// input param  : - AINx : ADC channel
// retval       : 
// note         : 
//******************************************************************************
void DRV_ADC_ChannelScan_Init(ADC_AIN_CHANNEL_LIST AINx)
{
    if (AINx < 8) {
        ADCSCAN0 |= ( 1 << AINx);
    } else {
        ADCSCAN1 |= ( 1 << (AINx - 8));
    }
}

//******************************************************************************
// name         : DRV_ADC_Init()
// description  : ADC initialization
// input param  : ADC_INIT_STRUCT ->sampleRate    : sampling rate
//                                ->refVoltage    : reference voltage
//                                ->autoPowerDown : enable auto power-down
//                                ->convertMode   : convertion mode selection
//                                ->event         : trigger event
//                                ->irq           : interrupt time
// retval       : 
// note         : 
//****************************************************************************** 
void DRV_ADC_Init(ADC_INIT_STRUCT *adcInit)
{
//====== function enable ======       // this must be enabled first, the relevant staging settings later to be useful
    PCKEN5_ADC = 1;                   // enable ADC peripheral clock



//====== parameter setting ======
// Set ADC Convert Speed
    ADCCKDIV_ADCKDIV = adcInit->sampleRate;

    ADCCR0_ADEN  = 1;

    ADCCR0_AMD       = adcInit->convertMode;
    ADCCR0_IRFEN     = adcInit->refVoltage;
    ADCCR0_AUTOPD    = adcInit->autoPowerDown;
    ADCCR1_EVSEL     = adcInit->event;
    ADCCR0_INTLV     = adcInit->irq;

    __ASM("DI");
    IER_ADC = 1;                      // enable ADC interrupt
    ADCCR0_INTEN = 1;                 // enable AD conversion interrupt
    __ASM("EI");
}

//******************************************************************************
// name         : DRV_ADC_LevelCompare_Init()
// description  : ADC level comparison initialization
// input param  : ADC_LEVEL_COMPARE_INIT_STRUCT -> levelSetting : ADC level comparison setting 
//                                              -> singleOrAll  : ADC samples
//                                              -> channel      : specific channel
//                                              -> LowLevel     : low level comparison value setting
//                                              -> HighLevel    : high level comparison value setting
// retval       : 
// note         : 
//******************************************************************************
void DRV_ADC_LevelCompare_Init(ADC_LEVEL_COMPARE_INIT_STRUCT *adcLVCMPInit)
{
    ADCLV_LVCMP   = adcLVCMPInit->levelSetting;
    ADCLV_LVINTVL = adcLVCMPInit->singleOrAll;
    ADCLV_LVSEL   = adcLVCMPInit->channel;      // set comparison ADC channel

    ADCLLV        = adcLVCMPInit->LowLevel;
    ADCHLV        = adcLVCMPInit->HighLevel;

    __ASM("DI");
    IER_ADC = 1;                          // enable ADC interrupt
    ADCCR2_LVINTEN = 1;                   // enable source level detected
    __ASM("EI");
}

//******************************************************************************
// name         : DRV_ADC_ForceStopAndDisable()
// description  : stop and disable ADC
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_ADC_ForceStopAndDisable(void)
{
    ADCCR0_AMD  = 0;        // disable ADC operation and forcibly stop ADC operation
    ADCCR0_ADEN = 0;        // disable ADC

    __ASM("DI");
    IER_ADC = 0;            // disable ADC interrupt
    ADCCR0_INTEN = 0;       // disable AD conversion interrupt
    ADCCR2_LVINTEN = 0;
    __ASM("EI");
}

//******************************************************************************
// name         : DRV_ADC_Convert_Start()
// description  : start AD conversion
// input param  :
// retval       : 
// note         : 
//******************************************************************************
void DRV_ADC_Convert_Start(void)
{
    while (!ADCSR_ADCRDY);       // wait for ADC ready
    ADCSR_ADRS = 1;              // start conversion
}

//******************************************************************************
// name         : DRV_ADC_Read_Value()
// description  : read AD conversion value
// input param  : - AINx : channel selection
// output param : - adcValue : ADC value
// retval       : 
// note         : 
//******************************************************************************
void DRV_ADC_Read_Value(ADC_AIN_CHANNEL_LIST AINx, uint8_t *adcValue)
{
    ADCCHSEL = AINx;

    while (!DRV_ADC_ConvertDone());
    ADCSR_EOCF = 1;               // clear ADC conversion completion flag

    adcValue[0] = ADCDRL;
    adcValue[1] = ADCDRH;
    __ASM("NOP");
}

//******************************************************************************
// name         : DRV_ADC_Convert_Repeat()
// description  : Perform ADC repeat mode conversion and store the conversion value in ADCDRL and ADCDRH and average them
// input param  : - AINx : ADC channel in
// output param : - adcValue : ADC value
// note         : SAIN: select ADC input channel
//                  AIN4        0x4
//                  AIN5        0x5
//                  AIN6        0x6
//                  AIN7        0x7
//                  AIN8        0x8
//                  AIN9        0x9
//                  AIN10       0xA
//
// *this example of the function is demonstrated repeated sampling within 18 times, and maximum 
//  and minimum values in the sampling process are not included in the average consideration
//******************************************************************************
void DRV_ADC_Convert_Repeat(ADC_AIN_CHANNEL_LIST AINx, uint8_t *adcValue)
{
    uint8_t i;
    uint16_t adcData = 0;
    uint16_t adMin = 0, adMax = 4096;
    BIT_LIST_16 ADC_READ;
//====== parameter setting ======
    ADCCHSEL_CHSEL = AINx;                  // select ADC input channel
    //ADCCR0_AMD = 0x03;                    // set ADC operating mode:repeat mode

    while (!ADCSR_ADCRDY);                  // wait for ADC to be ready
    ADCSR_ADRS = 1;                         // start AD conversion
    for ( i = 0; i < (ad_test_times + 2); i++ ) { // repeat sampling 18 times
        while ( !ADCSR_EOCF ) {
            CLR_WDT;                        // wait for AD conversion complete
        }
        ADCSR_EOCF = 1;                     // clear ADC conversion completion flag
        ADC_READ.byte[0] = ADCDRL;          // when reading ADC value,  read low byte first and then read high byte
        ADC_READ.byte[1] = ADCDRH;          // when reading ADC value,  read low byte first and then read high byte
        adcData += ADC_READ.word;
        if ( ADC_READ.word > adMin ) adMin = ADC_READ.word;
        if ( ADC_READ.word < adMax ) adMax = ADC_READ.word;
    }
    ADCSR_ADRS = 0;                         // stop AD conversion

    adcData -= adMax;                       // subtract the maximum value during sampling
    adcData -= adMin;                       // subtract the minimum value during sampling

    adcData /= ad_test_times;               // average
    adcValue[0] = adcData;
    adcValue[1] = adcData >> 8;
}

//******************************************************************************
// name         : DRV_ADC_ConvertDone()
// description  : ADC conversion completed
// retval       :
// note         :
//******************************************************************************
uint8_t DRV_ADC_ConvertDone(void)
{
    if (ADCSR_EOCF) {
        return 1;                       // if ADC conversion completed, return 1
    } else {
        return 0;                       // if ADC conversion not completed, return 0
    }
}

//******************************************************************************
// name         : DRV_ADC_IRQ()
// description  : ADC IRQ
// retval       :
// note         :
//******************************************************************************
void __interrupt DRV_ADC_IRQ(void)
{
    
}