//******************************************************************************
// file         : DRV_SQ7705_ADC.h
// version      : V1.2 2023/09/01
// description  : ADC header file
// note         : 
//******************************************************************************
#ifndef _DRV_SQ7705_ADC_H__
#define _DRV_SQ7705_ADC_H__

#include "SQ7705_SFR.h"
#include "TypeDefs.h"

#define ad_test_times 16    // average number of times for ADC repeat mode conversion

typedef enum{
    ADC_AIN0 = 0,
    ADC_AIN1,
    ADC_AIN2,
    ADC_AIN3,
    ADC_AIN4,
    ADC_AIN5,
    ADC_AIN6,
    ADC_AIN7,
    ADC_AIN8,
    ADC_AIN9,
    ADC_AIN10,
    ADC_AIN11,
    ADC_AIN12,
    ADC_AIN13,
    ADC_AIN14,
    ADC_AIN15
}ADC_AIN_CHANNEL_LIST;

typedef enum{
    ADC_ADCCKDIV_1 = 0,
    ADC_ADCCKDIV_2,
    ADC_ADCCKDIV_4,
    ADC_ADCCKDIV_8,
    ADC_ADCCKDIV_16,
    ADC_ADCCKDIV_32,
    ADC_ADCCKDIV_64,
    ADC_ADCCKDIV_128,
    ADC_ADCCKDIV_256,
    ADC_ADCCKDIV_512
}ADC_CLOCK_DIV_LIST;

typedef enum{
    ADC_IRFEN_VDD = 1,
    ADC_IRFEN_EXTVDD = 2
}ADC_VDD_REF_LIST;

typedef enum{
    ADC_CONVERT_SINGLE = 1,
    ADC_CONVERT_REPEAT = 3
}ADC_CONVERT_MODE_LIST;

typedef enum{
    ADC_AUTOPD_DISABLE = 0,
    ADC_AUTOPD_ENABLE  = 1
}ADC_AUTO_POWER_DOWN_LIST;

typedef enum{
    ADC_EVENT_ADRS         = 0,
    ADC_EVENT_TMRA00_COUNT = 8,
    ADC_EVENT_TMRA01_COUNT = 9,
    ADC_EVENT_TCA0_COUNT   = 10,
    ADC_EVENT_TCA1_COUNT   = 11,
    ADC_EVENT_EINT7        = 12,
    ADC_EVENT_TMRA0_PWM    = 13,
    ADC_EVENT_TMRA1_PWM    = 14,
    ADC_EVENT_TMRA2_PWM    = 15
}ADC_EVENT_LIST;

typedef enum{
    ADC_IRQ_WHAN_EACH_CONVERT_DONE = 0,
    ADC_IRQ_WHAN_SCAN_FINISHED     = 1
}ADC_IRQ_LIST;

typedef enum{
    ADC_LEVEL_COMPARE_DISABLE,
    ADC_VALUE_LESS_THAN_LOW_LEVEL,
    ADC_VALUE_MORE_THAN_HIGH_LEVEL,
    ADC_VALUE_COMPARE_BOTH
}ADC_LEVEL_COMPARE_SELECT_LIST;

typedef enum{
    ADC_LevelCompareAllChannel,
    ADC_LevelCompareSingleChannel
}ADC_LEVEL_COMPARE_CHANNEL_LIST;

typedef struct{
    ADC_CLOCK_DIV_LIST       sampleRate;
    ADC_VDD_REF_LIST         refVoltage;
    ADC_AUTO_POWER_DOWN_LIST autoPowerDown;
    ADC_CONVERT_MODE_LIST    convertMode;
    ADC_EVENT_LIST           event;
    ADC_IRQ_LIST             irq;
}ADC_INIT_STRUCT;

typedef struct{
    ADC_LEVEL_COMPARE_SELECT_LIST  levelSetting;
    ADC_LEVEL_COMPARE_CHANNEL_LIST singleOrAll;
    ADC_AIN_CHANNEL_LIST           channel;
    uint16_t                       LowLevel;
    uint16_t                       HighLevel;
}ADC_LEVEL_COMPARE_INIT_STRUCT;

//******************************************************************************
// name         : DRV_ADC_Channel_Init()
// description  : set IO as ADC channel
// input param  : - AINx : ADC channel
// retval       : 
// note         : 
//******************************************************************************
void DRV_ADC_Channel_Init(ADC_AIN_CHANNEL_LIST AINx);

//******************************************************************************
// name         : DRV_ADC_ChannelScan_Init()
// description  : initialize ADC scan channel
// input param  : - AINx : ADC channel
// retval       : 
// note         : 
//******************************************************************************
void DRV_ADC_ChannelScan_Init(ADC_AIN_CHANNEL_LIST AINx);

//******************************************************************************
// name         : DRV_ADC_Init()
// description  : ADC initialization
// input param  : ADC_INIT_STRUCT ->sampleRate    : sampling rate
//                                ->refVoltage    : reference voltage
//                                ->autoPowerDown : enable auto power-down
//                                ->convertMode   : convertion mode selection
//                                ->event         : trigger event
//                                ->irq           : interrupt time
// retval       : 
// note         : 
//****************************************************************************** 
void DRV_ADC_Init(ADC_INIT_STRUCT *adcInit);

//******************************************************************************
// name         : DRV_ADC_LevelCompare_Init()
// description  : ADC level comparison initialization
// input param  : ADC_LEVEL_COMPARE_INIT_STRUCT -> levelSetting : ADC level comparison setting 
//                                              -> singleOrAll  : ADC samples
//                                              -> channel      : specific channel
//                                              -> LowLevel     : low level comparison value setting
//                                              -> HighLevel    : high level comparison value setting
// retval       : 
// note         : 
//******************************************************************************
void DRV_ADC_LevelCompare_Init(ADC_LEVEL_COMPARE_INIT_STRUCT *adcLVCMPInit);

//******************************************************************************
// name         : DRV_ADC_ForceStopAndDisable()
// description  : stop and disable ADC
// input param  : 
// retval       : 
// note         : 
//******************************************************************************
void DRV_ADC_ForceStopAndDisable(void);

//******************************************************************************
// name         : DRV_ADC_Convert_Start()
// description  : start AD conversion
// input param  :
// retval       : 
// note         : 
//******************************************************************************
void DRV_ADC_Convert_Start(void);

//******************************************************************************
// name         : DRV_ADC_Read_Value()
// description  : read AD conversion value
// input param  : - AINx : channel selection
// output param : - adcValue : ADC value
// retval       : 
// note         : 
//******************************************************************************
void DRV_ADC_Read_Value(ADC_AIN_CHANNEL_LIST AINx, uint8_t *adcValue);

//******************************************************************************
// name         : DRV_ADC_Convert_Repeat()
// description  : Perform ADC repeat mode conversion and store the conversion value in ADCDRL and ADCDRH and average them
// input param  : - AINx : ADC channel in
// output param : - adcValue : ADC value
// note         : SAIN: select ADC input channel
//                  AIN4        0x4
//                  AIN5        0x5
//                  AIN6        0x6
//                  AIN7        0x7
//                  AIN8        0x8
//                  AIN9        0x9
//                  AIN10       0xA
//
// *this example of the function is demonstrated repeated sampling within 18 times, and maximum 
//  and minimum values in the sampling process are not included in the average consideration
//******************************************************************************
void DRV_ADC_Convert_Repeat(ADC_AIN_CHANNEL_LIST AINx, uint8_t *adcValue);

//******************************************************************************
// name         : DRV_ADC_ConvertDone()
// description  : ADC conversion completed
// retval       :
// note         :
//******************************************************************************
uint8_t DRV_ADC_ConvertDone(void);

//******************************************************************************
// name         : DRV_ADC_IRQ()
// description  : ADC IRQ
// retval       :
// note         :
//******************************************************************************
void __interrupt DRV_ADC_IRQ(void);

#endif