//*************************************************************************************
// file			: SQ7653_UART.c
// version		: V1.0 2020/07/17
// brief		: UART related functions
// note			: UART related functions are gathered in this subroutine
//**************************************************************************************
#include "main.h"
unsigned char rbuf;                     	 

// RT clocks
const unsigned char RTTable[]={
//1Mhz
	0x00,//4800
	0x02,//9600
	0x04,//19200
	0x04,//57600
//2MHz
	0x00,//4800
	0x00,//9600
	0x02,//19200
	0x04,//57600
	0x04,//115200
	0x03,//128000
//4MHz
	0x04,//4800
	0x00,//9600
	0x00,//19200
	0x04,//57600
	0x04,//115200
	0x03,//128000
//8MHz
	0x01,//4800
	0x04,//9600
	0x00,//19200
	0x03,//57600
	0x04,//115200
	0x03,//128000
//12MHz	
	0x04,//4800
	0x00,//9600
	0x00,//19200
	0x00,//57600
	0x02,//115200
	0x03,//128000	
//16MHz
	0x01,//4800
	0x01,//9600
	0x04,//19200
	0x03,//57600
	0x03,//115200
	0x03,//128000
	0x03,//256000
//24MHz
	0x04,//9600
	0x00,//19200
	0x00,//57600
	0x00,//115200
	0x04,//128000
	0x03,//256000
};
//------------------------------------------------------------------------------
// BaudRate
const unsigned char DRTable[]={
//1Mhz
	0x0C,//4800
	0x06,//9600
	0x02,//19200
	0x00,//57600
//2MHz
	0x19,//4800
	0x0C,//9600
	0x06,//19200
	0x01,//57600
	0x00,//115200
	0x00,//128000
//4MHz
	0x30,//4800
	0x19,//9600
	0x0C,//19200
	0x03,//57600
	0x01,//115200
	0x01,//128000
//8MHz
	0x64,//4800
	0x30,//9600
	0x19,//19200
	0x08,//57600
	0x03,//115200
	0x03,//128000
//12MHz
	0x92,//4800
	0x4D,//9600
	0x26,//19200
	0x0C,//57600
	0x06,//115200
	0x05,//128000
//16MHz
	0xC9,//4800
	0x64,//9600
	0x30,//19200
	0x11,//57600
	0x08,//115200
	0x07,//128000
	0x03,//256000
//24M
	0x92,//9600   
	0x4D,//19200
	0x19,//57600
	0x0C,//115200
	0x0A,//128000
	0x05,//256000
};


//*********************************************************************
// name			: UART0_Init()
// brief		: UART0 initialization
// note			: set serial data output in pins and UART related parameters
//*********************************************************************
void UART0_Init(){

//====== UART0 pin configuration initialization ======

/*
	//--TXD0 (P3.7) ; RXD0 (P4.0)-------
	P3CFGCR = 0xF2;                       // set P3.4 to TXD0
	P4CFGCR = 0x82;                       // set P4.0 to RXD0
*/


//====== UART0parameter setting ======
/*
	PCKEN1_UART0 	= 1;                // enable UART0
	UART0CR1_STOPBT = 0;                // set transmit stop bit length to 1 bit
	UART0CR2_STOPBR = 0;                // set receive stop bit length to 1 bit	
	UART0CR1_PE 	= 0;                // no parity

	//UART0CR1_PE 	= 1;                // parity added
	//UART0CR1_EVEN = 1;                // 0:odd-numbered parity 1:even-numbered parity

*/


}


//*********************************************************************
// name			: UART1_Init()
// brief		: UART1 initialization
// note			: set serial data output in pins and UART related parameters
//*********************************************************************
void UART1_Init(){

//====== UART1 pin configuration initialization ======
	
	//--TXD1 (P1.3) ; RXD1 (P1.4)-------  [for this example program]

	P1CFGCR = 0xB3;                      // set P1.3 to TXD1
	P1CFGCR = 0xC3;                      // set P1.4 to RXD1


//====== UART1parameter setting ======
	PCKEN1_UART1 	= 1;                // enable UART1
	UART1CR1_STOPBT = 0;                // set transmit stop bit length to 1 bit
	UART1CR2_STOPBR = 0;                // set receive stop bit length to 1 bit	
	UART1CR1_PE 	= 0;                // no parity
	
	//UART1CR1_PE 	= 1;                // parity added
	//UART1CR1_EVEN = 1;                // 0:odd-numbered parity 1:even-numbered parity

}


//*********************************************************************
// name			: UART2_Init()
// brief		: UART2 initialization
// note			: set serial data output in pins and UART related parameters
//*********************************************************************
void UART2_Init(){

//====== UART2 pin configuration initialization ======
	
/*	
	//--TXD2 (P3.1) ; RXD2 (P3.5)-------
	P3CFGCR = 1;                       // set P3.1 to TXD2
	P3CFGCR = 0;                       // set P3.5 to RXD2     
	
*/	
	

//====== UART2 parameter setting ======
/*
	PCKEN1_UART2 	= 1;                // enable UART2
	UART2CR1_STOPBT = 0;                // set transmit stop bit length to 1 bit
	UART2CR2_STOPBR = 0;                // set receive stop bit length to 1 bit	
	UART2CR1_PE 	= 0;                // no parity
		
	//UART2CR1_PE 	= 1;                // parity added
	//UART2CR1_EVEN = 1;                // 0:odd-numbered parity 1:even-numbered parity
*/


}

//*********************************************************************
// name			: UART0_IntEnable()
// brief		: enable UART0 interrupt
// note			: 
//*********************************************************************
void UART0_IntEnable(){
	__ASM("DI");						// enable all maskable interrupts
	IER_TX0_IRQ = 1;                    // enable TX0 interrupt
	IER_RX0_IRQ = 1;	                // enable RX0 interrupt
	__ASM("EI");						// enable all maskable interrupts
}


//*********************************************************************
// name			: UART1_IntEnable()
// brief		: enable UART1 interrupt
// note			: 
//*********************************************************************
void UART1_IntEnable(){
	__ASM("DI");						// enable all maskable interrupts
	IER_TX1_IRQ = 1;                    // enable TX1 interrupt
	IER_RX1_IRQ = 1;	                // enable RX1 interrupt
	__ASM("EI");						// enable all maskable interrupts
}

//*********************************************************************
// name			: UART2_IntEnable()
// brief		: enable UART2 interrupt
// note			: 
//*********************************************************************
void UART2_IntEnable(){
	__ASM("DI");						// enable all maskable interrupts
	IER_TX2_IRQ = 1;                    // enable TX2 interrupt
	IER_RX2_IRQ = 1;	                // enable RX2 interrupt
	__ASM("EI");						// enable all maskable interrupts
}




//*********************************************************************
// name			: UART0_SetBaudRate()  / UART1_SetBaudRate() / UART2_SetBaudRate()
// brief		: set Baud rate
// note			:          
//           baudSel: 
// 				SYSCLK@1MHZ
// 					SYSCLK_1MHZ_BAUD_4800		0
// 					SYSCLK_1MHZ_BAUD_9600		1
// 					SYSCLK_1MHZ_BAUD_19200		2
// 					SYSCLK_1MHZ_BAUD_57600		3
//
// 				SYSCLK@2MHZ
// 					SYSCLK_2MHZ_BAUD_4800		4
// 					SYSCLK_2MHZ_BAUD_9600		5
// 					SYSCLK_2MHZ_BAUD_19200		6
// 					SYSCLK_2MHZ_BAUD_57600		7
// 					SYSCLK_2MHZ_BAUD_115200		8
// 					SYSCLK_2MHZ_BAUD_128000		9
//
// 				SYSCLK@4MHZ
// 					SYSCLK_4MHZ_BAUD_4800		10
// 					SYSCLK_4MHZ_BAUD_9600		11
// 					SYSCLK_4MHZ_BAUD_19200		12
// 					SYSCLK_4MHZ_BAUD_57600		13
// 					SYSCLK_4MHZ_BAUD_115200		14
// 					SYSCLK_4MHZ_BAUD_128000		15
//
// 				SYSCLK@8MHZ
// 					SYSCLK_8MHZ_BAUD_4800		16
// 					SYSCLK_8MHZ_BAUD_9600		17
// 					SYSCLK_8MHZ_BAUD_19200		18
// 					SYSCLK_8MHZ_BAUD_57600		19
// 					SYSCLK_8MHZ_BAUD_115200		20
// 					SYSCLK_8MHZ_BAUD_128000		21
//
// 				SYSCLK@12MHZ
// 					SYSCLK_12MHZ_BAUD_4800		22
// 					SYSCLK_12MHZ_BAUD_9600		23
// 					SYSCLK_12MHZ_BAUD_19200		24
// 					SYSCLK_12MHZ_BAUD_57600		25
// 					SYSCLK_12MHZ_BAUD_115200	26
// 					SYSCLK_12MHZ_BAUD_128000	27
//
// 				SYSCLK@16MHZ
// 					SYSCLK_16MHZ_BAUD_4800		28
// 					SYSCLK_16MHZ_BAUD_9600		29
// 					SYSCLK_16MHZ_BAUD_19200		30
// 					SYSCLK_16MHZ_BAUD_57600		31
// 					SYSCLK_16MHZ_BAUD_115200	32
// 					SYSCLK_16MHZ_BAUD_128000	33
// 					SYSCLK_16MHZ_BAUD_256000	34
//
// 				SYSCLK@24MHZ
// 					SYSCLK_24MHZ_BAUD_9600		35
// 					SYSCLK_24MHZ_BAUD_19200		36
// 					SYSCLK_24MHZ_BAUD_57600		37
// 					SYSCLK_24MHZ_BAUD_115200	38
// 					SYSCLK_24MHZ_BAUD_128000	39
// 					SYSCLK_24MHZ_BAUD_256000	40             
//*********************************************************************

void UART0_SetBaudRate(unsigned char baudSel){

//====== UART0 ======
// select the number of RT clocks
	UART0CR2_RTSEL = RTTable[baudSel];
	UART0_Stop();						// set UART0CR1<RXE> and UART0CR1<TXE> to "0" before changing UART0DR
// select Baud rate	
	UART0DR  =  DRTable[baudSel];

}
//--------------------------------------------------------------

void UART1_SetBaudRate(unsigned char baudSel){
	
//====== use UART1 in this example ====== 

//UART1CR2 = (UART1CR2 & 0x07) | ( RTTable[baudSel] <<3);
//UART1DR  =  DRTable[baudSel];

// select the number of RT clocks
	UART1CR2_RTSEL = RTTable[baudSel];
	UART1_Stop();						// set UART1CR1<RXE> and UART1CR1<TXE> to "0" before changing UART1DR
// select Baud rate	
	UART1DR  =  DRTable[baudSel];
	
}
//--------------------------------------------------------------
void UART2_SetBaudRate(unsigned char baudSel){

//====== UART2 ======
// select the number of RT clocks
	UART2CR2_RTSEL = RTTable[baudSel];
	UART2_Stop();						// set UART2CR1<RXE> and UART2CR1<TXE> to "0" before changing UART2DR
// select Baud rate	
	UART2DR  =  DRTable[baudSel];
	
}



//*********************************************************************
// name			: UART0_Start()
// brief		: start UART0
// note			: enable TX0 and RX0
//*********************************************************************
void UART0_Start(){
    UART0CR1_TXE = 1;                   // enable TX0 
	UART0CR1_RXE = 1;                   // enable RX0
}


//*********************************************************************
// name			: UART0_Stop()
// brief		: stop UART0
// note			: disable TX0 and RX0
//*********************************************************************
void UART0_Stop(){
	UART0CR1_TXE = 0;                   // disable TX0
	UART0CR1_RXE = 0;                   // disable RX0
}



//*********************************************************************
// name			: UART1_Start()
// brief		: start UART1
// note			: enable TX1 and RX1
//*********************************************************************
void UART1_Start(){
    UART1CR1_TXE = 1;                   // enable TX1 
	UART1CR1_RXE = 1;                   // enable RX1
}


//*********************************************************************
// name			: UART1_Stop()
// brief		: stop UART1
// note			: disable TX1 and RX1
//*********************************************************************
void UART1_Stop(){
	UART1CR1_TXE = 0;                   // disable TX1
	UART1CR1_RXE = 0;                   // disable RX1
}

//*********************************************************************
// name			: UART2_Start()
// brief		: start UART2
// note			: enable TX2 and RX2
//*********************************************************************
void UART2_Start(){
    UART2CR1_TXE = 1;                   // enable TX2 
	UART2CR1_RXE = 1;                   // enable RX2
}


//*********************************************************************
// name			: UART2_Stop()
// brief		: stop UART2
// note			: disable TX2 and RX2
//*********************************************************************
void UART2_Stop(){
	UART2CR1_TXE = 0;                   // disable TX2
	UART2CR1_RXE = 0;                   // disable RX2
}



//*********************************************************************
// name			: UART0_TX_OneByte()
// brief		: UART0 TX tramsmits one byte
// note			:
//            data: data to be transmitted by TX
//*********************************************************************
void UART0_TX_OneByte(unsigned char data){
	
	while(UART0SR_TBFL == 1){			// wait for the send buffer to be empty before writing the data to be sent again
		CLR_WDT;						// clear the watchdog timer 
    }
	TD0BUF = data;						// write the data to be sent
    
	while(UART0SR_TBSY ==1){			// 1:transmission in progress , 0:before transmission or waiting for the end of transmission work
        CLR_WDT;						// clear the watchdog timer 
    }
}


//*********************************************************************
// name			: UART0_RX_OneByte()
// brief		: UART0 RX receives a single byte
// note			:
//            return : data to be received by RX
//*********************************************************************
unsigned char UART0_RX_OneByte(){	
	
    while(!UART0SR_RBFL);               // wait for UART0 RD0BUF to receive data
	return RD0BUF;                      // read out the received RD0BUF

}

//*********************************************************************
// name			: UART1_TX_OneByte()
// brief		: UART1 TX tramsmits one byte
// note			:
//            data: data to be transmitted by TX
//*********************************************************************
void UART1_TX_OneByte(unsigned char data){

	while(UART1SR_TBFL == 1){			 // wait for the send buffer to be empty before writing the data to be sent again
		CLR_WDT;						 // clear the watchdog timer 
    }
	TD1BUF = data;						 // write the data to be sent
    
	while(UART1SR_TBSY ==1){			 // 1:transmission in progress , 0:before transmission or waiting for the end of transmission work
        CLR_WDT;						 // clear the watchdog timer 
    }
}


//*********************************************************************
// name			: UART1_RX_OneByte()
// brief		: UART1 RX receives a single byte
// note			:
//            return : data to be received by RX
//*********************************************************************
unsigned char UART1_RX_OneByte(){	
    
    while(!UART1SR_RBFL);               // wait for UART1 RD1BUF to receive data
	return RD1BUF;                       // read out the received RD1BUF

}


//*********************************************************************
// name			: UART2_TX_OneByte()
// brief		: UART2 TX tramsmits one byte
// note			:
//            data: data to be transmitted by TX
//*********************************************************************
void UART2_TX_OneByte(unsigned char data){

	while(UART2SR_TBFL == 1){			 // wait for the send buffer to be empty before writing the data to be sent again      
		CLR_WDT;                     	 // clear the watchdog timer 
    }
	TD2BUF = data;						 // write the data to be sent
    
	while(UART2SR_TBSY ==1){			 // 1:transmission in progress , 0:before transmission or waiting for the end of transmission work
        CLR_WDT;                         // clear the watchdog timer 
    }
}

//*********************************************************************
// name			: UART2_RX_OneByte()
// brief		: UART2 RX receives a single byte
// param		: none
// retval		: data to be received by RX
// note			: only for UART2
//*********************************************************************
unsigned char UART2_RX_OneByte(){

    while(!UART2SR_RBFL);               // wait for UART2 RD2BUF to receive data
	return RD2BUF;                      // read out the received RD2BUF

}

//*********************************************************************
// name			: UART_ReadState()
// brief		: get UART status
// note			:
//           uart: UART_0, UART_1, UART_2
//           return UART status
//*********************************************************************

unsigned char UART_ReadState(unsigned char uart){
	if(uart==UART_0){
		return UART0SR;
	}
	else if(uart==UART_1){
		return UART1SR;
	}
	else if(uart==UART_2){
		return UART2SR;
	}
	else
		return 0;
}

//*********************************************************************
// name			: UART_TX_OneByte()
// brief		: UART TX tramsmits one byte
// note			:
//           uart: UART_0, UART_1, UART_2
//           data: data to be transmitted by TX
//*********************************************************************
void UART_TX_OneByte(unsigned char uart,unsigned char data){
	
	if(uart==UART_0){
		while(UART0SR_TBFL);			// wait for the send buffer to be empty before writing the data to be sent again
		TD0BUF = data;					// write the data to be sent
		while(UART0SR_TBFL);
	}
	else if(uart==UART_1){
		while(UART1SR_TBFL);			// wait for the send buffer to be empty before writing the data to be sent again
		TD1BUF = data;					// write the data to be sent
		while(UART1SR_TBFL);
	}
	else if(uart==UART_2){
		while(UART2SR_TBFL);			// wait for the send buffer to be empty before writing the data to be sent again
		TD2BUF = data;					// write the data to be sent
		while(UART2SR_TBFL);
	}	
	
}


//*********************************************************************
// name			: UART_Send()
// brief		: UART TX send bytes
// note			: transmit more bytes of data
//			  uart: UART_0, UART_1, UART_2
//           
//*********************************************************************
void UART_Send(unsigned char uart,unsigned char* buf, unsigned char len){
	
	unsigned char i;
	for(i=0;i<len;i++){
		if(uart==UART_0){
			while(UART0SR_TBFL);        // wait for UART0 TD0BUF empty
			TD0BUF = buf[i];            // write the data to be sent
		}
		else if(uart==UART_1){
			while(UART1SR_TBFL);        // wait for UART1 TD1BUF empty
			TD1BUF = buf[i];            // write the data to be sent
		}
		else if(uart==UART_2){
			while(UART2SR_TBFL);        // wait for UART2 TD2BUF empty
			TD2BUF = buf[i];            // write the data to be sent
		}

	}
}

//*********************************************************************
// name			: UART_Nibble2Ascii()
// brief		: nibble to ASCII
// note			:           
//*********************************************************************
unsigned char UART_Nibble2Ascii(unsigned char anibble)
{
	unsigned char	ascii_out = anibble;

	if(anibble > 9)						// If req ASCII A-F then add 7(hex)
	{	
		ascii_out = ascii_out + 0x07;
	}

	// Add offset to convert to Ascii
	ascii_out = ascii_out + 0x30;

	return(ascii_out);
}


//*********************************************************************
// name			: UART_PutByte()
// brief		: transmit data by nibble with ASCII format
// note			: 
//			  uart: UART_0, UART_1, UART_2
//*********************************************************************
void UART_PutByte(unsigned char uart,unsigned char abyte)
{
	unsigned char  temp1 = 0, temp2 = 0;	

	temp1 = (abyte >> 4) & 0x0F;		// get the high nibble
	temp2 = UART_Nibble2Ascii(temp1);	// high nibble to ASCII code
	UART_TX_OneByte(uart,temp2);		// TX transfer

	temp1 = abyte & 0x0F;				// get the low nibble
	temp2 = UART_Nibble2Ascii(temp1);	// low nibble to ASCII code
	UART_TX_OneByte(uart,temp2);		// TX transfer
}



//*********************************************************************
// name			: UART_PutDecNumber()
// brief		: 
// note			: ***Input value range : 0~9999 
//*********************************************************************
void UART_PutDecNumber(unsigned int value)
{
	unsigned char print_buf[4];  
	 
    if(value > 9999)
       value = 9999; 		
	
	print_buf[0] = ((value%10000)/1000)+'0';
	print_buf[1] = ((value%1000)/100)+'0';
	print_buf[2] = ((value%100)/10)+'0';
	print_buf[3] = ((value%10))+'0';
	 
	UART_Send(UART_1,print_buf,4);	
}

//*********************************************************************
// name			: UART_PutCrlf()
// brief		: transmit line break characters
// note			: 
//*********************************************************************
void UART_PutCrlf(void)
{

/*	
    UART_TX_OneByte(UART_0,'\r');
	UART_TX_OneByte(UART_0,'\n');
*/	

	UART_TX_OneByte(UART_1,'\r');
	UART_TX_OneByte(UART_1,'\n');

/*	
    UART_TX_OneByte(UART_2,'\r');
	UART_TX_OneByte(UART_2,'\n');
*/	
	
}


//*********************************************************************
// name			: UART_SendCString()
// brief		: TX transmits string
// note			:  
//*********************************************************************
void UART_SendCString(const char *pstr)
{
	while(*pstr != 0)
	{	
	    if(*pstr == '\n')
		   	UART_PutCrlf();
		else
		
			//UART_TX_OneByte(UART_0,*pstr);
		    UART_TX_OneByte(UART_1,*pstr);
		    //UART_TX_OneByte(UART_2,*pstr);

		pstr++;
	}
}



//*********************************************************************
// name			: __interrupt RX0_IRQ()
// brief		: RX0 interrupt service routine
// note			:
//*********************************************************************
void __interrupt RX0_IRQ(void)
{
	__asm("NOP");						// no operation
	

}


//*********************************************************************
// name			: __interrupt RX1_IRQ()
// brief		: RX1 interrupt service routine
// note			: if the received data matches the transmitted data, then light on the corresponding LED
//*********************************************************************
void __interrupt RX1_IRQ(void)
{
	__asm("NOP");						// no operation

	rbuf = UART1_RX_OneByte();          // readout data
	if(rbuf == 0x55){					
		LED1_out = 0;					// if received data is 0x55, light on LED1(P0.0)
        LED2_out = 1;
        LED3_out = 1;		
	}
	else if(rbuf == 0xAA){				
		LED1_out = 1;					// if received data is 0xAA, light on LED2(P0.1)
	    LED2_out = 0;
	    LED3_out = 1;
	}
	else if(rbuf == 0x76){				
		LED1_out = 1;					// if received data is 0x76, light on LED3(P0.2)
	    LED2_out = 1;
	    LED3_out = 0;
	
	}		

}


//*********************************************************************
// name			: __interrupt RX2_IRQ()
// brief		: RX2 interrupt service routine
// note			:
//*********************************************************************
void __interrupt RX2_IRQ(void)
{
	__asm("NOP");						// no operation

}



//*********************************************************************
// name			: __interrupt TX0_IRQ()
// brief		: TX0 interrupt service routine
// note			:
//*********************************************************************
void __interrupt TX0_IRQ(void)
{
	__asm("NOP");						// no operation

	
}



//*********************************************************************
// name			: __interrupt TX1_IRQ()
// brief		: TX1 interrupt service routine
// note			:
//*********************************************************************
void __interrupt TX1_IRQ(void)
{
	__asm("NOP");						// no operation

}



//*********************************************************************
// name			: __interrupt TX2_IRQ()
// brief		: TX2 interrupt service routine
// note			:
//*********************************************************************
void __interrupt TX2_IRQ(void)
{
	__asm("NOP");						// no operation

}
