//*************************************************************************************
// file			: SQ7653_I2C_Slave.c
// version		: V1.0 2020/07/17
// brief		: I2C related functions
// note			: I2C related functions are gathered in this subroutine
//**************************************************************************************
#include "main.h"


//*********************************************************************
// name			: I2C0_Init()
// brief		: I2C0 initialization
// note			: set I2C0 pins and transfer mode		  
//*********************************************************************
void I2C0_Init(){
	
//====== I2C0 pin configuration initialization ======
//--SDA0 (P0.7) ; SCL0 (P0.6)-------

	P0CFGCR = 0xF2;  				    // set P0.7 to SDA0	1 111 0010
	P0CFGCR = 0xE2;					    // set P0.6 to SCL0	1 110 0010

//====== Function Enable ======
	PCKEN2_I2C0 = 1;                    // this must be enabled first, the relevant staging settings later to be useful,I2C0 Enable
    
//====== parameter setting ======    
    
	SBI0CR1_ACK = 1;                    // generate the clocks for an acknowledge signal and an interrupt request when the data transfer is finished (acknowledgement mode)
	SBI0CR1_BC = 0x00;                  // number of clocks for data transfer = 8 + 1 = 9
	SBI0CR1_SCK = SCK_HI_135_LO_138;    // HIGH and LOW periods of the serial clock in the master mode @fsysclk=24MHz, fscl is about 88 kHz

	//---set slave address in slave mode
	I2C0AR = SLAVE_ADDR << 1;			// set slave addressI2C0AR[7:1]
	
	//---set SBICR2<PIN> , <SBIM> in slave mode
	SBI0CR2 = 0x18;						// Slave mode 0001 1000

}


//SBI0CR2
//-----bit7 | bit6 | bit5 | bit4  |  bit3  | bit2 | bit1~bit0 ---------
//---  <MST>, <TRX>, <BB>, <PIN> and <SBIM>  <NA>   <SWRST>   ---------
//---  <MST>   1 : Master
//             0 : Slave
//---  <TRX>   1 : Transmitter   ; 0: Receiver
//---  <BB>    1 : Generate the start condition (when MST, TRX and PIN are "1")
//             0 : Generate the stop condition (when MST, TRX and PIN are "1")
//---  <PIN>   1 : Cancel interrupt service request
//
//---  <SBIM>  1 : 1 : Serial bus interface mode ;  0 : Port mode
//---  <SWRST> The software reset starts by first writing "10" and next writing "01"  
//----------------------------------------------------------------------



//*********************************************************************
// name			: I2C0_IntEnable()
// brief		: enable I2C0 interrupt
// note			: **
//*********************************************************************
void I2C0_IntEnable(void){
	__ASM("DI");						// enable all maskable interrupts
    
	IFR_I2C0_IRQ = 0;                   // clear interrupt flag
	IER_I2C0_IRQ = 1;	                // enable I2C0 interrupt(Enable)
    
	__ASM("EI");						// enable all maskable interrupts
}


//*********************************************************************
// name			: I2C0_IntDisable() 
// brief		: disable I2C0 interrupt
// note			: **
//*********************************************************************
void I2C0_IntDisable(void){
	__ASM("DI");						// enable all maskable interrupts
    
	IFR_I2C0_IRQ = 0;                   // clear interrupt flag
	IER_I2C0_IRQ = 0;	                // disable I2C0 interrupt
    
	__ASM("EI");						// enable all maskable interrupts	
}




//*********************************************************************
// name			: I2C_AckEnable()
// brief		: generate I2C 1-bit ACK signal or not
// note			: 
//            enable : 1 enable
//				       0 disable
//*********************************************************************
void I2C_AckEnable(unsigned char enable){
	SBI0CR1_ACK = enable;				// generate I2C0 1-bit ACK signal or not

}




//*********************************************************************
// name			: IsAck()
// brief		: check ACK or NACK
// note			: 
//            return : 1 : NACK
//					   0 : ACK
//*********************************************************************
unsigned char IsAck(){
	if(SBI0SR_LRB ==0){
	return SBI0SR_LRB;
	
	}
	else {								// SBI0SR_LRB =1,  NACK
	return I2C_NACK;
	}
	
	
}



// //*********************************************************************
// // name			: NACK_Stop_I2C()
// // brief		: check ACK or NACK, if NACK, I2C generate the stop condition
// // note			: 
// //*********************************************************************
// void NACK_Stop_I2C(){
// 	if(IsAck()==I2C_NACK){
// 	I2C_Stop();							// I2C generate the stop condition
// 	return 0;
// 	
// 	}
// 	
// 
// }



//*********************************************************************
// name			: I2C_BufCount()
// brief		: number of data bits
// note			: 
//            bufcount : 
//                   BC_8bit      0x00
//                   BC_1bit      0x01
//                   BC_2bit      0x02
//                   BC_3bit      0x03
//                   BC_4bit      0x04
//                   BC_5bit      0x05
//                   BC_6bit      0x06
//                   BC_7bit      0x07 
//*********************************************************************
void I2C_BufCount(unsigned char bufcount){
	SBI0CR1_BC = bufcount;


}


//*********************************************************************
// name			: I2C_ReadState()
// brief		: read I2C status
// note			: 
//*********************************************************************
unsigned char I2C_ReadState(){
	return SBI0SR;						// read I2C0 status


}


//*********************************************************************
// name			: I2C_SetBufvalue()
// brief		: set I2C buffer data
// note			: 
//            data:  buffer data
//*********************************************************************
void I2C_SetBufvalue(unsigned char data){
	SBI0DBR = data;


}



//*********************************************************************
// name			: I2C_ReadBufvalue()
// brief		: read I2C buffer data
// note			: 
//            return: buffer data
//*********************************************************************
unsigned char I2C_ReadBufvalue(){
	return SBI0DBR;						// return I2C0 data buffer register


}


//*********************************************************************
// name			: I2C_FinishSlaveTx() [Slave mode]
// brief		: I2C Slave transfer completion and release the I2C bus
// note			: 
//*********************************************************************
void I2C_FinishSlaveTx(){
	SBI0CR2 = 0x58;
	SBI0CR2 = 0x18;
	
	
}


//*********************************************************************
// name			: I2C_Reset()
// brief		: I2C software reset. reset SBI0CR1,I2C0AR,SBI0CR2, not reset SBI0CR2<SBIM>
// note			: software reset I2C0
//*********************************************************************
void I2C_Reset(){
	SBI0CR2 = 0x1A;						// first write SWRST[1:0]=0x02
	SBI0CR2 = 0x19;						// then write [1:0]=0x01

}


//*********************************************************************
// name			: __interrupt I2C0_IRQ()
// brief		: I2C0 interrupt service routine
// note			:
//*********************************************************************
void __interrupt I2C0_IRQ(void)
{
	__asm("NOP");						// no operation

	I2C_Slave_Int(&i2c_slave);
}



//*********************************************************************
// name			: I2C_Slave_Int()
// brief		: 
// note			: 
//*********************************************************************
void I2C_Slave_Int(I2C_SlavePacket* pi2c){
	unsigned char tmp;
	if(I2C_ReadState()&SBISR_TX){
		// transmit
		if(I2C_ReadState()&SBISR_AAS){	  	  // check slave address
			// match slave address
			pi2c->status = I2C_Slave_read;
			// check command
			if(pi2c->cmd == 0x09){			  // if received command is 0x09
				LED1_out = 0;
				
				pi2c->buf[0] = 0x12;
				pi2c->buf[1] = 0x34;
				pi2c->len = 2;                // set transmit data length to 2
			}
			else if(pi2c->cmd == 0x0A){	  // if received command is 0x0A
			    LED2_out = 0;
			
				pi2c->buf[0] = 0x55;
				pi2c->buf[1] = 0xAA;
				pi2c->len = 2;                // set transmit data length to 2
			}
			else{							  // not supprt command
				pi2c->len = 0;                // set transmit data length to 0
			}
			
			pi2c->pointer= 0;
			if(pi2c->pointer >= pi2c->len){   // check data need to be transmitted or not
				I2C_SetBufvalue(0xFF);    	  // transmit dummy data
			}
			else{
				I2C_SetBufvalue(pi2c->buf[pi2c->pointer]);	
			}				
		}
		else{
			if(I2C_ReadState()&SBISR_LRB){
				I2C_SetBufvalue(0x00);		      // transmit dummy data
				I2C_FinishSlaveTx();
			}
			else{
				if(pi2c->pointer >= pi2c->len){
					I2C_SetBufvalue(0xFF);
				}
				else{
				    I2C_SetBufvalue(pi2c->buf[pi2c->pointer]);
				}
			}			
		}
		pi2c->pointer++;
	}
	else{
		// receive
		if(I2C_ReadState()&SBISR_AAS){
			// match slave address
			pi2c->pointer = 0;
			pi2c->status = I2C_Slave_adr;
			tmp = I2C_ReadBufvalue();
			I2C_SetBufvalue(0x00);		         // transmit dummy data						
		}
		else{
		    if(pi2c->status == I2C_Slave_adr){
                pi2c->cmd = I2C_ReadBufvalue();
				pi2c->status = I2C_Slave_cmd;
				I2C_SetBufvalue(0x00);		 	 // transmit dummy data
			}
			else{
				pi2c->buf[pi2c->pointer] = I2C_ReadBufvalue();
				pi2c->pointer++;
				I2C_SetBufvalue(0x00);		 	 // transmit dummy data
			}
		}
	}
}


