//*************************************************************************************
// file			: SQ7653_I2C_Master.c
// version		: V1.0 2020/07/17
// brief		: I2C related functions
// note			: I2C related functions are gathered in this subroutine
//**************************************************************************************
#include "main.h"

unsigned char I2C0_Init_SBI0CR1_Set_Value;

//*********************************************************************
// name			: I2C0_Init()
// brief		: I2C0 initialization
// note			: set I2C0 pins and transfer mode
//*********************************************************************
void I2C0_Init(){
	
//====== I2C0 pin configuration initialization ======
//--SDA0 (P0.7) ; SCL0 (P0.6)-------

	P0CFGCR = 0xF3;  				    // set P0.7 to SDA0	1 111 0011
	P0CFGCR = 0xE3;					    // set P0.6 to SCL0	1 110 0011

//====== Function Enable ======
	PCKEN2_I2C0 = 1;                    // this must be enabled first, the relevant staging settings later to be useful,I2C0 Enable

//====== parameter setting ======
// SBI0CR2_SBIM = 1;

// SBICR1_ACK=1
// SBICR1_SCK=0x06, @fsysclk=24MHz, fscl is about 88 kHz
    SBI0CR1 = 0x16;  

/*	
	// SBI0CR1_ACK = 1;                 // generate the clocks for an acknowledge signal and an interrupt request when the data transfer is finished (acknowledgement mode)
	// SBI0CR1_BC = 0x00;               // number of clocks for data transfer = 8 + 1 = 9
	// SBI0CR1_SCK = SCK_HI_135_LO_138; // HIGH and LOW periods of the serial clock in the master mode
*/
    
    
    //set SBI0CR2<MST> , <PIN> , <SBIM>
    SBI0CR2 = 0x18;						// 0001 1000
    
/*
	//SBI0CR2 = SBICR2_PIN|SBICR2_SBIM;
*/

	// IFR_I2C0_IRQ = 0;                // clear interrupt flag                      
	// IER_I2C0_IRQ = 0;                // disable I2C0 interrupt

}


//SBI0CR2
//-----bit7 | bit6 | bit5 | bit4  |  bit3  | bit2 | bit1~bit0 ---------
//---  <MST>, <TRX>, <BB>, <PIN> and <SBIM>  <NA>   <SWRST>   ---------
//---  <MST>   1 : Master
//             0 : Slave
//---  <TRX>   1 : Transmitter   ; 0: Receiver
//---  <BB>    1 : Generate the start condition (when MST, TRX and PIN are "1")
//             0 : Generate the stop condition (when MST, TRX and PIN are "1")
//---  <PIN>   1 : Cancel interrupt service request
//
//---  <SBIM>  1 : 1 : Serial bus interface mode ;  0 : Port mode
//---  <SWRST> The software reset starts by first writing "10" and next writing "01"  
//----------------------------------------------------------------------	



//*********************************************************************
// name			: I2C0_IntEnable()
// brief		: enable I2C0 interrupt
// note			: **
//*********************************************************************
void I2C0_IntEnable(void){
	__ASM("DI");						// enable all maskable interrupts
    
	IFR_I2C0_IRQ = 0;                   // clear interrupt flag
	IER_I2C0_IRQ = 1;	                // enable I2C0 interrupt(Enable)
    
	__ASM("EI");						// enable all maskable interrupts
}


//*********************************************************************
// name			: I2C0_IntDisable() 
// brief		: disable I2C0 interrupt
// note			: **
//*********************************************************************
void I2C0_IntDisable(void){
	__ASM("DI");						// enable all maskable interrupts
    
	IFR_I2C0_IRQ = 0;                   // clear interrupt flag
	IER_I2C0_IRQ = 0;	                // disable I2C0 interrupt
    
	__ASM("EI");						// enable all maskable interrupts	
}


//*********************************************************************
// name			: I2C_Start() [Master mode]
// brief		: I2C generate the start condition
// note			: 
//            adr : slave address
//*********************************************************************
void I2C_Start(unsigned char adr){
    unsigned char tWait_I2C0 = 0;

	while( SBI0SR_BB == 1 ){}			// SBI0SR.BB: Bus status confirms is free ; 1 :busy  0: free

	SBI0CR1 = I2C0_Init_SBI0CR1_Set_Value;	
    
    //****** Write data to the registers before the start condition is generated ******	
 	SBI0DBR = adr;						// set slave address
    
	SBI0CR2 = 0xf8;						// Write "1" to SBI0CR2<MST>, <TRX>, <BB>, <PIN>  and <SBIM>to "1"
										// 1111 1000
										// set I2C to master mode, transmitter, generating start condition, cancel interrupt service request and serial bus interface mode
/*		
    // SBI0CR2 = SBICR2_MST|SBICR2_TX|SBICR2_BB|SBICR2_PIN|SBICR2_SBIM;
*/

	
 	while(SBI0SR_PIN){};             	// wait for releasing interrupt service request  @SBI0SR_PIN
 		
 	/*
	// --- check last  received bit <LRB> is 1---
		while(SBI0SR_LRB == 1){					
				tWait_I2C0++;			// no response and increase counting
				if(tWait_I2C0>250){		// if there is no response for too long, the loop will be broken		
					I2C_Stop(i2c);		// I2C stop -> release SCK
					break;
				}	
		}
    */

	
}

//SBI0CR2
//-----bit7 | bit6 | bit5 | bit4  |  bit3  | bit2 | bit1~bit0 ---------
//---  <MST>, <TRX>, <BB>, <PIN> and <SBIM>  <NA>   <SWRST>   ---------
//---  <MST>   1 : Master
//             0 : Slave
//---  <TRX>   1 : Transmitter   ; 0: Receiver
//---  <BB>    1 : Generate the start condition (when MST, TRX and PIN are "1")
//             0 : Generate the stop condition (when MST, TRX and PIN are "1")
//---  <PIN>   1 : Cancel interrupt service request
//
//---  <SBIM>  1 : 1 : Serial bus interface mode ;  0 : Port mode
//---  <SWRST> The software reset starts by first writing "10" and next writing "01"  
//----------------------------------------------------------------------




//*********************************************************************
// name			: I2C_ReStart() [Master mode]
// brief		: I2C generate the restart condition
// note			: 
//            adr : slave address
//*********************************************************************
void I2C_ReStart(unsigned char adr){

	SBI0CR2 = 0x18;						// 0001 1000
										// set I2C to master mode, transmitter, generating start condition, cancel interrupt service request and serial bus interface mode
/*                                        
	// SBI0CR2 = SBICR2_PIN|SBICR2_SBIM;
*/

	while(SBI0SR_BB == 1);				// SBI0SR_BB: Bus status confirms is free ; 1 :busy  0: free	
	while(SBI0SR_LRB == 0);				// wait <LRB>=1
	I2C_Start(adr);                 	// transmit the start condition and slave address again

}


//SBI0CR2
//-----bit7 | bit6 | bit5 | bit4  |  bit3  | bit2 | bit1~bit0 ---------
//---  <MST>, <TRX>, <BB>, <PIN> and <SBIM>  <NA>   <SWRST>   ---------
//---  <MST>   1 : Master
//             0 : Slave
//---  <TRX>   1 : Transmitter   ; 0: Receiver
//---  <BB>    1 : Generate the start condition (when MST, TRX and PIN are "1")
//             0 : Generate the stop condition (when MST, TRX and PIN are "1")
//---  <PIN>   1 : Cancel interrupt service request
//
//---  <SBIM>  1 : 1 : Serial bus interface mode ;  0 : Port mode
//---  <SWRST> The software reset starts by first writing "10" and next writing "01"  
//---------------------------------------------------------------------- 



//*********************************************************************
// name			: I2C_Stop() [Master mode]
// brief		: I2C generate the stop condition
// note			: 
//*********************************************************************
void I2C_Stop(void){
	SBI0CR2 = 0xd8;			   			// Set SBI0CR2<MST>, <TRX> and <PIN> to "1" and SBI0CR2<BB> to "0"
										// 1101 1000
										// set I2C0 to master mode, transmitter, generating stop condition, cancel interrupt service request and port mode
/*
    // SBI0CR2 = SBICR2_MST|SBICR2_TX|SBICR2_PIN|SBICR2_SBIM;
*/

	while(SBI0SR_BB);              		// SBI0SR.BB: Bus status confirms is free ; 1 :busy  0: free
}


//*********************************************************************
// name			: I2C_TXOneByte() [Master mode]
// brief		: I2C transmits a single byte data
// note			: 
//            TxBuf: data to be transmit
//*********************************************************************
void I2C_TXOneByte(unsigned char TxBuf){
	SBI0DBR = TxBuf;
	SBI0CR2 = SBICR2_MST|SBICR2_TX|SBICR2_BB|SBICR2_PIN|SBICR2_SBIM;
	while(SBI0CR2_PIN);             	// wait for tranfer completion


}


//*********************************************************************
// name			: I2C_RXOneByte() [Master mode]
// brief		: I2C receive a single byte data
// note			: 
//            return: received data
//*********************************************************************
unsigned char I2C_RXOneByte(){
	SBI0CR2 = SBICR2_MST|SBICR2_RX|SBICR2_BB|SBICR2_PIN|SBICR2_SBIM;
	SBI0DBR = 0x00;                 	// transmit dummy datafer

	while(SBI0CR2_PIN == 1);
	return SBI0DBR;					    // return I2C0 data buffer register
	
}




//*********************************************************************
// name			: I2C_AckEnable()
// brief		: generate I2C 1-bit ACK signal or not
// note			: 
//            enable : 1 enable
//				       0 disable
//*********************************************************************
void I2C_AckEnable(unsigned char enable){
	SBI0CR1_ACK = enable;				// generate I2C0 1-bit ACK signal or not

}




//*********************************************************************
// name			: IsAck()
// brief		: check ACK or NACK
// note			: 
//            return : 1 : NACK
//					   0 : ACK
//*********************************************************************
unsigned char IsAck(){
	if(SBI0SR_LRB ==0){
	return SBI0SR_LRB;
	
	}
	else {								// SBI0SR_LRB =1,  NACK
	return I2C_NACK;
	}
	
	
}



//*********************************************************************
// name			: NACK_Stop_I2C()
// brief		: check ACK or NACK, if NACK, I2C generate the stop condition
// note			: 
//*********************************************************************
unsigned char NACK_Stop_I2C(){
	if(IsAck()==I2C_NACK){
	I2C_Stop();							// I2C generate the stop condition
	return 0;
	
	}
	

}



//*********************************************************************
// name			: I2C_BufCount()
// brief		: number of data bits
// note			: 
//            bufcount : 
//                   BC_8bit      0x00
//                   BC_1bit      0x01
//                   BC_2bit      0x02
//                   BC_3bit      0x03
//                   BC_4bit      0x04
//                   BC_5bit      0x05
//                   BC_6bit      0x06
//                   BC_7bit      0x07 
//*********************************************************************
void I2C_BufCount(unsigned char bufcount){
	SBI0CR1_BC = bufcount;


}


//*********************************************************************
// name			: I2C_ReadState()
// brief		: read I2C status
// note			: 
//*********************************************************************
unsigned char I2C_ReadState(){
	return SBI0SR;						// read I2C0 status


}


//*********************************************************************
// name			: I2C_SetBufvalue()
// brief		: set I2C buffer data
// note			: 
//            data:  buffer data
//*********************************************************************
void I2C_SetBufvalue(unsigned char data){
	SBI0DBR = data;


}



//*********************************************************************
// name			: I2C_ReadBufvalue()
// brief		: read I2C buffer data
// note			: 
//            return: buffer data
//*********************************************************************
unsigned char I2C_ReadBufvalue(){
	return SBI0DBR;					// return I2C0 data buffer register


}



//*********************************************************************
// name			: I2C_Reset()
// brief		: I2C software reset. reset SBI0CR1,I2C0AR,SBI0CR2, not reset SBI0CR2<SBIM>
// note			: software reset I2C0
//*********************************************************************
void I2C_Reset(){
	SBI0CR2 = 0x1A;						// first write SWRST[1:0]=0x02
	SBI0CR2 = 0x19;						// then write [1:0]=0x01

}


//*********************************************************************
// name			: __interrupt I2C0_IRQ()
// brief		: I2C0 interrupt service routine
// note			:
//*********************************************************************
void __interrupt I2C0_IRQ(void){
	__asm("NOP");						// no operation

}



//*********************************************************************
// name			: I2C_Master_Write()
// brief		: 
// note			: 
//*********************************************************************
unsigned char I2C_Master_Write(I2C_MasterPacket* pi2c){
unsigned char cnt;
	
	I2C_Start((unsigned char)(pi2c->adr<<1)|I2C_Write);
	NACK_Stop_I2C();					// if NACK, I2C generate the stop condition
	
	I2C_TXOneByte(pi2c->cmd);			// I2C transmits one byte
	NACK_Stop_I2C();					// if NACK, I2C generate the stop condition

	
	for(cnt=0;cnt<pi2c->len;cnt++){
		I2C_TXOneByte(pi2c->buf[cnt]);  // I2C transmits one byte
		
		NACK_Stop_I2C();				// if NACK, I2C generate the stop condition
	}
	I2C_Stop();							// I2C generate the stop condition
	return 1;
}


//*********************************************************************
// name			: I2C_Master_Read()
// brief		: 
// note			: 
//*********************************************************************
unsigned char I2C_Master_Read(I2C_MasterPacket* pi2c){
unsigned char cnt;
	
	I2C_Start((unsigned char)(pi2c->adr<<1)|I2C_Write); // I2C generate the start condition
	NACK_Stop_I2C();							  // if NACK, I2C generate the stop condition
	
	I2C_TXOneByte(pi2c->cmd);				  	  // I2C transmits one byte ( send command )
	NACK_Stop_I2C();							  // if NACK, I2C generate the stop condition

	I2C_ReStart((unsigned char)(pi2c->adr<<1)|I2C_Read);// I2C generate the restart condition
	NACK_Stop_I2C();							  // if NACK, I2C generate the stop condition

	
	for(cnt=0;cnt<pi2c->len;cnt++){
		if(cnt == pi2c->len-1){
			I2C_AckEnable(0);                 	  // ACK disable 
			I2C_BufCount(BC_8bit);            	  // set to receive 8-bit data
			pi2c->buf[cnt] = I2C_RXOneByte();
			I2C_BufCount(BC_1bit);            	  // transmit 1-bit ACK
			I2C_TXOneByte(0xFF);              	  // write NACK to 0xFF

			I2C_AckEnable(1);                 	  // ACK enable 
			I2C_BufCount(BC_8bit);            	  // receive 8-bit data
		}
		else{
    	pi2c->buf[cnt] = I2C_RXOneByte();     	  // receive 8-bit data + 1-bit ACK
		}		
	}
	I2C_Stop();								      // I2C generate the stop condition
	return 1;
}


