//*************************************************************************************
// file			: SQ7653_I2C.c
// version		: V1.1 (Sep. 4, 20)
// brief		: I2C0 related functions
// note			: I2C0 related functions are gathered in this subroutine
//**************************************************************************************
#include "main.h"

unsigned char errCode = 0x00;
unsigned char tWait = 0;
unsigned char addrDirection;

//*********************************************************************
// name			: I2C0_SetSlaveAddr()
// brief		: set slave address
// param		: none
// retval		: none
//
// note			:
//*********************************************************************
void I2C0_SetSlaveAddr(uchar addr){
	I2C0AR = addr & 0xfe;				// I2C0AR[7:1]: Slave address setting
										// 0Xfe	1111 1110		
										// &0xfe   : and clear bit0  
}



//*********************************************************************
// name			: I2C0_Init()
// brief		: I2C0 initialization
// param		: none
// retval		: none
//
// note			: set I2C0 pins and transfer mode		  
//*********************************************************************
void I2C0_Init(){

    //====== I2C0 pin configuration initialization ======
    
        while(P0DI_P7!=1 || P0DI_P6 !=1){   // check SDA and SCL are both H level
            CLR_WDT;						// clear the watchdog timer 
            errCode = initBusLowErr;
			I2C0_Error_Handling();          // error handling
            
        }	
	
      //  SDA0 (P0.7) ; SCL0 (P0.6)-------
      
        P0CFGCR = 0xF3;  				    // set P0.7 to SDA0	1 111 0011
        P0CFGCR = 0xE3;					    // set P0.6 to SCL0	1 110 0011

    //====== Function Enable ======
        PCKEN2_I2C0 = 1;                    // this must be enabled first, the relevant staging settings later to be useful,I2C0 Enable
        

        SBI0CR2_SBIM = 1;                   // set to serial bus interface mode
    
    //====== parameter setting ======
        //SBI0CR1 = 0x16; 

        SBI0CR1_ACK = 1;                    // generate the clocks for an acknowledge signal and an interrupt request when the data transfer is finished (acknowledgement mode)        
        SBI0CR1_BC  = BC_8bit;              // number of clocks for data transfer = 8 + 1 = 9
        SBI0CR1_SCK = SCK_HI_135_LO_138;    // HIGH and LOW periods of the serial clock in the master mode @fsysclk=24MHz, fscl is about 88 kHz
      
        // set SBI0CR2<MST>, <TRX> and <BB> to "0" and SBI0CR2<PIN> and <SBIM> to "1"
        SBI0CR2 = 0x18;						// 0001 1000

}




//SBI0CR2--------------------------------------------------------------- 
//-----bit7 | bit6 | bit5 | bit4  |  bit3  | bit2 | bit1~bit0 ---------
//---  <MST>, <TRX>, <BB>, <PIN> and <SBIM>  <NA>   <SWRST>   ---------
//---  <MST>   1 : Master
//             0 : Slave
//---  <TRX>   1 : Transmitter   ; 0: Receiver
//---  <BB>    1 : Generate the start condition (when MST, TRX and PIN are "1")
//             0 : Generate the stop condition (when MST, TRX and PIN are "1")
//---  <PIN>   1 : Cancel interrupt service request
//
//---  <SBIM>  1 : 1 : Serial bus interface mode ;  0 : Port mode
//---  <SWRST> The software reset starts by first writing "10" and next writing "01"  
//----------------------------------------------------------------------  


//*********************************************************************
// name			: I2C0_Int()
// brief		: enable / disable I2C0 interrupt
// param		: - config : enable  enable I2C0 interrupt
//                     : disable disable I2C0 interrupt
// retval		: none
//
// note			: **
//*********************************************************************
void I2C0_Int(uchar config){
	__asm("DI");						// enable all maskable interrupts
	
	IFR_I2C0_IRQ = 0;                   // clear interrupt flag
	IER_I2C0_IRQ = config;	            // if config = enable  ,enable I2C0 interrupt
                                        // if config = disable ,disable I2C0 interrupt
	
	__asm("EI");						// enable all maskable interrupts
}


//*********************************************************************
// name			: I2C0_Start()
// brief		: I2C0 generate the start condition
// param		: - deviceId  : slave address
//                          e.g. slave address of AT24C08 is 0xa0
//            - addr      : start address of device
// retval		:  errCode
//
// note			: 
//*********************************************************************
uchar I2C0_Start(uchar deviceId, uchar addr){
	
	while(SBI0SR_BB == 1){
    }                                   // bus status 0: free ; 1 :busy

	SBI0CR1_ACK = 1;                    // *** SBI0CR1<ACK> must be set before transmit data ***
 
    I2C0_SetSlaveAddr_Direction(deviceId, directionW); // set slave device address and direction
	
    SBI0CR2 = 0xf8;						// Write "1" to SBI0CR2<MST>, <TRX>, <BB>, <PIN>  and <SBIM>to "1"
										// 1111 1000
                                        // set I2C to master mode, transmitter, generating start condition, cancel interrupt service request and serial bus interface mode
	
 	I2C0_Confirm_Tx_Completed();        // wait for tranfer completion
	I2C0_Wait_Slave_ACK();              // wait for Slave return ACK

    NOP;
    
    //---------------------------------------------------------------------------------------------    
	
    if(errCode != 0)  return errCode;
  
	SBI0DBR = addr;					    // write start address
	SBI0CR2 = 0xf8;						// Write "1" to SBI0CR2<MST>, <TRX>, <BB>, <PIN>  and <SBIM>to "1"
										// 1111 1000
                                        
    I2C0_Confirm_Tx_Completed();        // wait for tranfer completion
    I2C0_Wait_Slave_ACK();              // wait for Slave return ACK

    NOP;
}


//*********************************************************************
// name			: I2C0_Confirm_Tx_Completed()
// brief		: wait for tranfer completion
// param		: none
// retval		: none
// 
// note			:
//*********************************************************************
void I2C0_Confirm_Tx_Completed(){
   	while(SBI0SR_PIN)                   // after tranfer completion, PIN will chaange to 0	
	{
    }			                         
  

}



//*********************************************************************
// name			: I2C0_Wait_Slave_ACK()
// brief		: wait for Slave return ACK
// param		: none
// retval		:  - errCode
//
// note			:
//*********************************************************************
uchar I2C0_Wait_Slave_ACK(){
    uchar tWait;
    errCode = 0x00;
	tWait = 0;

    while(SBI0SR_LRB == 1){			    // check <LRB> wait for last received bit monitor is "0"							
            tWait++;					    // no response and increase counting
                
            if(tWait>waitCountMax){	        // if there is no response for too long, the loop will be broken		
                I2C0_Stop();			    // I2C0 stop -> release SCK
				errCode = nackTimeoutErr;
 				break;
            }
	NOP;
    }
	
	NOP;
	if(errCode != 0)  return errCode;
	
    // ACK
    errCode = 0x00;                    // if no error, return 0x00
    return errCode;
 
}


//*********************************************************************
// name			: I2C0_SetSlaveAddr_Direction() 
// brief		: set slave device address and direction
// param		: - deviceId     : slave address
//            - i2cDirection :  directionW (write data to slave)
//                              directionR (read data from slave)
// retval		:  errCode
//
// note			: 
//*********************************************************************
void I2C0_SetSlaveAddr_Direction(uchar deviceId, uchar i2cDirection){

    addrDirection = (deviceId | i2cDirection);
	SBI0DBR = addrDirection;

}


//*********************************************************************
// name			: I2C0_ReStart()
// brief		: I2C0 generate the restart condition
// param		: none
// retval		: none
//
// note			: 
//            
//*********************************************************************
uchar I2C0_ReStart(uchar deviceId){

	SBI0CR2 = 0x18;						// 0001 1000
										// set I2C to master mode, transmitter, generating start condition, cancel interrupt service request and serial bus interface mode

	while(SBI0SR_BB){                  // bus status 0: free ; 1 :busy
    }	  
	while(SBI0SR_LRB == 0){}
    
    I2C0_SetSlaveAddr_Direction(deviceId, directionR); // set slave device address and direction
	SBI0CR2 = 0xf8;						// Write "1" to SBI0CR2<MST>, <TRX>, <BB>, <PIN>  and <SBIM>to "1"
										// 1111 1000
                                        
    I2C0_Confirm_Tx_Completed();        // wait for tranfer completion
    I2C0_Wait_Slave_ACK();              // wait for Slave return ACK
    return errCode;
}



//*********************************************************************
// name			: I2C0_Stop()
// brief		: I2C0 generate the stop condition
// param		: none
// retval		: none
//
// note			: 
//            
//*********************************************************************
void I2C0_Stop(void){
	SBI0CR2 = 0xd8;						// Set SBI0CR2<MST>, <TRX> and <PIN> to "1" and SBI0CR2<BB> to "0"
										// 1101 1000
										// set I2C0 to master mode, transmitter, generating stop condition, cancel interrupt service request and port mode
										
	while(SBI0SR_BB == 1){}			    // bus status 0: free ; 1 :busy
}





//*********************************************************************
// name			: I2C0_Bytes_Write() 
// brief		: I2C0 transmits data
// param		: 
//            - *data   :data to be transmitted
//            - len     :data length
// retval		:  errCode
//
// note			:  
//            
//*********************************************************************
uchar I2C0_Bytes_Write(uchar *data, uchar len){
	uchar i;

	if(errCode != 0)  return errCode;
    
	for(i=0; i<len; i++){	
		tWait = 0;
		SBI0DBR = *data;
		data++;
        
        I2C0_Confirm_Tx_Completed();     // wait for tranfer completion
        I2C0_Wait_Slave_ACK();	         // wait for Slave return ACK
	}
    return errCode;
}



//*********************************************************************
// name			: I2C0_Bytes_Read() 
// brief		: I2C0 receives data
// param		: - addr    :start address to read
//            - *data   :read data
//            - len     :data length
// retval		:  errCode
//
// note			:  
//            
//*********************************************************************
uchar I2C0_Bytes_Read(uchar *data, uchar len){
	uchar i;
    if(I2C0_Wait_Slave_ACK()!= 0x00){
       return errCode;

    }
    
    NOP;
	len -= 1;
    // (len-1)byte..................
	for(i=0; i<len; i++){
		SBI0DBR   = 0x00;				// write dummy data to set PIN = 1
		I2C0_Confirm_Tx_Completed();
		*data = SBI0DBR;
		data++;
	}

    // һbyte..................
	SBI0CR1_ACK = 0;					// ACK=0
	SBI0DBR     = 0x00;					// write dummy data to set PIN = 1
	I2C0_Confirm_Tx_Completed();
	*data = SBI0DBR;

    //	Termination of data transfer by master mode
    //	Sent 1 bit negative acknowledge............
	SBI0CR1_BC = 1;						// BC, bit count = 1
	SBI0DBR    = 0x00;					// write dummy data to set PIN = 1
	I2C0_Confirm_Tx_Completed();
}



//*********************************************************************
// name			: I2C0_Reset()
// brief		: software reset I2C0
// param		: none
// retval		: none
// 
// note			: reset SBI0CR1,I2C0AR,SBI0CR2, not reset SBI0CR2<SBIM>
//*********************************************************************
void I2C0_Reset(){
	SBI0CR2 = 0x1A;						// first write SWRST[1:0] = 0x02
	SBI0CR2 = 0x19;						// then write [1:0] = 0x01

}



//*********************************************************************
// name			: I2C0_IRQ()
// brief		: I2C0interrupt service routine
// note			:
//*********************************************************************
void __interrupt I2C0_IRQ(void){
    __asm("NOP");                         // no operation

}


//*********************************************************************
// name			: I2C0_Error_Handling() 
// brief		: error handling
// param		: none
// retval		: none
// 
// note			: if one of SDA or SCL is not H level
//            
//*********************************************************************
void I2C0_Error_Handling(){
	if (errCode == initBusLowErr){
	
	//Add your error handling error handling......
		while(1){
			CLR_WDT;					 // clear the watchdog timer 
			__asm("NOP");
		}
    }

}



