//******************************************
// file 		: SQ7653_TMRA_Complementary
// version		: V 1.00 2019/03/20
// brief		: TMRA0 Complementary related functions
// note			:
//******************************************

#include "main.h"

unsigned char TMRA00_Int_Flag;		  	// TMRA00 interrupt flag
unsigned char TMRA01_Int_Flag;		  	// TMRA01 interrupt flag
//*********************************************************************
// name			: TMRA00_Timer_Init
// brief		: TMRA00 initialization to 8-bit timer mode
// note			:
//*********************************************************************
void TMRA00_Timer_Init(void)
{
//====== enable function ======
    PCKEN0_TMRA0 = 1;                   // this must be enabled first, the relevant staging settings later to be useful
    
	
//====== parameter setting ======
    TMRA0CFG_TCFG   = set_8bit;         // set to 8-bit mode
    TMRA0MD0_TCM    = timer;            // set to 8-bit timer mode
    TMRA0MD0_TCK    = 5;                // fcgck/4
    TMRA0MD0_DBE    = 1;                // enable double buffer
    
    TMRA0DR0        = 100;              // set counter value to 100
}


//*********************************************************************
// name			: TMRA0_16BIT_Complementary_Init
// brief		: TMRA0 initialization to 16-bit complementary mode
// note			: 
//*********************************************************************
void TMRA0_16BIT_Complementary_Init(void)
{
	//====== pin configuration initialization ======
    P0DO_P4         = 1;                // set P0.4 output level as the same as TMRA0MD1_TFF
    P0DO_P5         = 1;
	P0OE_P4         = 1;                // set P0.4 as output pin
    P0OE_P5         = 1;
//====== enable function ======
    PCKEN0_TMRA0    = 1;                // this must be enabled first, the relevant staging settings later to be useful

//====== parameter setting ======
    TMRA0CFG_TCFG   = complementary;    // set TMRA0 to complementary      
    TMRA0MD1_TCM    = PPG;              // set TMRA0 to PPG mode
	
	TMRA0MD1_TFF    = 0;                // set output pin inital state to high level (if set TMRMD0.TFF to 0, TMRMD0.TFF will be set to 1 automatically)

    TMRA0MD1_TCK    = 5;                // fcgck/4 = 1MHz = 1us
	TMRA0MD0_DBE    = 1;                // enable double buffer
    TMRA0MD1_DBE    = 1;                // enable double buffer

    TMRA0CPL_CPLMD  = 0x00;             // set to independent mode
   
    TMRA0PWM0       = 160;              // set PPG duty = 14496
    TMRA0PWM1       = 54; 
	
	TMRA0DR0        = 200;              // set PPG cycle,18120   	  
	TMRA0DR1        = 70;               // 14496/18120 = 80%
	//	***it is recommended to set PxCFGCR after the function initial value is set
    P0CFGCR         = 0xC1;             // set ~PPG pin to P0.4
	P0CFGCR         = 0xD1;             // set ~PPG pin to P0.5
	
}

//*********************************************************************
// name			: TMRA1_16BIT_Complementary_Init
// brief		: TMRA1 initialization to 16-bit complementary mode
// note			: 
//*********************************************************************
void TMRA1_16BIT_Complementary_Init(uchar CP_mode)
{
	//====== pin configuration initialization ======
    P0DO_P2         = 1;                // set P0.2 output level as the same as TMRA1MD1_TFF
    P0DO_P3         = 1;
	P0OE_P2         = 1;                // set P0.2 as output pin
    P0OE_P3         = 1;
//====== enable function ======
    PCKEN0_TMRA1    = 1;                // this must be enabled first, the relevant staging settings later to be useful

//====== parameter setting ======
    TMRA1CFG_TCFG   = complementary;    // set TMRA1 to complementary  
    TMRA1MD1_TCM    = PPG;              // set TMRA1 to PPG mode
	
	TMRA1MD1_TFF    = 0;                // set output pin inital state to high level (if set TMRMD1.TFF to 0, TMRMD1.TFF will be set to 1 automatically)
     
    TMRA1MD1_TCK    = 5;                // fcgck/4 = 1MHz = 1us
	TMRA1MD0_DBE    = 1;                // enable double buffer
    TMRA1MD1_DBE    = 1;                // enable double buffer

    TMRA1CPL_CPLMD  = CP_mode;          // set fault mode
   
    if(CP_mode == 2){
		NOP;                            // copy duty  cycle of Complementary mode	  
	}
	else if(CP_mode == 3){
		TMRA1PWM0   = 198;              // set duty cycle = (6342/18120)= 35%, cycle is asa the same as TMRA0  
		TMRA1PWM1   = 24;
	}
	else{                               // set to independent mode
		TMRA1PWM0   = 100;            
		TMRA1PWM1   = 40;
		TMRA1DR0    = 200;          	  
		TMRA1DR1    = 80;
	}
	//	***it is recommended to set PxCFGCR after the function initial value is set
    P0CFGCR         = 0xA1;             // set PPG pin to P0.2
	P0CFGCR         = 0xB1;             // set ~PPG pin to P0.3
}

//*********************************************************************
// name			: TMRA0_16BIT_SetDuty
// brief		: set duty cycle
// note			: 
//*********************************************************************
void TMRA0_16BIT_SetDuty(uint duty)
{
    
	TMRA0PWM0       = (duty & 0xFF);
	TMRA0PWM1       = ((duty & 0xFF00)>>8);
}

//*********************************************************************
// name			: TMRA00_16BIT_SetPeriod
// brief		: set period
// note			: 
//*********************************************************************
void TMRA0_16BIT_SetPeriod(uint period)
{
    TMRA0DR0        = (period & 0xFF);
	TMRA0DR1        = ((period & 0xFF00)>>8);
}



//*********************************************************************
// name			: TMRA0_Setdead time
// brief		: set TMRA0 dead time
// note			: dead time = scale x Dtime
//*********************************************************************
void TMRA0_Setdead time(uchar scale, uchar Dtime)
{
    TMRA0DTM_DTMPSC  = scale;
	if(Dtime > 63)
		TMRA0DTM_DTM = 63;
	else 
		TMRA0DTM_DTM = Dtime;	
}

//*********************************************************************
// name			: TMRA0_Fault_Detect_Init
// brief		: TMRA0 fault mode initialization
// note			: 
//*********************************************************************
void TMRA0_Fault_Detect_Init(uchar FD_mode)
{
    DI;	
	if(FD_mode < 4)
		TMRA0CPL_FLTMD = FD_mode;       // fault mode selection
	else 
		TMRA0CPL_FLTMD = 1;             // enable fault mode, output 0 when fault occurs
	
	P4CFGCR         = 0x91;             // set FLTIN0(TMRA0 Fault Detection) pin to P4.1
	IER_TMRA00_IRQ = 1;                // enable TMRA00 interrupt
	TMRA0CPL_FLTIE  = 1;                // enable fault interrupt
	EI;		
}

//*********************************************************************
// name			: TMRA00_Start
// brief		: start to run TMRA00
// note			:
//*********************************************************************
void TMRA00_Start(void)
{
    TMRA0CR0_RUN   = 1;                
}

//*********************************************************************
// name			: TMRA01_Start
// brief		: start to run TMRA01
// note			:
//*********************************************************************
void TMRA01_Start(void)
{
    TMRA0CR1_RUN   = 1;                
}

//*********************************************************************
// name			: TMRA11_Start
// brief		: start to run TMRA11
// note			:
//*********************************************************************
void TMRA11_Start(void)
{
    TMRA1CR1_RUN   = 1;                
}

//*********************************************************************
// name			:TMRA00_Stop
// brief		: stop TMRA00
// note			:
//*********************************************************************
void TMRA00_Stop(void)
{
    TMRA0CR0_RUN   = 0;                
}

//*********************************************************************
// name			:TMRA01_Stop
// brief		: stop TMRA00 & 16BIT TMRA0
// note			:
//*********************************************************************
void TMRA01_Stop(void)
{
    TMRA0CR1_RUN   = 0;                
}

//*********************************************************************
// name			:TMRA11_Stop
// brief		: stop TMRA11 & 16BIT TMRA1
// note			:
//*********************************************************************
void TMRA11_Stop(void)
{
    TMRA1CR1_RUN   = 0;                // stop TMRA11
}
//*********************************************************************
// name			: TMRA00_IntEnable
// brief		: enable TMRA00 interrupt
// note			: 
//*********************************************************************
void TMRA00_IntEnable(void)
{
    DI;                                 // enable all maskable interrupts // EIRL_IMF=0
    IFR_TMRA00_IRQ = 0;                // clear TMRA00 interrupt flag
    IER_TMRA00_IRQ = 1;                // enable TMRA00 interrupt
    EI;                                 // enable all maskable interrupts // EIRL_IMF=1
}


//*********************************************************************
// name			: TMRA01_IntEnable
// brief		: enalbe TMRA0 interrupt or enalbe TMRA01 interrupt
// note			: 
//*********************************************************************
void TMRA01_IntEnable(void)
{
    DI;                                 // enable all maskable interrupts // EIRL_IMF=0
    IFR_TMRA01_IRQ = 0;                // clear TMRA00 interrupt flag
    IER_TMRA01_IRQ = 1;                // enable TMRA00 interrupt
    EI;                                 // enable all maskable interrupts // EIRL_IMF=1
}


//*********************************************************************
// name			: TMRA00_IntDisable
// brief		: disable TMRA00 interrupt
// note			: 
//*********************************************************************
void TMRA00_IntDisable(void)
{
    DI;                                 // enable all maskable interrupts // EIRL_IMF=0
    IFR_TMRA00_IRQ = 0;                // clear TMRA00 interrupt flag
    IER_TMRA00_IRQ = 0;                // disalbe TMRA00 interrupt
    EI;                                 // enable all maskable interrupts // EIRL_IMF=1
}

//*********************************************************************
// name			: TMRA01_IntDisable
// brief		: disalbe TMRA0 interrupt or disalbe TMRA01 interrupt
// note			: 
//*********************************************************************
void TMRA01_IntDisable(void)
{
    DI;                                 // enable all maskable interrupts // EIRL_IMF=0
    IFR_TMRA01_IRQ = 0;                // clear TMRA00 interrupt flag
    IER_TMRA01_IRQ = 0;                // disalbe TMRA01 interrupt
    EI;                                 // enable all maskable interrupts // EIRL_IMF=1
}

//*********************************************************************
// name			: TMRA00_IRQ
// brief		: TMRA00 interrupt service routine
// note			:
//*********************************************************************
void __interrupt TMRA00_IRQ(void)
{
    
	TMRA00_Int_Flag = 1;
    NOP;                                // no operation
}

//*********************************************************************
// name			: TMRA01_IRQ
// brief		: TMRA01 interrupt service routine
// note			:
//*********************************************************************
void __interrupt TMRA01_IRQ(void)
{
    
	TMRA01_Int_Flag = 1;
    NOP;                                // no operation
}

//*********************************************************************
// name			: TMRA11_IRQ
// brief		: TMRA11 interrupt service routine
// note			:
//*********************************************************************
void __interrupt TMRA11_IRQ(void)
{
    
    NOP;                                // no operation
}


