//*************************************************************************************
// file			: SQ7653_ADC.c
// version		: V1.0 2020/07/17
// brief		: ADC related functions
// note			: ADC related functions are gathered in this subroutine
//*************************************************************************************

#include "main.h"


//*********************************************************************
// name			: ADC_Init()
// brief		: ADC initialization
// note			: 
//*********************************************************************
void ADC_Init(){
//====== ADC pin configuration initialization (select the analog signal input channel for AD conversion) ======
// SQ7653 has 8 pins available from AIN0 to AIN7---------------------

/*
	//--- AIN0 ---//	
	P1CFGCR = 0x8F;                       // set P1.0 as AIN0
	 
*/

/*
	//--- AIN1 ---//	
	P1CFGCR = 0x9F;                       // set P1.1 as AIN1
*/

/*
	//--- AIN2 ---//	
	P1CFGCR = 0xAF;                       // set P1.2 as AIN2
	
*/
	
/*
	//--- AIN3 ---//	
	P1CFGCR = 0xBF;                       // set P1.3 as AIN3
	
*/


	//--- AIN4 ---//	 [this pin is selected for this example]
	P1CFGCR = 0xCF;                       // set P1.4 as AIN4
	 


/*
	//--- AIN5 ---//	
	P1CFGCR = 0xDF;                       // set P1.5 as AIN5
	
*/	
	
/*
	//--- AIN6 ---//	
	P1CFGCR = 0xEF;                       // set P1.6 as AIN6
	
*/		

/*
	//--- AIN7 ---//	
	P1CFGCR = 0xFF;                       // set P1.7 as AIN7
	
*/	

// -------------------------------------------------------------
//====== parameter setting ======
	PCKEN6_ADC   = 1;                     // ADC peripheral clock enable
	ADCCR0_IRFEN = 0x01;                  // set VDDA as Vref
	
	ADCCKDIV = 0x04;                      // ADC clock = fsysclk/(2^4)
    ADCCR0_INTLV = 1;                     // interrupt interval is set to wait until all AIN scans are completed before an interrupt is generated
	
	ADCCR1_EVSEL = 0x00;                  // set event to ADC convertion
	
	//Multiple AIN channel scanning
	ADCSCAN0 = 0x10;                      // set ADC scan channel to AIN4            
	//ADCSCAN1 = 0x00;                    
}

//*********************************************************************
// name			: ADC_Start()
// brief		:  ADC
// note			:
//*********************************************************************
void ADC_Start()
{
	ADCCR0_ADEN = 1;                      // enable ADC
}


//*********************************************************************
// name			: ADC_Stop()
// brief		: ֹͣ ADC	
// note			: 
//*********************************************************************
void ADC_Stop()
{
	ADCCR0_AMD  = 0;                      // disable ADC operation and forcibly stop ADC operation
    ADCCR0_ADEN = 0;                      // disable ADC    
}


//*********************************************************************
// name			: ADC_IntEnable()
// brief		: enable ADC interrupt
// note			: 
//*********************************************************************
void ADC_IntEnable(){
	__ASM("DI");

	IER_ADC_IRQ = 1;                     // enable ADC interrupt
	ADCCR0_INTEN = 1;                    // enable AD conversion finished interrupt
	__ASM("EI");
}


//*********************************************************************
// name			: ADC_Disable()
// brief		: disable ADC interrupt
// note			: 
//*********************************************************************
void ADC_Disable(){
	__ASM("DI");

	IER_ADC_IRQ = 0;                     // disable ADC interrupt
	ADCCR0_INTEN = 0;                    // disable ADC interrupt
	__ASM("EI");
}



//*********************************************************************
// name			: ADC_Convert_Repeat()
// brief	  	: Perform ADC repeat mode conversion and store the conversion value in ADCDRL and ADCDRH and average them
// note			:
//*********************************************************************
unsigned int ADC_Convert_Repeat(uchar SAIN)
{
    uchar i;
    ulong adc_data = 0;
    uint  ad_min   = 0, ad_max = 4096;
    BIT_LIST_16 ADC_READ;

//====== parameter setting ======
	ADCCHSEL_CHSEL = SAIN;                  // select ADC input channel

	ADCCR0_AMD = 0x03;                      // set ADC operating mode:
	while(!ADCSR_ADRDY);                    // wait for ADC to be ready=	
    ADCSR_ADRS = 1;                         // start AD conversion


    for( i=0; i<(ad_test_times+2); i++ )    // repeat sampling 18 times
    {
        while( !ADCSR_EOCF ){ CLR_WDT; }    // wait for AD conversion complete
        ADCSR_EOCF = 1;                     // clear ADC conversion completion flag		
        ADC_READ.byte[0] = ADCDRL;          // when reading ADC value,  read low byte first and then read high byte
        ADC_READ.byte[1] = ADCDRH;          // when reading ADC value,  read low byte first and then read high byte
        adc_data += ADC_READ.word;
        if( ADC_READ.word > ad_min ) ad_min = ADC_READ.word;
        if( ADC_READ.word < ad_max ) ad_max = ADC_READ.word;
    }
    ADCSR_ADRS = 0;                         // stop AD conversion

    adc_data -= ad_max;                     // subtract the maximum value during sampling
    adc_data -= ad_min;                     // subtract the minimum value during sampling
    
	adc_data /= ad_test_times;              // average

	return (unsigned int)adc_data;
}

//*********************************************************************
// name			: ADC_Convert_Single()
// brief		: ADC conversion in single mode
// note			: 
//*********************************************************************
void ADC_Convert_Single(){

	ADCCR0_AMD = 0x01;                      // set ADC operating mode: single mode
	while(!ADCSR_ADRDY);                    // wait for ADC to be ready
	ADCSR_ADRS = 1;                         // start ADC conversion	
}

//*********************************************************************
// name			: ADC_ConvertDone()
// brief		: ADC conversion completed
// note			: 
//*********************************************************************
char ADC_ConvertDone(){
	if(ADCSR_EOCF){
		return 1;                           // if ADC conversion completed, return 1
	}
	else{
		return 0;                           // if ADC conversion not completed, return 0
	}
}



//*********************************************************************
// name			: ADC_Read_Value()
// brief		: readout ADC value
// note			: chsel : AIN setting value  
// AIN0	       0x0
// AIN1	       0x1
// AIN2	       0x2
// AIN3	       0x3
// AIN4	       0x4
// AIN5	       0x5
// AIN6	       0x6
// AIN7	       0x7

//*********************************************************************
long ADC_Read_Value(unsigned char chsel){
	unsigned int adc_value = 0;
	
	ADCCHSEL = chsel;                       // select ADC input channel
	
	adc_value |= ADCDRL;                    // when reading ADC value,  read low byte first and then read high byte
	adc_value |= (unsigned int)ADCDRH<<8;
	return (long)adc_value;
}


//*********************************************************************
// name			: __interrupt ADC_IRQ()
// brief		: ADC interrupt subroutine
// note			:
//*********************************************************************
void __interrupt ADC_IRQ(void){
	__asm("NOP");                           // no operation
	LED1_out = ~LED1_out;                   // LED1 inverted (LED1 (P0.0))

}

