//*************************************************************************************
// file			: SQ7653_ADC_LVCMP.c
// version		: V1.0 2020/07/17
// brief		: ADC_LVCMP related functions
// note			: ADC_LVCMP related functions are gathered in this subroutine
//*************************************************************************************

#include "main.h"

unsigned char ADC_Int_Flag;			      // ADC interrupt flag
unsigned char ADCLVCMP_Int_Flag =0;	          // interrupt flag of ADC if level comparison condition is met

unsigned char r_tempData[2];
long ain7_value;                       		  // store the ADC value of AIN7
long ain7_mv;                          		  // ADC value converted to voltage value (in mv)

//*********************************************************************
// name			: ADC_Init()
// brief		: ADC initialization
// note			: 
//*********************************************************************
void ADC_Init(){
    
//=== ADC pin configuration initialization (select the analog signal input channel for AD conversion)===
// P1CFGCR [bit7]=1 (write), [bit6:4]=(pin), [bit3:0] set according to the PNIC chapter of datasheet Pin Function
// SQ7553 has 8 pins available from AIN0 to AIN7---
// PINC selection

	//--- AIN0 ---//	
	//P1CFGCR = 0x8F;                     // set P1.0 as AIN0 1 000 1111
    
	//--- AIN1 ---//	
    //P1CFGCR = 0x9F;                     // set P1.1 as AIN1 1 001 1111

	//--- AIN2 ---//	
	//P1CFGCR = 0xAF;                     // set P1.2 as AIN2 1 010 1111
	
	//--- AIN3 ---//	
	//P1CFGCR = 0xBF;                     // set P1.3 as AIN3 1 011 1111

	//--- AIN4 ---//	                  
	//P1CFGCR = 0xCF;                     // set P1.4 as AIN4 1 100 1111 

	//--- AIN5 ---//	
    //P1CFGCR = 0xDF;                     // set P1.5 as AIN5 1 101 1111
	
	//--- AIN6 ---//	
    //P1CFGCR = 0xEF;                     // set P1.6 as AIN6 1 110 1111
	
	//--- AIN7 ---//[this pin is selected for this example]	
	P1CFGCR = 0xFF;                       // set P1.7 as AIN7 1 111 1111
   
//====== Function Enable ======          // this must be enabled first, the relevant staging settings later to be useful
	PCKEN6_ADC = 1;                       // enable ADC peripheral clock 

//====== parameter setting ======
// Set ADC Convert Speed
    ADCCKDIV_ADCKDIV = 2;	              // fsysclk/(2^2)

// after ADEN is set, the ADC needs to wait for clock warm up to allow the user to set ADCSR_ADRS to start the AD conversion
	ADCCR0_ADEN  = 1;                     // enable ADC
	ADCCR0_IRFEN = 0x01;                  // set VDDA as Vref
    //ADCCR0_INTLV  = 1;                  // interrupt interval is set to wait until all AIN scans are completed before an interrupt is generated

	ADCCR0_AUTOPD = 0;                    // disable automatic power down after each ADC conversion
    
// set ADC operating mode
    ADCCR0_AMD   = 0x01;                  // single mode

// ADC data buffer automatic overwrite
    //ADCSR_BUFOVR = 1;                   // ADC data buffer automatic overwrite

// set event	
	ADCCR1_EVSEL = 0x00;                  // set event to ADC convertion
	
// Multiple AIN channel scanning
	ADCSCAN0 = 0x80;                      // set ADC scan channel to AIN7

}

//*********************************************************************
// name			: ADCLVLCMP_Init()
// brief		: ADC level comparison initialization
// note			: 
// 
// lvcmpsel : CMP_Disable	       0x0   // disable ADC level comparison
//            CMP_ADCLLV           0x1   // ADC value < ADCLLV             ** CMP ADCLLV ** 
//            CMP_ADCLLH           0x2   // ADC value > ADCHLV             ** CMP ADCHLV ** 
//            CMP_ADCL_BETWEEN     0x3   // ADCLLV > ADC value > ADCHLV    ** CMP ADCLLV and ADCHLV **
//*********************************************************************
void ADCLVLCMP_Init(unsigned char lvcmpsel){
    
    ADCLV_LVCMP  = lvcmpsel;             // set ADC level comparison setting
    
    //ADCLV_LVINTVL = 0;    			 // all samples were compared 
	ADCLV_LVINTVL = 1;    				 // level comparison specified by LVSEL   
	ADCLV_LVSEL   = 7;                   // Specify ADC channel : set AIN7 for level comparison


}



//*********************************************************************
// name			: ADCLVLCMP_SetTh()
// brief		: set threshold of ADC level comparison (0x0000~0x0FFF)
//
// note			: set low threshold of ADC level comparison
//            set high threshold of ADC level comparison
//*********************************************************************
void ADCLVLCMP_SetTh(uchar adcllv_Th,uchar adchlv_Th){
/*
	ADCLLV = 0xFA0;
	ADCHLV = 0x3E8;
*/ 

    ADCLLV = 4000;          // if ADC < ADCLLV, then ADCSR_LVDET = 1
	ADCHLV = 1000;          // if ADC > ADCHLV, then ADCSR_LVDET = 1

}


//*********************************************************************
// name			: ADC_Stop_and_Disable()
// brief		: disable ADC operation and forcibly stop ADC operation
// note			: 
//*********************************************************************
void ADC_Stop_and_Disable()
{
	ADCCR0_AMD  = 0;                      // disable ADC operation and forcibly stop ADC operation
    ADCCR0_ADEN = 0;                      // disable ADC    
}


//*********************************************************************
// name			: ADC_IntEnable()
// brief		: enable ADC interrupt
// note			: 
//*********************************************************************
void ADC_IntEnable(){
	__ASM("DI");

	IER_ADC_IRQ = 1;                     // enable ADC interrupt
	ADCCR0_INTEN = 1;                    // enable AD conversion finished interrupt
	
	__ASM("EI");
}


//*********************************************************************
// name			: ADC_IntDisable()
// brief		: disable ADC interrupt
// note			: 
//*********************************************************************
void ADC_IntDisable(){
	__ASM("DI");

	IER_ADC_IRQ  = 0;                     // disable ADC interrupt
	IFR_ADC_IRQ  = 0;	                  // disable IRQ
	ADCCR0_INTEN = 0;                     // disable AD conversion finished interrupt
	
	__ASM("EI");
}



//*********************************************************************
// name			: ADCLVLCMP_IntEnable()
// brief		: enable ADC level comparison interrupt
// note			: 
//*********************************************************************
void ADCLVLCMP_IntEnable(){
    __ASM("DI");
    
    IER_ADC_IRQ = 1;                      // enable ADC interrupt
    ADCCR2_LVINTEN = 1;                   // enable ADC level comparison
    
    __ASM("EI");
}



//*********************************************************************
// name			: ADC_Convert_Single()
// brief		: ADC conversion in single mode
// note			: 
//*********************************************************************
void ADC_Convert_Single(){
	while(!ADCSR_ADRDY);                  // wait for ADC to be ready
	ADCSR_ADRS = 1;                       // start ADC conversion
	

}



//*********************************************************************
// name			: ADC_ConvertDone()
// brief		: ADC conversion completed
// note			: 
//*********************************************************************
char ADC_ConvertDone(){
	if(ADCSR_EOCF){
		return 1;                         // if ADC conversion completed, return 1
	}
	else{
		return 0;                         // if ADC conversion not completed, return 0
	}
}



//*********************************************************************
// name			: ADC_Read_Value()
// brief		: readout ADC value
// note			: chsel : AIN setting value  
//                   AIN0	       0x0
//                   AIN1	       0x1
//                   AIN2	       0x2
//                   AIN3	       0x3
//                   AIN4	       0x4
//                   AIN5	       0x5
//                   AIN6	       0x6
//                   AIN7	       0x7
//*********************************************************************
long ADC_Read_Value(unsigned char chsel){
	unsigned int adc_value = 0;
	
	ADCCHSEL = chsel;                     // select ADC input channel

    r_tempData[0] = ADCDRL;               // when reading ADC value,  read low byte first and then read high byte
    r_tempData[1] = ADCDRH;

	adc_value |= r_tempData[0];                  
	adc_value |= (unsigned int)r_tempData[1]<<8;
	return (long)adc_value;
}


//*********************************************************************
// name			: __interrupt ADC_IRQ()
// brief		: ADC interrupt subroutine
// note			:
//*********************************************************************
void __interrupt ADC_IRQ(void){
	__asm("NOP");                         // no operation
    
	
	for(i=0;i<6000;i++);
	
	//--- If ADC meets level comparison condition, interrupt is generated ---
	if (ADCSR_LVDET){			          // interrupt is generated when the level comparison condition is detected
		ADCLVCMP_Int_Flag = 1;			  // the interrupt flag is set when meet the level comparison condition        
        ADCSR_EOCF = 1;		              // clear ADC conversion completion flag
        ADCSR_LVDET = 1;	              // clean flag
		
        ain7_value = ADC_Read_Value(AIN7);  			   // read the ADC value of AIN7
	    ain7_mv = ain7_value*ADC_RefScale(3090000)/1000;   // ADC value to voltage value (unit:mV)
/*
		ADCCHSEL_CHSEL = 7;	              // set Ain7 and then read ADCDRL and ADCDRH
        r_tempData[0] = ADCDRL;	          // read Low and then High
		r_tempData[1] = ADCDRH;	          // read Low and then High
*/       


 
		LED1_out = ~LED1_out;             // LED1 (P0.0) reverse
		__asm("NOP");                     // no operation
		__asm("NOP");                     // no operation, the break point can be set here

     }


	//--- ADC conversion completed ---
/*
	if (ADCSR_EOCF){			          // ADC conversion completed, interrupt is generated
		ADC_Int_Flag = 1;				  // set ADC interrupt flag
        ADCSR_EOCF = 1;		              // clear ADC conversion completion flag
	}
*/	



}


