//*************************************************************************************
// file			: SQ7617_TCA.c
// version		: V1.0
// brief		: TCA related functions
// note			: TCA related functions are gathered in this subroutine
//**************************************************************************************
#include "main.h"

unsigned int captch_count;
unsigned int TCA0_Int_Flag;					// TCA0 interrupt flag
unsigned int TCA1_Int_Flag;					// TCA1 interrupt flag
unsigned int TCA5_Int_Flag;					// TCA5 interrupt flag

//*********************************************************************
// name			: TCA0_Timer_Init()
// brief		: TCA0 initialization to 16-bit timer mode
// note			: 
//*********************************************************************
void TCA0_Timer_Init(){
//====== ܿ( Enable) ======
	PCKEN0_TCA0 = 1;                    	  // TCA0 Enable
	
//====== parameter setting ======
	TA0MOD_TAM = 0x00;                  	  // set to 16-bit timer mode
	TA0MOD_TADBE    = 1;                	  // enable double buffer
	
    TA0MOD_TACK     = 0x00;                	  // set TCA0 source clock to fcgck/(2^10) = 24M/1024 = 23.437kHz => 42.6667us
}


//*********************************************************************
// name			: TCA0_IntEnable()
// brief		: enable TCA0 interrupt
// note			: 
//*********************************************************************
void TCA0_IntEnable(){
	__ASM("DI");							  // disable all maskable interrupts
	
	IER_TCA0 = 1;                       	  // enable TCA0 interrupt
	//TA0CR_TAOVE = 0;                    	  // generate no TCA0 interrupt request when the counter overflow occurs
	TA0CR_TAOVE = 1;                    	  // generate a TCA0 interrupt request when the counter overflow occurs
	
	__ASM("EI");							  // enable all maskable interrupts
}


//*********************************************************************
// name			: TCA0_SetPeriod()
// brief		: set timer counter
// note			: 
//*********************************************************************
void TCA0_SetPeriod(unsigned int period){
	
 	TA0DRAL = period&0xFF;              	  // set timer counter
 	TA0DRAH = (period>>8)&0xFF;
	//TA0DRA = period;

}


//*********************************************************************
// name			: TCA0_Start()
// brief		: start TCA0
// note			: 
//*********************************************************************
void TCA0_Start(){
	
	TA0CR_TAS = 1;                      	  // start TCA0

}


//*********************************************************************
// name			: TCA0_Stop()
// brief		: stop TCA0
// note			: 
//*********************************************************************
void TCA0_Stop(){
	
	TA0CR_TAS = 0;                      	  // stop TCA0

}


//*********************************************************************
// name			: TCA1_PulseMeasure_Init()
// brief		: TCA1 initialization to pulse width measurement mode
// note			: 
//*********************************************************************
void TCA1_PulseMeasure_Init(){
//====== pin configuration initialization ======

// set measurement pin to P0.1
	FSELR0 = (FSELR0&0xF0)|0x05;              
	PCSELR4_TCA1 = 0x00;              		  
    
	P0FC1_P1 = 1;                       	  
	P0FC2_P1 = 0;



/*
// set measurement pin to P2.1
    FSELR2 = (FSELR2&0xF0)|0x05;              
	PCSELR4_TCAPC1 = 0x01;              	  	
	P2FC1_P1 = 1;                       	  
	P2FC2_P1 = 0;
*/


//====== enable function ======
	PCKEN0_TCA1 = 1;                    	  // TCA1 Enable
	
//====== parameter setting ======	
	TA1MOD_TAM = 0x06;                  	  // set TCA1 to pulse width measurement mode
	
	// if the system frequency is 24MHz, the input frequency of TMRA00 is 23.437KHz
    TA1MOD_TACK   = 0x00;                	  // set TCA0 source clock to fcgck/(2^10) = 24M/1024 = 23.437kHz => 42.6667us
	
	TA1MOD_TAMCAP = 0;                	  	  // double edge capture
	//TA1MOD_TAMCAP = 1;                	  // single edge capture
	
	//TA1MOD_TATED  = 0;                   	  // rising edge/H level
	TA1MOD_TATED    = 1;                   	  // falling edge/L level

}


//*********************************************************************
// name			: TCA1_IntEnable()
// brief		: enable TCA1 interrupt
// note			: 
//*********************************************************************
void TCA1_IntEnable(){
	__ASM("DI");							  // disable all maskable interrupts
	
	IER_TCA1 = 1;                       	  // enable TCA1 interrupt
	//TA0CR_TAOVE = 0;                    	  // generate no TCA1 interrupt request when the counter overflow occurs
	TA1CR_TAOVE   = 1;                    	  // generate a TCA1 interrupt request when the counter overflow occurs
	
	__ASM("EI");							  // enable all maskable interrupts
}


//*********************************************************************
// name			: TCA1_Start()
// brief		: start TCA1
// note			: 
//*********************************************************************
void TCA1_Start(){
	
	TA1CR_TAS = 1;                      	  // start TCA1
	
}


//*********************************************************************
// name			: TCA1_Stop()
// brief		: stop TCA1
// note			: 
//*********************************************************************
void TCA1_Stop(){
	
	TA1CR_TAS = 0;                      	  // stop TCA1
	
}


//*********************************************************************
// name			: TCA1_ReadSingleCapture()
// brief		: get the counter value of single edge capture mode
// note			: 
//*********************************************************************
unsigned int TCA1_ReadSingleCapture(){

    return *(unsigned int*)&TA1DRBL;    	  
   	  			      
}


//*********************************************************************
// name			: TCA1_ReadDoubleCapture()
// brief		: get the counter value of double edge capture mode 
// note			: 
//*********************************************************************
unsigned int TCA1_ReadDoubleCapture()
{
	return *(unsigned int*)&TA1DRAL;    	  

}

//*********************************************************************
// name			: TCA5_PPG_Init()
// brief		: TCA5 initialization to PPG mode
// note			: 
//*********************************************************************
void TCA5_PPG_Init()
{
//====== pin configuration initialization ======


// set PPG output pin to P0.5
    P0DO_P5  = 1;                       // set P0.5 output level as the same as TA5CR_TATFF
    P0OE_P5  = 1;                       // set P0.5 as output pin

	FSELR0 = (FSELR0&0x0F)|0x50;		
	PCSELR5_TCA5 = 0x00;              	
    
	P0FC1_P5 = 1;                       
	P0FC2_P5 = 0;



/*	
// set PPG output pin to P2.5
    P2DO_P5  = 1;                       // set P2.5 output level as the same as TA5CR_TATFF
    P2OE_P5  = 1;                       // set P2.5 as output pin

    FSELR2 = (FSELR2&0x0F)|0x50;		
	PCSELR5_TCA5 = 0x01;              	
	
	P2FC1_P5 = 1;                       
	P2FC2_P5 = 0;
*/

	
//====== enable function ======
	PCKEN1_TCA5 = 1;                    // TCA5 Enable



//====== parameter setting ======
	TA5MOD_TAM      = 0x03;             // set operation mode to PPG output mode
    TA5CR_TATFF     = 1;                // set output pin inital state to high level
	
	TA5CR_TAMPPG    = 0; 				// PPG continuous output
    //TA5CR_TAMPPG    = 1;              // PPG one-shot output
	
    TA5MOD_TADBE    = 1;                // enable double buffer
    TA5MOD_TACK     = 0x02;             // set source clock to fcgck/(2^2) = 4M/4 = 1MHz => 1us
    TA5DRA          = 10000;            // set PPG period = 10000 * 1us = 10ms
    TA5DRB          = 6500;               // set PPG duty cycle = 6500/10000 = 65%
}


//*********************************************************************
// name			: TCA5_Start()
// brief		: start TCA5
// note			: 
//*********************************************************************
void TCA5_Start(){
	
	TA5CR_TAS = 1;                      	  // start TCA5
	
}


//*********************************************************************
// name			: TCA5_Stop()
// brief		: stop TCA5
// note			: 
//*********************************************************************
void TCA5_Stop(){
	
	TA5CR_TAS = 0;                      	  // stop TCA5
	
}

//*********************************************************************
// name			: TCA5_SetDuty()
// brief		: set output duty cycle
// note			: 
//*********************************************************************
void TCA5_SetDuty(unsigned int duty)
{
    TA5DRB          = duty;
}

//*********************************************************************
// name			: TCA5_SetPeriod
// brief		: set timer counter
// note			: 
//*********************************************************************
void TCA5_SetPeriod(unsigned int period)
{
    TA5DRA          = period;
}



//*********************************************************************
// name			: __interrupt IntTCA0()
// brief		: TCA0 interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntTCA0(void){
	__asm("NOP");							  // no operation

	TCA0_Int_Flag = 1;						  // set TCA0_Int_Flag to 1
}

//*********************************************************************
// name			: __interrupt IntTCA1()
// brief		: TCA1 interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntTCA1(void){
	__asm("NOP");							  // no operation

	TCA1_Int_Flag = 1;						  // set TCA1_Int_Flag to 1
	captch_count = TCA1_ReadSingleCapture();  // get counter value
}



//*********************************************************************
// name			: __interrupt IntTCA5()
// brief		: TCA5 interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntTCA5(void){
	__asm("NOP");							  // no operation

	TCA5_Int_Flag = 1;						  // set TCA5_Int_Flag to 1
}

