//*************************************************************************************
// file			: SQ7613_I2C.c
// version		: V1.2
// brief		: I2C related functions
// note			: I2C related functions are gathered in this subroutine
//**************************************************************************************
#include "main.h"



//*********************************************************************
// name			: I2C0_Init()
// brief		: I2C0 initialization
// note			: set I2C0 pins and transfer mode 
//			  *set I2C0 to slave mode and slave address to 0x68 in this example
//*********************************************************************
void I2C0_Init(){

	
//====== I2C0 pin configuration initialization ======
	//--SDA0 (P0.5) ; SCL0 (P0.6)------- [this pin is selected for this example]
	FSELR0 = (FSELR0&0x0F)|0x30;		//set function selection register 0   
	PCSELR1_I2C0 = 0x00;				//set peripheral channel selection register   
    
	P0FC1_P5 = 1;                       //set P0.5 to SDA0
	P0FC2_P5 = 0;
	P0FC1_P6 = 1;                       //set P0.6 to SCL0
	P0FC2_P6 = 0;


/*	
	//--SDA0 (P0.4) ; SCL0 (P0.5)-------
	FSELR0 = (FSELR0&0x0F)|0x40;		//set function selection register 0   
	PCSELR1_I2C0 = 0x01;				//set peripheral channel selection register
    
	P0FC1_P4 = 1;                       //set P0.4 to SDA0
	P0FC2_P4 = 0;
	P0FC1_P5 = 1;                       //set P0.5 to SCL0
	P0FC2_P5 = 0;
*/	

	
/*	
	//--SDA0 (P3.2) ; SCL0 (P3.3)-------
	FSELR3 = (FSELR3&0xF0)|0x00;		//set function selection register 3    
	PCSELR1_I2C0 = 0x03;				//set peripheral channel selection register
    
	P3FC1_P2 = 1;                       //set P3.2 to SDA0
	P3FC2_P2 = 0;
	P3FC1_P3 = 1;                       //set P3.3 to SCL0
	P3FC2_P3 = 0;
*/


//====== I2C0 configuration ======
	PCKEN2_I2C0 = 1;                    // enable I2C0
	SBI0CR1_SCK = SCK_HI_135_LO_138;    // time before the release of the SCL pin in the slave mode
	
	SBI0CR1_ACK = 1;                    // generate the clocks for an acknowledge signal and an interrupt request when the data transfer is finished (acknowledgement mode)
	SBI0CR1_BC = 0x00;                  // number of clocks for data transfer = 8 + 1 = 9

	//--- I2C0 is slave in this example
	//------------bit7 | bit6 | bit5 | bit4  |  bit3  | bit2 | bit1~bit0 ----
	//--- SBI0CR2 <MST>, <TRX>, <BB>, <PIN> and <SBIM>  <NA>   <SWRST>   ----
	//--- <MST>   1 : Master
	//            0 : Slave
	//--- <TRX>   1 : Transmitter   ; 0: Receiver
	//--- <BB>    1 : Generate the start condition (when MST, TRX and PIN are "1")
	//            0 : Generate the stop condition (when MST, TRX and PIN are "1")
	//--- <PIN>	  1 : Cancel interrupt service request
	//
	//--- <SBIM>  1 : 1 : Serial bus interface mode ;  0 : Port mode
	//--- <SWRST> The software reset starts by first writing "10" and next writing "01"  
	//------------------------------------------------------------------------	
	//---set slave address in slave mode
	I2C0AR = SLAVE_ADDR << 1;			// set slave address SA[7:1]
	
	//---set SBICR2<PIN> , <SBIM> in slave mode
	SBI0CR2 = 0x18;						// Slave mode 0001 1000
	
	
}


//*********************************************************************
// name			: I2C0_IntEnable()
// brief		: enable I2C0 interrupt
// note			: 
//*********************************************************************
void I2C0_IntEnable(void)
{
	__ASM("DI");						// enable all maskable interrupts
	IFR_I2C0 = 0;                       // clear interrupt flag
	IER_I2C0 = 1;	                    // enable I2C0 interrupt(Enable) 
	__ASM("EI");						// enable all maskable interrupts
}


//*********************************************************************
// name			: I2C0_IntDisable() 
// brief		: disable I2C0 interrupt
// note			: 
//*********************************************************************
void I2C0_IntDisable(void)
{
	__ASM("DI");						// enable all maskable interrupts
	IFR_I2C0 = 0;                       // clear interrupt flag
	IER_I2C0 = 0;	                    // disable I2C0 interrupt 
	__ASM("EI");						// enable all maskable interrupts
}



//*********************************************************************
// name			: I2C1_Init()
// brief		: I2C1 initialization
// note			: set I2C1 pins and transfer mode
//			  *set I2C1 to master mode in this example
//*********************************************************************

void I2C1_Init(){
	
	
//====== I2C1 pin configuration initialization ======

/*
	//--SDA1 (P0.0) ; SCL1 (P0.2)-------
	FSELR0 = (FSELR0&0xF0)|0x03;		// set function selection register 0   
	PCSELR1_I2C1 = 0x00;				// set peripheral channel selection register
    
	P0FC1_P0 = 1;                       // set P0.0 to SDA1
	P0FC2_P0 = 0;
	P0FC1_P2 = 1;                       // set P0.2 to SCL1
	P0FC2_P2 = 0;
*/



	//--SDA1 (P2.1) ; SCL1 (P2.2)-------
	FSELR2 = (FSELR2&0xF0)|0x00;		// set function selection register 2   
	PCSELR1_I2C1 = 0x01;				// set peripheral channel selection register
    
	P2FC1_P1 = 1;                       // set P2.1 to SDA1
	P2FC2_P1 = 0;
	P2FC1_P2 = 1;                       // set P2.2 to SCL1
	P2FC2_P2 = 0;

	


//====== I2C1 configuration ======
	PCKEN2_I2C1 = 1;                    // enable I2C1
	SBI1CR1_SCK = SCK_HI_135_LO_138;    // time before the release of the SCL pin in the slave mode
	
	SBI1CR1_ACK = 1;                    // generate the clocks for an acknowledge signal and an interrupt request when the data transfer is finished (acknowledgement mode)
	SBI1CR1_BC = 0x00;                  // number of clocks for data transfer = 8 + 1 = 9	
	

	
	//--- I2C0 is master in this example
	//------------bit7 | bit6 | bit5 | bit4  |  bit3  | bit2 | bit1~bit0 ----
	//--- SBI1CR2 <MST>, <TRX>, <BB>, <PIN> and <SBIM>  <NA>   <SWRST>   ----
	//--- <MST>   1 : Master
	//            0 : Slave
	//--- <TRX>   1 : Transmitter   ; 0: Receiver
	//--- <BB>    1 : Generate the start condition (when MST, TRX and PIN are "1")
	//            0 : Generate the stop condition (when MST, TRX and PIN are "1")
	//--- <PIN>	  1 : Cancel interrupt service request
	//
	//--- <SBIM>  1 : 1 : Serial bus interface mode ;  0 : Port mode
	//--- <SWRST> The software reset starts by first writing "10" and next writing "01"  
	//------------------------------------------------------------------------	
	//---set SBICR2<MST> , <PIN> , <SBIM> in master mode
	SBI1CR2 = 0xC8;						// Master mode 1100 1000
	//SBI1CR2 = SBICR2_MST|SBICR2_PIN|SBICR2_SBIM;  // Master mode
	IFR_I2C1 = 0;                       // clear interrupt flag                     
	IER_I2C1 = 0;                       // disable I2C1 interrupt

}


//*********************************************************************
// name			: I2C1_IntEnable()
// brief		: enable I2C1 interrupt
// note			: 
//*********************************************************************
void I2C1_IntEnable(void)
{
	__ASM("DI");						// enable all maskable interrupts
	IFR_I2C1 = 0;                       // clear interrupt flag
	IER_I2C1 = 1;	                    // enable I2C1 interrupt
	__ASM("EI");						// enable all maskable interrupts
}


//*********************************************************************
// name			: I2C1_IntDisable() 
// brief		: disable I2C1 interrupt
// note			: 
//*********************************************************************
void I2C1_IntDisable(void)
{
	__ASM("DI");						// enable all maskable interrupts
	IFR_I2C1 = 0;                       // clear interrupt flag
	IER_I2C1 = 0;	                    // disable I2C1 interrupt 
	__ASM("EI");						// enable all maskable interrupts	
}


//*********************************************************************
// name			: I2C_Start() [Master mode]
// brief		: I2C generate the start condition
// note			: i2c : i2c : I2Cx selection
//                  I2C_0, I2C_1
//            adr : slave address
//*********************************************************************
//SBI0CR2/SBI1CR2
//-----bit7 | bit6 | bit5 | bit4  |  bit3  | bit2 | bit1~bit0 ---------
//---  <MST>, <TRX>, <BB>, <PIN> and <SBIM>  <NA>   <SWRST>   ---------
//--- <MST>   1 : Master
//            0 : Slave
//--- <TRX>   1 : Transmitter   ; 0: Receiver
//--- <BB>    1 : Generate the start condition (when MST, TRX and PIN are "1")
//            0 : Generate the stop condition (when MST, TRX and PIN are "1")
//--- <PIN>	  1 : Cancel interrupt service request
//
//--- <SBIM>  1 : 1 : Serial bus interface mode ;  0 : Port mode
//--- <SWRST> The software reset starts by first writing "10" and next writing "01"  
//----------------------------------------------------------------------


void I2C_Start(unsigned char i2c,unsigned char adr){
	
	if(i2c==I2C_0){
		unsigned char tWait_I2C0 = 0;		// waiting counter if no response
		while(SBI0SR_BB);					// SBI0SR_BB: Bus status confirms is free ; 1 :busy  0: free

// 		SBI0CR1 = 0x10;						
// 		//****** Write data to the registers before the start condition is generated ******	
// 		
// 	
 		SBI0DBR = adr;						// set slave address
// 		SBI0CR2 = 0xf8;						// Write "1" to SBI0CR2<MST>, <TRX>, <BB>, <PIN>  and <SBIM>to "1"
// 											// 1111 1000
// 											// set I2C to master mode, transmitter, generating start condition, cancel interrupt service request and serial bus interface mode
		SBI0CR2 = SBICR2_MST|SBICR2_TX|SBICR2_BB|SBICR2_PIN|SBICR2_SBIM;
		
 		while(SBI0SR_PIN){};             	// wait for releasing interrupt service request
 		
 		
// 		// --- check last  received bit <LRB> is 1---
// 		while(SBI0SR_LRB = 1){					
// 				tWait_I2C0++;				// no response and increase counting
// 				if( tWait_I2C0>250 ){		// if there is no response for too long, the loop will be broken		
// 					I2C_Stop(i2c);				// I2C stop -> release SCK
// 					break;
// 				}	
// 		}
		
	}
	
	else if(i2c==I2C_1){
		unsigned char tWait_I2C1 = 0;		// waiting counter if no response
		while(SBI1SR_BB);					// SBI1SR_BB: Bus status confirms is free ; 1 :busy  0: free

// 		SBI1CR1 = 0x10;						
// 		//****** Write data to the registers before the start condition is generated ******	
// 	
 		SBI1DBR = adr;						// set slave address
// 		SBI1CR2 = 0xf8;						// Write "1" to SBI1CR2<MST>, <TRX>, <BB>, <PIN>  and <SBIM>to "1"
// 											// 1111 1000
// 											// set I2C to master mode, transmitter, generating start condition, cancel interrupt service request and serial bus interface mode
		
		SBI1CR2 = SBICR2_MST|SBICR2_TX|SBICR2_BB|SBICR2_PIN|SBICR2_SBIM;

 		while(SBI1CR2_PIN){};             	// wait for releasing interrupt service request
 		
// 		// --- check last  received bit <LRB> is 1---
// 		while(SBI1SR_LRB = 1){					
// 				tWait_I2C1++;				// no response and increase counting
// 				if( tWait_I2C1>250 ){		// if there is no response for too long, the loop will be broken		
// 					I2C_Stop(i2c);		    // I2C stop -> release SCK
// 					break;
// 				}	
// 		}
// 

		
 	}
	
}


//*********************************************************************
// name			: I2C_ReStart() [Master mode]
// brief		: I2C generate the restart condition
// note			: i2c : i2c : I2Cx selection
//                  I2C_0, I2C_1
//            adr : slave address
//*********************************************************************
void I2C_ReStart(unsigned char i2c,unsigned char adr){
	if(i2c==I2C_0)
	{
// 		SBI0CR2 = 0x18;						// Write "1" to SBI0CR2<PIN>  and <SBIM>to "1"
// 											// 0001 1000
// 											// set I2C to slaver mode, receiver, generating stop condition, cancel interrupt service request and port mode
		SBI0CR2 = SBICR2_PIN|SBICR2_SBIM;
		while(SBI0SR_BB == 1);				// SBI0SR_BB: Bus status confirms is free ; 1 :busy  0: free	
		while(SBI0SR_LRB == 0);				// wait <LRB>=1
	}
	else if(i2c==I2C_1)
	{
// 		SBI1CR2 = 0x18;						// Write "1" to SBI1CR2<PIN>  and <SBIM>to "1"
// 											// 0001 1000
// 											// set I2C to slaver mode, receiver, generating stop condition, cancel interrupt service request and port mode
		SBI1CR2 = SBICR2_PIN|SBICR2_SBIM;
		while(SBI1SR_BB == 1);				// SBI1SR_BB: Bus status confirms is free ; 1 :busy  0: free	
		while(SBI1SR_LRB == 0);				// wait <LRB>=1
	}
	I2C_Start(i2c,adr);                 	// transmit the start condition and slave address again

}


//*********************************************************************
// name			: I2C_Stop() [Master mode]
// brief		: I2C generate the stop condition
// note			: i2c :i2c : I2Cx selection
//                 I2C_0, I2C_1
//*********************************************************************
//SBI0CR2/SBI1CR2
//-----bit7 | bit6 | bit5 | bit4  |  bit3  | bit2 | bit1~bit0 ---------
//---        <MST>, <TRX>, <BB>, <PIN> and <SBIM>  <NA>   <SWRST>   ---
//--- <MST>   1 : Master
//            0 : Slave
//--- <TRX>   1 : Transmitter   ; 0: Receiver
//--- <BB>    1 : Generate the start condition (when MST, TRX and PIN are "1")
//            0 : Generate the stop condition (when MST, TRX and PIN are "1")
//--- <PIN>	  1 : Cancel interrupt service request
//
//--- <SBIM>  1 : 1 : Serial bus interface mode ;  0 : Port mode
//--- <SWRST> The software reset starts by first writing "10" and next writing "01"  
//----------------------------------------------------------------------

void I2C_Stop(unsigned char i2c){
	if(i2c==I2C_0){
// 	SBI0CR2 = 0xd8;			   				  // Set SBI0CR2<MST>, <TRX> and <PIN> to "1" and SBI0CR2<BB> to "0"
// 											  // 1101 1000
// 											  // set I2C to master mode, transmitter, generating stop condition, cancel interrupt service request and port mode
	SBI0CR2 = SBICR2_MST|SBICR2_TX|SBICR2_PIN|SBICR2_SBIM;
	
	while(SBI0SR_BB);              		  // wair for Bus free,SBI0SR_BB: Bus status confirms is free ; 1 :busy  0: free
	}
	
	else if(i2c==I2C_1){
// 		SBI0CR2 = 0xd8;			   			  // Set SBI0CR2<MST>, <TRX> and <PIN> to "1" and SBI0CR2<BB> to "0"
// 											  // 1101 1000
// 											  // set I2C to master mode, transmitter, generating stop condition, cancel interrupt service request and port mode
	SBI1CR2 = SBICR2_MST|SBICR2_TX|SBICR2_PIN|SBICR2_SBIM;
	while(SBI1SR_BB);              	 	  // wair for Bus free,SBI1SR_BB: Bus status confirms is free ; 1 :busy  0: free
	}
}

//*********************************************************************
// name			: I2C_TXOneByte() [Master mode]
// brief		: I2C transmits a single byte data
// note			: i2c : I2Cx selection
//                   I2C_0, I2C_1
//            TxBuf: data to be transmit
//*********************************************************************
void I2C_TXOneByte(unsigned char i2c,unsigned char TxBuf){
	
	if(i2c==I2C_0)
	{
		SBI0DBR = TxBuf;
		SBI0CR2 = SBICR2_MST|SBICR2_TX|SBICR2_BB|SBICR2_PIN|SBICR2_SBIM;
		while(SBI0CR2_PIN);             // wait for tranfer completion
	}
	else if(i2c==I2C_1)
	{
		SBI1DBR = TxBuf;
		SBI1CR2 = SBICR2_MST|SBICR2_TX|SBICR2_BB|SBICR2_PIN|SBICR2_SBIM;
		while(SBI1CR2_PIN);             // wait for tranfer completion
	}
}



//*********************************************************************
// name			: I2C_RXOneByte() [Master mode]
// brief		: I2C receive a single byte data
// note			: i2c : I2Cx selection
//                   I2C_0, I2C_1
//            return: received data
//*********************************************************************
unsigned char I2C_RXOneByte(unsigned char i2c){
	if(i2c==I2C_0)
	{
		SBI0CR2 = SBICR2_MST|SBICR2_RX|SBICR2_BB|SBICR2_PIN|SBICR2_SBIM;
		SBI0DBR = 0x00;                 // transmit dummy datafer
		while(SBI0CR2_PIN == 1);        
		return SBI0DBR;
	}
	else if(i2c==I2C_1)
	{
		SBI1CR2 = SBICR2_MST|SBICR2_RX|SBICR2_BB|SBICR2_PIN|SBICR2_SBIM;
		SBI1DBR = 0x00;                 // transmit dummy datafer
		while(SBI1CR2_PIN == 1);
		return SBI1DBR;
	}
	else
		return 0;
}


//*********************************************************************
// name			: IsAck()
// brief		: check ACK or NACK
// note			: i2c : I2Cx selection
//                   I2C_0, I2C_1
//            return : 1 : NACK   0: ACK
//*********************************************************************
unsigned char IsAck(unsigned char i2c){
	if(i2c==I2C_0)
	{
		return SBI0SR_LRB;
	}
	else if(i2c==I2C_1)
	{
		return SBI1SR_LRB;
	}
	return I2C_NACK;
}


//*********************************************************************
// name			: I2C_AckEnable()
// brief		: generate I2C 1-bit ACK signal or not
// note			: i2c : I2Cx selection
//                   I2C_0, I2C_1
//            enable : 1 : enable    0 : disable
//*********************************************************************
void I2C_AckEnable(unsigned char i2c, unsigned char enable){
	if(i2c==I2C_0)
	{
		SBI0CR1_ACK = enable;				// generate I2C0 1-bit ACK signal or not
	}
	else if(i2c==I2C_1)
	{
		SBI1CR1_ACK = enable;				// generate I2C1 1-bit ACK signal or not
	}
}


//*********************************************************************
// name			: I2C_BufCount()
// brief		: number of data bits
// note			: i2c : I2Cx selection
//                   I2C_0, I2C_1
//            bufcount : 
//                   BC_8bit      0x00
//                   BC_1bit      0x01
//                   BC_2bit      0x02
//                   BC_3bit      0x03
//                   BC_4bit      0x04
//                   BC_5bit      0x05
//                   BC_6bit      0x06
//                   BC_7bit      0x07 
//*********************************************************************
void I2C_BufCount(unsigned char i2c, unsigned char bufcount){
	if(i2c==I2C_0)
	{
		SBI0CR1_BC = bufcount;
	}
	else if(i2c==I2C_1)
	{
		SBI1CR1_BC = bufcount;
	}
}


//*********************************************************************
// name			: I2C_ReadState()
// brief		: read I2C status
// note			: i2c : I2Cx selection
//                   I2C_0, I2C_1

//*********************************************************************
unsigned char I2C_ReadState(unsigned char i2c){
	if(i2c==I2C_0)
	{
		return SBI0SR;						// read I2C0 status
	}
	else if(i2c==I2C_1)
	{
		return SBI1SR;						// read I2C1 status
	}
	else 
		return 0;
}


//*********************************************************************
// name			: I2C_SetBufvalue()
// brief		: set I2C buffer data
// note			: i2c : I2Cx selection
//                   I2C_0, I2C_1
//            data:  buffer data
//*********************************************************************
void I2C_SetBufvalue(unsigned char i2c,unsigned char data){
	if(i2c==I2C_0)
	{
		SBI0DBR = data;
	}
	else if(i2c==I2C_1)
	{		
		SBI1DBR = data;
	}
}

//*********************************************************************
// name			: I2C_ReadBufvalue()
// brief		: read I2C buffer data
// note			: i2c :   i2c : I2Cx selection
//                    I2C_0, I2C_1
//            return: buffer data
//*********************************************************************
unsigned char I2C_ReadBufvalue(unsigned char i2c){
	if(i2c==I2C_0){
		return SBI0DBR;
	}
	else if(i2c==I2C_1){
		return SBI1DBR;
	}
	else{
		return 0;
	}
}


//*********************************************************************
// name			: I2C_FinishSlaveTx() [Slave mode]
// brief		: I2C Slave transfer completion and release the I2C bus
// note			: i2c : I2Cx selection
//                   I2C_0, I2C_1
//*********************************************************************
void I2C_FinishSlaveTx(unsigned char i2c){
	if(i2c==I2C_0)
	{
		SBI0CR2 = 0x58;
		SBI0CR2 = 0x18;
	}
	else if(i2c==I2C_1)
	{
		SBI1CR2 = 0x58;
		SBI1CR2 = 0x18;
	}

}


//*********************************************************************
// name			: I2C_Reset()
// brief		: I2C software reset. reset SBI0CR1,I2C0AR,SBI0CR2, not reset SBI0CR2<SBIM>
// note			: i2c : I2Cx selection
//                   I2C_0, I2C_1
//*********************************************************************
void I2C_Reset(unsigned char i2c){
	if(i2c==I2C_0)
	{
		SBI0CR2 = 0x1A;
		SBI0CR2 = 0x19;
	}
	else if(i2c==I2C_1)
	{
		SBI1CR2 = 0x1A;
		SBI1CR2 = 0x19;
	}

}


//*********************************************************************
// name			: __interrupt IntI2C0()
// brief		: I2C0 interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntI2C0(void)
{
	__asm("NOP");						// no operation

	I2C_Slave_Int(I2C_0,&i2c_slave);
}


//*********************************************************************
// name			: __interrupt IntI2C1()
// brief		: I2C1 interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntI2C1(void)
{
	__asm("NOP");						// no operation

	//LED2_out = ~LED2_out;               // LED2(P1.1) inverted, ʾI2C1ж˳
}


//*********************************************************************
// name			: I2C_Master_Write()
// brief		: 
// note			: i2c : I2Cx selection
//                   I2C_0, I2C_1
//*********************************************************************
unsigned char I2C_Master_Write(unsigned char i2c,I2C_MasterPacket* pi2c){
	unsigned char cnt;
	
	I2C_Start(i2c,(unsigned char)(pi2c->adr<<1)|I2C_Write);
	if(IsAck(i2c)==I2C_NACK){
		I2C_Stop(i2c);
		return 0;
	}		
	I2C_TXOneByte(i2c,pi2c->cmd);			// I2C transmits one byte
	if(IsAck(i2c)==I2C_NACK){
		I2C_Stop(i2c);
		return 0;
	}	
	for(cnt=0;cnt<pi2c->len;cnt++){
		I2C_TXOneByte(i2c,pi2c->buf[cnt]);  // I2C transmits one byte
		
		if(IsAck(i2c)==I2C_NACK){
		    I2C_Stop(i2c);
		    return 0;
	    }	
	}
	I2C_Stop(i2c);
	return 1;
}


//*********************************************************************
// name			: I2C_Master_Read()
// brief		: 
// note			: i2c : I2Cx selection
//                   I2C_0, I2C_1
//*********************************************************************
unsigned char I2C_Master_Read(unsigned char i2c,I2C_MasterPacket* pi2c){
	unsigned char cnt;
	
	I2C_Start(i2c,(unsigned char)(pi2c->adr<<1)|I2C_Write); // I2C generate the start condition
	
	if(IsAck(i2c)==I2C_NACK){					  // check ACK or NACK, if NACK, I2C generate the stop condition
		I2C_Stop(i2c);
		return 0;
	}		
	I2C_TXOneByte(i2c,pi2c->cmd);				  // I2C transmits one byte ( send command )
	
	if(IsAck(i2c)==I2C_NACK){					  // check ACK or NACK, if NACK, I2C generate the stop condition
		I2C_Stop(i2c);
		return 0;
	}
	
	I2C_ReStart(i2c,(unsigned char)(pi2c->adr<<1)|I2C_Read);// I2C generate the restart condition
	
	if(IsAck(i2c)==I2C_NACK){					  // check ACK or NACK, if NACK, I2C generate the stop condition
		I2C_Stop(i2c);
		return 0;
	}
	
	for(cnt=0;cnt<pi2c->len;cnt++){
		if(cnt == pi2c->len-1){
			I2C_AckEnable(i2c,0);                 // ACK disable 
			I2C_BufCount(i2c,BC_8bit);            // set to receive 8-bit data
			pi2c->buf[cnt] = I2C_RXOneByte(i2c);
			I2C_BufCount(i2c,BC_1bit);            // transmit 1-bit ACK
			I2C_TXOneByte(i2c,0xFF);              // write NACK to 0xFF
			
			I2C_AckEnable(i2c,1);                 // ACK enable 
			I2C_BufCount(i2c,BC_8bit);            // receive 8-bit data
		}
		else{
    		pi2c->buf[cnt] = I2C_RXOneByte(i2c); // receive 8-bit data + 1-bit ACK
		}		
	}
	I2C_Stop(i2c);								 // I2C generate the stop condition
	return 1;
}


//*********************************************************************
// name			: I2C_Slave_Int()
// brief		: 
// note			: i2c : I2Cx selection
//                   I2C_0, I2C_1
//*********************************************************************
void I2C_Slave_Int(unsigned char i2c,I2C_SlavePacket* pi2c){
	unsigned char tmp;
	if(I2C_ReadState(i2c)&SBISR_TX){
		// transmit
		if(I2C_ReadState(i2c)&SBISR_AAS){
			// match slave address
			pi2c->status = I2C_Slave_read;
			// check command
			if(pi2c->cmd == 0x09){
				pi2c->buf[0] = 0x12;
				pi2c->buf[1] = 0x34;
				pi2c->len = 2;                // set data length to 2
			}
			else if(pi2c->cmd == 0x0A){
				pi2c->buf[0] = 0x55;
				pi2c->buf[1] = 0xAA;
				pi2c->len = 2;                // set data length to 2
			}
			else{
				pi2c->len = 0;                // not supprt command, set data length to 0
			}
			
			pi2c->pointer= 0;
			if(pi2c->pointer >= pi2c->len){   // check data need to be transmitted or not
				I2C_SetBufvalue(i2c,0xFF);    // transmit dummy data
			}
			else{
				I2C_SetBufvalue(i2c,pi2c->buf[pi2c->pointer]);	
			}				
		}
		else{
			if(I2C_ReadState(i2c)&SBISR_LRB){
				I2C_SetBufvalue(i2c,0x00);		 // transmit dummy data
				I2C_FinishSlaveTx(i2c);
			}
			else{
				if(pi2c->pointer >= pi2c->len){
					I2C_SetBufvalue(i2c,0xFF);
				}
				else{
				    I2C_SetBufvalue(i2c,pi2c->buf[pi2c->pointer]);
				}
			}			
		}
		pi2c->pointer++;
	}
	else{
		// receive
		if(I2C_ReadState(i2c)&SBISR_AAS){
			// match slave address
			pi2c->pointer = 0;
			pi2c->status = I2C_Slave_adr;
			tmp = I2C_ReadBufvalue(i2c);
			I2C_SetBufvalue(i2c,0x00);		     // transmit dummy data						
		}
		else{
		    if(pi2c->status == I2C_Slave_adr){
                pi2c->cmd = I2C_ReadBufvalue(i2c);
				pi2c->status = I2C_Slave_cmd;
				I2C_SetBufvalue(i2c,0x00);		 // transmit dummy data
			}
			else{
				pi2c->buf[pi2c->pointer] = I2C_ReadBufvalue(i2c);
				pi2c->pointer++;
				I2C_SetBufvalue(i2c,0x00);		 // transmit dummy data
			}
		}
	}
}


