//*************************************************************************************
// file			: SQ7613_ADC_LVCMP.c
// version		: V1.0 2020/07/30
// brief		: ADC_LVCMP related functions
// note			: ADC_LVCMP related functions are gathered in this subroutine
//*************************************************************************************

#include "main.h"

unsigned char ADC_Int_Flag;			      // ADC interrupt flag
unsigned char ADCLVCMP_Int_Flag =0;	      // interrupt flag of ADC if level comparison condition is met

unsigned char r_tempData[2];
long ain6_value;                       		  // store the ADC value of AIN6
long ain6_mv;                          		  // ADC value converted to voltage value (in mv)

//*********************************************************************
// name			: ADC_Init()
// brief		: ADC initialization
// note			: 
//*********************************************************************
void ADC_Init(){

//====== ADC pin configuration initialization (select the analog signal input channel for AD conversion) ======
// SQ7613 has 7 pins available from AIN4 to AIN10---------------------
// PINC selection

/*
	//--- AIN4 ---//	
	P1FC1_P3 = 1;                       // P1.3 is set to AIN4
	P1FC2_P3 = 1; 
*/



/*
	//--- AIN5 ---//	
	P1FC1_P2 = 1;                       // P1.2 is set to AIN5
	P1FC2_P2 = 1; 
*/	
	
	
	

	//--- AIN6 ---//					[this pin is selected for this example]
	P1FC1_P1 = 1;                       // P1.1 is set to AIN6
	P1FC2_P1 = 1; 
	



/*
	//--- AIN7 ---//	
	P1FC1_P0 = 1;                       // P1.0 is set to AIN7
	P1FC2_P0 = 1; 
*/	



/*
	//--- AIN8 ---//	*Note:  another function of P5.3 pin is ADC reference voltage input (ADC_VREF) = VDD
	P5FC1_P3 = 1;                       // P5.3 is set to AIN8
	P5FC2_P3 = 1; 
*/	



/*
	//--- AIN9 ---//	
	P5FC1_P2 = 1;                       // P5.2 is set to AIN9
	P5FC2_P2 = 1; 
*/	



/*
	//--- AIN10 ---//	
	P5FC1_P1 = 1;                       // P5.1 is set to AIN10
	P5FC2_P1 = 1; 
*/	
   
//====== Function Enable ======          // this must be enabled first, the relevant staging settings later to be useful
	PCKEN6_ADC = 1;                       // enable ADC peripheral clock 

//====== parameter setting ======
// Set ADC Convert Speed
    ADCCKDIV_ADCKDIV = 2;	              // fsysclk/(2^2)

// after ADEN is set, the ADC needs to wait for clock warm up to allow the user to set ADCSR_ADRS to start the AD conversion
	ADCCR0_ADEN  = 1;                     // enable ADC
	ADCCR0_IRFEN = 0x01;                  // set VDDA as Vref
    //ADCCR0_INTLV  = 1;                  // interrupt interval is set to wait until all AIN scans are completed before an interrupt is generated

	ADCCR0_AUTOPD = 0;                    // disable automatic power down after each ADC conversion
    
// set ADC operating mode
    ADCCR0_AMD   = 0x01;                  // single mode

// ADC data buffer automatic overwrite
    //ADCSR_BUFOVR = 1;                   // ADC data buffer automatic overwrite

// set event	
	ADCCR1_EVSEL = 0x00;                  // set event to ADC convertion
	
// Multiple AIN channel scanning
	ADCSCAN0 = 0x40;                      // set ADC scan channel to AIN6

}

//*********************************************************************
// name			: ADCLVLCMP_Init()
// brief		: ADC level comparison initialization
// note			: 
// 
// lvcmpsel : CMP_Disable	       0x0   // disable ADC level comparison
//            CMP_ADCLLV           0x1   // ADC value < ADCLLV             ** CMP ADCLLV ** 
//            CMP_ADCHLV           0x2   // ADC value > ADCHLV             ** CMP ADCHLV ** 
//            CMP_ADCL_BETWEEN     0x3   // ADCLLV > ADC value > ADCHLV    ** CMP ADCLLV and ADCHLV **
//*********************************************************************
void ADCLVLCMP_Init(unsigned char lvcmpsel){
    
    ADCLV_LVCMP  = lvcmpsel;             // set ADC level comparison setting
    
    //ADCLV_LVINTVL = 0;    			 // all samples were compared 
	ADCLV_LVINTVL = 1;    				 // level comparison specified by LVSEL   
	ADCLV_LVSEL   = AIN6;                // Specify ADC channel : set AIN6 for level comparison


}



//*********************************************************************
// name			: ADCLVLCMP_SetTh()
// brief		: set threshold of ADC level comparison (0x0000~0x0FFF)
//
// note			: set low threshold of ADC level comparison
//            set high threshold of ADC level comparison
//*********************************************************************
void ADCLVLCMP_SetTh(uchar adcllv_Th,uchar adchlv_Th){
/*
	ADCLLV = 0xFA0;
	ADCHLV = 0x3E8;
*/ 

    ADCLLV = 4000;          // if ADC < ADCLLV, then ADCSR_LVDET = 1
	ADCHLV = 1000;          // if ADC > ADCHLV, then ADCSR_LVDET = 1

}


//*********************************************************************
// name			: ADC_Stop_and_Disable()
// brief		: disable ADC operation and forcibly stop ADC operation
// note			: 
//*********************************************************************
void ADC_Stop_and_Disable()
{
	ADCCR0_AMD  = 0;                      // disable ADC operation and forcibly stop ADC operation
    ADCCR0_ADEN = 0;                      // disable ADC    
}


//*********************************************************************
// name			: ADC_IntEnable()
// brief		: enable ADC interrupt
// note			: 
//*********************************************************************
void ADC_IntEnable(){
	__ASM("DI");

	IER_ADC = 1;                        // enable ADC interrupt
	ADCCR0_INTEN = 1;                   // enable AD conversion finished interrupt
	
	__ASM("EI");
}


//*********************************************************************
// name			: ADC_IntDisable()
// brief		: disable ADC interrupt
// note			: 
//*********************************************************************
void ADC_IntDisable(){
	__ASM("DI");

	IER_ADC  = 0;                         // disable ADC interrupt
	IFR_ADC  = 0;	                      // disable IRQ
	ADCCR0_INTEN = 0;                     // disable AD conversion finished interrupt
	
	__ASM("EI");
}



//*********************************************************************
// name			: ADCLVLCMP_IntEnable()
// brief		: enable ADC level comparison interrupt
// note			: 
//*********************************************************************
void ADCLVLCMP_IntEnable(){
    __ASM("DI");
    
    IER_ADC = 1;                          // enable ADC interrupt
    ADCCR2_LVINTEN = 1;                   // enable ADC level comparison
    
    __ASM("EI");
}



//*********************************************************************
// name			: ADC_Convert_Single()
// brief		: ADC conversion in single mode
// note			: 
//*********************************************************************
void ADC_Convert_Single(){
	while(!ADCSR_ADRDY);                  // wait for ADC to be ready
	ADCSR_ADRS = 1;                       // start ADC conversion
	

}



//*********************************************************************
// name			: ADC_ConvertDone()
// brief		: ADC conversion completed
// note			: 
//*********************************************************************
char ADC_ConvertDone(){
	if(ADCSR_EOCF){
		return 1;                         // if ADC conversion completed, return 1
	}
	else{
		return 0;                         // if ADC conversion not completed, return 0
	}
}




//*********************************************************************
// name			: ADC_Read_Value()
// brief		: readout ADC value    [for single mode]
// note			: chsel : AIN setting value  
// 						AIN4	       0x4
// 						AIN5	       0x5
// 						AIN6	       0x6
// 						AIN7	       0x7
// 						AIN8	       0x8
// 						AIN9	       0x9
// 						AIN60	   	   0xA
//*********************************************************************
long ADC_Read_Value(unsigned char chsel){
	unsigned int adc_value = 0;
	
	ADCCHSEL = chsel;                   // select ADC input channel
	
	adc_value |= ADCDRL;                // when reading ADC value,  read low byte first and then read high byte
	
	adc_value |= (unsigned int)ADCDRH<<8;
	return (long)adc_value;
	
}



//*********************************************************************
// name			: __interrupt IntADC()
// brief		: ADC interrupt subroutine
// note			:
//*********************************************************************
void __interrupt IntADC(void){
	__asm("NOP");                         // no operation
    
	
	for(i=0;i<6000;i++);
	
	//--- If ADC meets level comparison condition, interrupt is generated ---
	if (ADCSR_LVDET){			          // interrupt is generated when the level comparison condition is detected
		ADCLVCMP_Int_Flag = 1;			  // the interrupt flag is set when meet the level comparison condition        
        ADCSR_EOCF = 1;		              // clear ADC conversion completion flag
        ADCSR_LVDET = 1;	              // clean flag
		
        ain6_value = ADC_Read_Value(AIN6);  			   //read the ADC value of AIN6
	    ain6_mv = ain6_value*ADC_RefScale(3090000)/1000;   // ADC value to voltage value (unit:mV)
/*
		ADCCHSEL_CHSEL = 6;	              // set Ain6 and then read ADCDRL and ADCDRH
        r_tempData[0] = ADCDRL;	          // read Low and then High
		r_tempData[1] = ADCDRH;	          // read Low and then High
*/       


 
		LED1_out = ~LED1_out;             // LED1 (P2.0) reverse
		__asm("NOP");                     // no operation
		__asm("NOP");                     // no operation, the break point can be set here

     }


	//--- ADC conversion completed ---
/*
	if (ADCSR_EOCF){			          // ADC conversion completed, interrupt is generated
		ADC_Int_Flag = 1;				  // set ADC interrupt flag
        ADCSR_EOCF = 1;		              // clear ADC conversion completion flag
	}
*/	



}


