//*************************************************************************************
// file			: SQ7515_UART.c
// version		: V1.0 2022/05/18
// brief		: UART related functions
// note			: UART related functions are gathered in this subroutine
//**************************************************************************************
#include "main.h"
unsigned char RT;
unsigned char DR;
unsigned char rbuf;                         // buffer for RX1 reveived value
unsigned char RX1_Int_Flag;                 // IntRX1 interrupt flag	
unsigned char rx1_int_count = 0;            // number of RX1 interrupts

uchar sentBuf[]={0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x76, 0x15, 0x5A, 0xA5};	
uchar reciveBuf[]={0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};

//*********************************************************************
// name			: UART0_Init()
// brief		: UART0 initialization
// param		: - parityCheck : PARITY_NONE  no parity
//                          : PARITY_ODD   odd-numbered parity 
//                          : PARITY_EVEN  even-numbered parity  
//
// retval		: none
// note			: set serial data output in pins and UART related parameters
//*********************************************************************
void UART0_Init(uchar parityCheck){

  //====== UART0 pin configuration initialization ======

	
    //--TX0 (P0.1) ; RX0 (P0.0)-------
    FSELR0 = (FSELR0&0xF0)|0x00;        // set function selection register 0, function group selection b00
    PCSELR0_UART0 = 0x00;               // select peripheral function channel

    P0FC_P1_TX0;                        // set P0.1 to TX0
    P0FC_P0_RX0;                        // set P0.0 to RX0
		
    P0PU_P1 = 0;                        // disable TX0 built-in pull-up resistor
        
  /*	
    //--TX0 (P0.0) ; RX0 (P0.1)-------        
    FSELR0 = (FSELR0&0xF0)|0x01;        // set function selection register 0, function group selection b01
    PCSELR0_UART0 = 0x00;               // select peripheral function channel

    P0FC_P0_TX0;                        // set P0.0 to TX0
    P0FC_P1_RX0;                        // set P0.1 to RX0
        
    P0PU_P0 = 0;                        // disable TX0 built-in pull-up resistor
  */


  /*
    //--TX0 (P3.7) ; RX0 (P3.6)-------
    FSELR3 = (FSELR3&0x0F)|0x00;        // set function selection register 3, function group selection b00
    PCSELR0_UART0 = 0x01;               // select peripheral function channel

    P3FC_P7_TX0;                        // set P3.7 to TX0
    P3FC_P6_RX0;                        // set P3.6 to RX0
    
    P3PU_P7 = 0;                        // disable TX0 built-in pull-up resistor
  */


  /*
    //--TX0 (P3.6) ; RX0 (P3.7)-------
    FSELR3 = (FSELR3&0x0F)|0x10;        // set function selection register 3, function group selection b10
    PCSELR0_UART0 = 0x01;               // select peripheral function channel
        
    P3FC_P6_TX0;                        // set P3.6 to TX0
    P3FC_P7_RX0;                        // set P3.7 to RX0
        
    P3PU_P6 = 0;                        // disable TX0 built-in pull-up resistor
  */	

    //====== Function Enable ======
    PCKEN1_UART0 = 1;                   // this must be enabled first, the relevant staging settings later to be useful,UART0 Enable

    //====== parameter setting ======
    UART0CR1  = (UART0CR1|parityCheck); // set parity  

    UART0CR1_STOPBT = STOP_1BIT;        // set the bit length of the TX0 stop bit to 1 bit
    UART0CR2_STOPBR = STOP_1BIT;        // set the bit length of the RX0 stop bit to 1 bit

    // TX0 pin output selection
    UART0CR1_IRDASEL = TX_OUTPUT_UART;  // UART output
    //UART0CR1_IRDASEL = TX_OUTPUT_IRDA;  // IrDA output
      
    // transfer base clock selection
    UART0CR1_BRG = 0;                   // fcgck
      
    // dividing frequency selection
    UART0CR2_DV = DIV_BY1;              // UART0 fsysclk/1

}


//*********************************************************************
// name			: UART1_Init()
// brief		: UART1 initialization
// param		: - parityCheck : PARITY_NONE  no parity
//                          : PARITY_ODD   odd-numbered parity 
//                          : PARITY_EVEN  even-numbered parity  
//
// retval		: none
// note			: set serial data output in pins and UART related parameters
//*********************************************************************
void UART1_Init(uchar parityCheck){

  //====== UART1 pin configuration initialization ======
  
  /*  
    //--TX1 (P2.6) ; RX1 (P2.5)-------
    FSELR2 = (FSELR2&0x0F)|0x00;        // set function selection register 2, function group selection b00    
    PCSELR0_UART1 = 0x00;               // select peripheral function channel
        
    P2FC_P6_TX1;                        // set P2.6 to TX1
    P2FC_P5_RX1;                        // set P2.5 to RX1
        
    P2PU_P6 = 0;                        // disable TX1 built-in pull-up resistor
  */
	
    //--TX1 (P2.5) ; RX1 (P2.6)-------      [for this example program]    
    FSELR2 = (FSELR2&0x0F)|0x10;        // set function selection register 2, function group selection b10
    PCSELR0_UART1 = 0x00;               // select peripheral function channel

    P2FC_P5_TX1;                        // set P2.5 to TX1
    P2FC_P6_RX1;                        // set P2.6 to RX1

    P2PU_P5 = 0;                        // disable TX1 built-in pull-up resistor

  /*
    //--TX1 (P5.2) ; RX1 (P5.1)-------
    FSELR5 = (FSELR5&0xF0)|0x00;        // set function selection register 5, function group selection b00    
    PCSELR0_UART1 = 0x01;               // select peripheral function channel

    P5FC_P2_TX1;                        // set P5.2 to TX1
    P5FC_P1_RX1;                        // set P5.1 to RX1
    
    P5PU_P2 = 0;                        // disable TX1 built-in pull-up resistor
  */


  /* 
    //--TX1 (P5.1) ; RX1 (P5.2)-------
    FSELR5 = (FSELR5&0xF0)|0x01;        // set function selection register 5, function group selection b01
    PCSELR0_UART1 = 0x01;               // select peripheral function channel
	
    P5FC_P1_TX1;                        // set P5.1 to TX1
    P5FC_P2_RX1;                        // set P5.2 to RX1
    
    P5PU_P1 = 0;                        // disable TX1 built-in pull-up resistor
  */


    //====== Function Enable ======
    PCKEN1_UART1 = 1;                   // this must be enabled first, the relevant staging settings later to be useful,UART1 Enable

    //====== parameter setting ======
    UART1CR1  = (UART1CR1|parityCheck); // set parity  

	UART1CR1_STOPBT = STOP_1BIT;        // set the bit length of the TX1 stop bit to 1 bit
	UART1CR2_STOPBR = STOP_1BIT;        // set the bit length of the RX1 stop bit to 1 bit

    // TX1 pin output selection
    UART1CR1_IRDASEL = TX_OUTPUT_UART;  // UART output
    //UART1CR1_IRDASEL = TX_OUTPUT_IRDA;  // IrDA output

    // transfer base clock selection
    UART1CR1_BRG = 0;                   // fcgck
 
    // dividing frequency selection
    UART1CR2_DV = DIV_BY1;              // UART1 fsysclk/1

}


//*********************************************************************
// name			: UART2_Init()
// brief		: UART2 initialization
// param		: - parityCheck : PARITY_NONE  no parity
//                          : PARITY_ODD   odd-numbered parity 
//                          : PARITY_EVEN  even-numbered parity  
//
// retval		: none
// note			: set serial data output in pins and UART related parameters
//*********************************************************************
void UART2_Init(uchar parityCheck){

   //====== UART2 pin configuration initialization ======
	
  /*	
    //--TX2 (P0.5) ; RX2 (P0.4)-------
    FSELR0 = (FSELR0&0x0F)|0x00;        // set function selection register 0, function group selection b00    
    PCSELR0_UART2 = 0x00;               // select peripheral function channel

    P0FC_P5_TX2;                        // set P0.5 to TX2
    P0FC_P4_RX2;                        // set P0.4 to RX2
    
    //P0PU_P4 = 1;                        // enable RX2 built-in pull-up resistor
  */	


	
  /*	
    //--TX2 (P0.4) ; RX2 (P0.5)-------
    FSELR0 = (FSELR0&0x0F)|0x10;        // set function selection register 0, function group selection b10
    PCSELR0_UART2 = 0x00;               // select peripheral function channel

    P0FC_P4_TX2;                        // set P0.4 to TX2
    P0FC_P5_RX2;                        // set P0.5 to RX2
        
    //P0PU_P5 = 1;                        // enable RX2 built-in pull-up resistor
  */



  /*
    //--TX2 (P3.1) ; RX2 (P3.0)-------
    FSELR3 = (FSELR3&0xF0)|0x00;        // set function selection register 3, function group selection b00    
    PCSELR0_UART2 = 0x01;               // select peripheral function channel
    
    P3FC_P1_TX2;                        // set P3.1 to TX2
    P3FC_P0_RX2;                        // set P3.0 to RX2

    //P3PU_P0 = 1;                        // enable RX2 built-in pull-up resistor
  */


  /*
    //--TX2 (P3.0) ; RX2 (P3.1)-------   
    FSELR3 = (FSELR3&0xF0)|0x01;        // set function selection register 3, function group selection b01
    PCSELR0_UART2 = 0x01;               // select peripheral function channel
    
    P3FC_P0_TX2;                        // set P3.0 to TX2
    P3FC_P1_RX2;                        // set P3.1 to RX2

    //P3PU_P1 = 1;                        // enable RX2 built-in pull-up resistor
  */

    //====== Function Enable ======
    PCKEN1_UART2 = 1;                   // this must be enabled first, the relevant staging settings later to be useful,UART2 Enable

    //====== parameter setting ======
    UART2CR1  = (UART2CR1|parityCheck); // set parity  

    UART2CR1_STOPBT = STOP_1BIT;        // set the bit length of the TX2 stop bit to 1 bit
    UART2CR2_STOPBR = STOP_1BIT;        // set the bit length of the RX2 stop bit to 1 bit

    // TX2 pin output selection
    UART2CR1_IRDASEL = TX_OUTPUT_UART;  // UART output
    //UART2CR1_IRDASEL = TX_OUTPUT_IRDA;  // IrDA output
      
    // transfer base clock selection
    UART2CR1_BRG = 0;                     // fcgck
      
    // dividing frequency selection
    UART2CR2_DV = DIV_BY1;                // UART2 fsysclk/1

}



//*********************************************************************
// name			: UART0_Int()
// brief		: enable/disable UART0 interrupt
// param1		: - intEn     : ENABLE  enable
//                        : DISABLE disable
// param2		: - trxSelect : TX
//                        : RX
//                        : TRX
// retval		: none
// note			: it will turn on the interrupt master permission flag (IMF) after execution**
//
// example		: if only enable RX0 interrupt , then writeUART0_Int(ENABLE, RX);
//				: if only disable RX0 interrupt , then writeUART0_Int(DISABLE, RX);
//
//				: if both enable TX0 and RX0 interrupt , then writeUART0_Int(ENABLE, TRX);
//*********************************************************************
void UART0_Int(uchar intEn, uchar trxSelect){

	__asm("DI");						// enable all maskable interrupts
    
	switch(trxSelect){
	    case TX:						// enable or disable TX0 interrupt
            IER_UART0_TX = intEn;       // if intEn = ENABLE, enable TX0 interrupt
                                        // if intEn = DISABLE, disable TX0 interrupt
        break;
    
    	case RX:						// enable or disable RX0 interrupt
            IER_UART0_RX = intEn;	    // if intEn = ENABLE, enable RX0 interrupt
                                        // if intEn = DISABLE, disable RX0 interrupt
        break;

    	case TRX:                       // both enable or disable TX0 and RX0 interrupts
            IER_UART0_TX = IER_UART0_RX = intEn;
                                        // if intEn = ENABLE, both enable TX0 and RX0 interrupts
                                        // if intEn = DISABLE, both disable TX0 and RX0 interrupts
        break;
        
        default:
        break;
    }    
	__asm("EI");						// enable all maskable interrupts
}


//*********************************************************************
// name			: UART1_Int()
// brief		: enable/disable UART1 interrupt
// param1		: - intEn     : ENABLE  enable
//                        : DISABLE disable
// param2		: - trxSelect : TX
//                        : RX
//                        : TRX
// retval		: none
// note			: it will turn on the interrupt master permission flag (IMF) after execution**
//*********************************************************************
void UART1_Int(uchar intEn, uchar trxSelect){
	__asm("DI");						// enable all maskable interrupts

	switch(trxSelect){
	    case TX:						// enable or disable TX1 interrupt
            IER_UART1_TX1 = intEn;      // if intEn = ENABLE, enable TX1 interrupt
                                        // if intEn = DISABLE, disable TX1 interrupt
        break;
    
    	case RX:						// enable or disable RX1 interrupt
            IER_UART1_RX1 = intEn;	    // if intEn = ENABLE, enable RX1 interrupt
                                        // if intEn = DISABLE, disable RX1 interrupt
        break;
        
        case TRX:                       // both enable or disable TX1 and RX1 interrupts
            IER_UART1_TX1 = IER_UART1_RX1 = intEn;
                                        // if intEn = ENABLE, both enable TX1 and RX1 interrupts
                                        // if intEn = DISABLE, both disable TX1 and RX1 interrupts
        break;
        
        default:
        break;
    }    
	__asm("EI");						// enable all maskable interrupts
}


//*********************************************************************
// name			: UART2_Int()
// brief		: enable/disable UART2 interrupt
// param1		: - intEn      : ENABLE  enable
//                         : DISABLE disable
// param2		: - trxSelect  : TX
//                         : RX
//                         : TRX
// retval		: none
// note			: it will turn on the interrupt master permission flag (IMF) after execution**
//*********************************************************************
void UART2_Int(uchar intEn, uchar trxSelect){

	__asm("DI");						// enable all maskable interrupts

	switch(trxSelect){
	    case TX:						// enable or disable TX2 interrupt
            IER_UART2_TX2 = intEn;      // if intEn = ENABLE, enable TX2 interrupt
                                        // if intEn = DISABLE, disable TX2 interrupt
        break;
    
    	case RX:						// enable or disable RX2 interrupt
            IER_UART2_RX2 = intEn;	    // if intEn = ENABLE, enable RX2 interrupt
                                        // if intEn = DISABLE, disable RX2 interrupt
        break;
        
        case TRX:                       // both enable or disable TX2 and RX2 interrupts
            IER_UART2_TX2 = IER_UART2_RX2 = intEn;
                                        // if intEn = ENABLE, both enable TX2 and RX2 interrupts
                                        // if intEn = DISABLE, both disable TX2 and RX2 interrupts
        break;    
        
        default:
        break;
    }    
	__asm("EI");						// enable all maskable interrupts
}


//*********************************************************************
// name			: UART0_SetBaudRate()
// brief		: set Baud rate
// param		: - baudSel : Operat24M_Baud128000
//                        Operat24M_Baud115200
//                        Operat24M_Baud76800
//                        Operat24M_Baud62500
//                        Operat24M_Baud57600
//                        Operat24M_Baud38400
//                        Operat24M_Baud19200
//                        Operat24M_Baud9600
// retval		: none
//
// note			: the above baudSel is in the case of UART0 operating at 24MHz
//            for other frequencies, please refer to the SQ7515_UART.h for the corresponding baudSel settings
//*********************************************************************

void UART0_SetBaudRate(uchar baudSel){

    // select the number of RT clocks
    UART0CR2_RTSEL = RT;
        
    // select Baud rate	    
    // before changing UART0DR, set UART0CR1_RXE and UART0CR1_TXE to "0"
    UART0_Stop();						
    UART0DR = DR;
}
    
    
//*********************************************************************
// name			: UART1_SetBaudRate()
// brief		: set Baud rate
// param		: - baudSel : Operat24M_Baud128000
//                        Operat24M_Baud115200
//                        Operat24M_Baud76800
//                        Operat24M_Baud62500
//                        Operat24M_Baud57600
//                        Operat24M_Baud38400
//                        Operat24M_Baud19200
//                        Operat24M_Baud9600
// retval		: none
//
// note			: the above baudSel is in the case of UART0 operating at 24MHz 
//            for other frequencies, please refer to the SQ7515_UART.h for the corresponding baudSel settings
//*********************************************************************

void UART1_SetBaudRate(uchar baudSel){

    // select the number of RT clocks
    UART1CR2_RTSEL = RT;
        
    // select Baud rate	    
    // before changing UART1DR, set UART1CR1_RXE and UART1CR1_TXE to "0"
    UART1_Stop();						
    UART1DR = DR;
}


//*********************************************************************
// name			: UART2_SetBaudRate()
// brief		: set Baud rate
// param		: - baudSel : Operat24M_Baud128000
//                        Operat24M_Baud115200
//                        Operat24M_Baud76800
//                        Operat24M_Baud62500
//                        Operat24M_Baud57600
//                        Operat24M_Baud38400
//                        Operat24M_Baud19200
//                        Operat24M_Baud9600
// retval		: none
//
// note			: the above baudSel is in the case of UART0 operating at 24MHz 
//            for other frequencies, please refer to the SQ7515_UART.h for the corresponding baudSel settings
//*********************************************************************

void UART2_SetBaudRate(uchar baudSel){

    // select the number of RT clocks
    UART2CR2_RTSEL = RT;
        
    // select Baud rate	    
    // before changing UART2DR, set UART2CR1_RXE and UART2CR1_TXE to "0"
    UART2_Stop();						
    UART2DR = DR;
}



//*********************************************************************
// name			: UART0_Start()
// brief		: start UART0
// param		: none
// retval		: none
// note			: enable TX0 and RX0
//*********************************************************************
void UART0_Start(){
    UART0CR1_TXE = ENABLE;                   // enable TX0 
	UART0CR1_RXE = ENABLE;                   // enable RX0
}


//*********************************************************************
// name			: UART0_Stop()
// brief		: stop UART0
// param		: none
// retval		: none
// note			: disable TX0 and RX0
//*********************************************************************
void UART0_Stop(){
	UART0CR1_TXE = DISABLE;                   // disable TX0
	UART0CR1_RXE = DISABLE;                   // disable RX0
}

//*********************************************************************
// name			: UART1_Start()
// brief		: start UART1
// param		: none
// retval		: none
// note			: enable TX1 and RX1
//*********************************************************************
void UART1_Start(){
    UART1CR1_TXE = ENABLE;                   // enable TX1 
	UART1CR1_RXE = ENABLE;                   // enable RX1
}


//*********************************************************************
// name			: UART1_Stop()
// brief		: stop UART1
// param		: none
// retval		: none
// note			: disable TX1 and RX1
//*********************************************************************
void UART1_Stop(){
	UART1CR1_TXE = DISABLE;                   // disable TX1
	UART1CR1_RXE = DISABLE;                   // disable RX1
}


//*********************************************************************
// name			: UART2_Start()
// brief		: start UART2
// param		: none
// retval		: none
// note			: enable TX2 and RX2
//*********************************************************************
void UART2_Start(){
    UART2CR1_TXE = ENABLE;                   // enable TX2 
	UART2CR1_RXE = ENABLE;                   // enable RX2
}

//*********************************************************************
// name			: UART2_Stop()
// brief		: stop UART2
// param		: none
// retval		: none
// note			: disable TX2 and RX2
//*********************************************************************
void UART2_Stop(){
	UART2CR1_TXE = DISABLE;                   // disable TX2
	UART2CR1_RXE = DISABLE;                   // disable RX2
}

//*********************************************************************
// name			: UART0_TX_OneByte()
// brief		: UART0 TX transmits a single byte
// param		: data: data to be transmitted by TX
// retval		: none
// note			: only for UART0
//*********************************************************************
void UART0_TX_OneByte(uchar data){
	
	while(UART0SR_TBFL == 1){			// wait for the send buffer to be empty before writing the data to be sent again
		CLR_WDT;						// clear the watchdog timer 
    }

	TD0BUF = data;						// write the data to be sent
    
	while(UART0SR_TBSY == 1){			// 1:transmission in progress , 0:before transmission or waiting for the end of transmission work
        CLR_WDT;						// clear the watchdog timer 
    }
}


//*********************************************************************
// name			: UART0_RX_OneByte()
// brief		: UART0 RX receives a single byte
// param		: none
// retval		: data to be received by RX
// note			: only for UART0        
//*********************************************************************
uchar UART0_RX_OneByte(){
	
    while(!UART0SR_RBFL);               // wait for UART0 RD0BUF to receive data
	return RD0BUF;                      // read out the received RD0BUF

}



//*********************************************************************
// name			: UART1_TX_OneByte()
// brief		: UART1 TX transmits a single byte
// param		: data: data to be transmitted by TX
// retval		: none
// note			: only for UART1
//*********************************************************************
void UART1_TX_OneByte(uchar data){

	while(UART1SR_TBFL == 1){			 // wait for the send buffer to be empty before writing the data to be sent again
		CLR_WDT;						 // clear the watchdog timer 
    }
	TD1BUF = data;						 // write the data to be sent
    
	while(UART1SR_TBSY ==1){			 // 1:transmission in progress , 0:before transmission or waiting for the end of transmission work
        CLR_WDT;						 // clear the watchdog timer 
    }
}


//*********************************************************************
// name			: UART1_RX_OneByte()
// brief		: UART1 RX receives a single byte
// param		: none
// retval		: data to be received by RX
// note			: only for UART1
//*********************************************************************
uchar UART1_RX_OneByte(){
    
    while(!UART1SR_RBFL);               // wait for UART1 RD1BUF to receive data
	return RD1BUF;                      // read out the received RD1BUF

}


//*********************************************************************
// name			: UART2_TX_OneByte()
// brief		: UART2 TX transmits a single byte
// param		: data: data to be transmitted by TX
// retval		: none
// note			: only for UART2
//*********************************************************************
void UART2_TX_OneByte(uchar data){

	while(UART2SR_TBFL == 1){			 // wait for the send buffer to be empty before writing the data to be sent again      
		CLR_WDT;                     	 // clear the watchdog timer 
    }
	TD2BUF = data;						 // write the data to be sent
    
	while(UART2SR_TBSY ==1){			 // 1:transmission in progress , 0:before transmission or waiting for the end of transmission work
        CLR_WDT;                         // clear the watchdog timer 
    }
}


//*********************************************************************
// name			: UART2_RX_OneByte()
// brief		: UART2 RX receives a single byte
// param		: none
// retval		: data to be received by RX
// note			: only for UART2
//*********************************************************************
uchar UART2_RX_OneByte(){

    while(!UART2SR_RBFL);               // wait for UART2 RD2BUF to receive data
	rbuf = RD2BUF;                       // read out the received RD2BUF
	
    while(UART1SR_RBSY ==1){		     // 1:transmission in progress , 0:before transmission or waiting for the end of transmission work
            CLR_WDT;                     // clear the watchdog timer 
    }
	return rbuf;
	
}



//*********************************************************************
// name			: UART_ReadState()
// brief		: get UART status
// param		: - uartSel: UART0, UART1, UART2
// retval		: UART status
// note			: UART selectable       
//*********************************************************************

uchar UART_ReadState(uchar uartSel){
	if(uartSel == UART0){
		return UART0SR;
	}
	else if(uartSel == UART1){
		return UART1SR;
	}
	else if(uartSel == UART2){
		return UART2SR;
	}
	else{
		return 0;
    }
}


//*********************************************************************
// name			: UART_TX_OneByte()
// brief		: TX transmits a single byte
// param		: - uartSel: UART0, UART1, UART2
//            - data   : data to be transmitted by TX
// retval		: none
// note			: UART selectable        
//*********************************************************************
void UART_TX_OneByte(uchar uartSel, uchar data){
	
    if(uartSel == UART0){
        while(UART0SR_TBFL){            // wait for the send buffer to be empty before writing the data to be sent again
            CLR_WDT;                    // clear the watchdog timer 
        }

        TD0BUF = data;					// write the data to be sent

		while(UART0SR_TBSY){
            CLR_WDT;                    // clear the watchdog timer 
        }	
    }	
    else if(uartSel == UART1){
        while(UART1SR_TBFL){            // wait for the send buffer to be empty before writing the data to be sent again
            CLR_WDT;                    // clear the watchdog timer 
        }

		TD1BUF = data;                  // write the data to be sent

		while(UART1SR_TBSY){
		    CLR_WDT;                    // clear the watchdog timer 
        }
	
    }
	else if(uartSel == UART2){
		while(UART2SR_TBFL){            // wait for the send buffer to be empty before writing the data to be sent again
            CLR_WDT;                    // clear the watchdog timer 
		}

		TD2BUF = data;                  // write the data to be sent

		while(UART2SR_TBSY){
		    CLR_WDT;                    // clear the watchdog timer 
        }
    }

}

//*********************************************************************
// name			: UART_RX_OneByte()
// brief		: RX receives a single byte
// param		: - uartSel: UART0, UART1, UART2
// retval		: data to be received by RX
// note			: UART selectable           
//*********************************************************************
uchar UART_RX_OneByte(uchar uartSel){

    if(uartSel == UART0){
        while(!UART0SR_RBFL);           // wait for UART0 RD0BUF to receive data
        return RD0BUF;                  // read out the received RD0BUF
    }
    else if(uartSel == UART1){
        while(!UART1SR_RBFL);           // wait for UART1 RD1BUF to receive data
        return RD1BUF;                  // read out the received RD1BUF
    }
    else if(uartSel == UART2){
        while(!UART2SR_RBFL);           // wait for UART2 RD2BUF to receive data
        return RD2BUF;                  // read out the received RD2BUF
	}
	else{
	    return 0;
    }
}


//*********************************************************************
// name			: UART_Send_Bytes()
// brief		: TX transmits bytes
// param		: - uartSel: UART0, UART1, UART2
//            - sentBuf: the bytes to be transferred are first placed in the sentBuf[]
//            - len    : the length of data
// retval		: none
//
// note			: when the data to be transferred is more than 1 Byte, the user can set the length of data
//
// example		: to transfer 12 bytes, the bytes to be transferred are first placed in sentBuf[] 
//            and transfer in the order sentBuf[0]~sentBuf[11]
//
//            UART_Send_Bytes(UART1,sentBuf, 12);          
//*********************************************************************
void UART_Send_Bytes(uchar uartSel, uchar* sentBuf, uchar len){

	uchar i;

    for(i=0; i<len; i++){
        if(uartSel == UART0){
            while(UART0SR_TBFL){
                CLR_WDT;                // clear the watchdog timer 
            }

            TD0BUF = sentBuf[i];        // write the data to be sent

            while(UART0SR_TBSY){
                CLR_WDT;                // clear the watchdog timer 
            }
        }		
		else if(uartSel == UART1){
            while(UART1SR_TBFL){
                CLR_WDT;                // clear the watchdog timer 
            }

            TD1BUF = sentBuf[i];        // write the data to be sent

            while(UART1SR_TBSY){
                CLR_WDT;                // clear the watchdog timer 
            }
        }
        else if(uartSel == UART2){
            while(UART2SR_TBFL){
                CLR_WDT;                // clear the watchdog timer 
            }

            TD2BUF = sentBuf[i];        // write the data to be sent

            while(UART2SR_TBSY){
                CLR_WDT;                // clear the watchdog timer 
            }
        }
    }
}

//*********************************************************************
// name			: UART_Nibble2Ascii()
// brief		: nibble to ASCII
// note			:           
//*********************************************************************
uchar UART_Nibble2Ascii(uchar anibble){
    uchar	ascii_out = anibble;

    if(anibble > 9){                    // If req ASCII A-F then add 7(hex)	
        ascii_out = ascii_out + 0x07;
    }

    // Add offset to convert to Ascii
    ascii_out = ascii_out + 0x30;

    return(ascii_out);
}


//*********************************************************************
// name			: UART_PutByte()
// brief		: get the high nibble converted to ASCII, then TX transfer
//            get the low nibble converted to ASCII, then TX transfer
// note			: 
//*********************************************************************
void UART_PutByte(uchar uartSel,uchar abyte){
    uchar  temp1 = 0, temp2 = 0;	

    temp1 = (abyte >> 4) & 0x0F;        // get the high nibble
    temp2 = UART_Nibble2Ascii(temp1);   // high nibble to ASCII code
    UART_TX_OneByte(uartSel,temp2);     // TX transfer

    temp1 = abyte & 0x0F;               // get the low nibble
    temp2 = UART_Nibble2Ascii(temp1);   // low nibble to ASCII code
    UART_TX_OneByte(uartSel,temp2);     // TX transfer
}



//*********************************************************************
// name			: UART_PutDecNumber()
// brief		: 
// note			: ***Input value range : 0~9999 
//*********************************************************************
void UART_PutDecNumber(uint value){
    uchar print_buf[4];
	 
    if(value > 9999){
        value = 9999;
    }		
	
    print_buf[0] = ((value%10000)/1000)+'0';
    print_buf[1] = ((value%1000)/100)+'0';
    print_buf[2] = ((value%100)/10)+'0';
    print_buf[3] = ((value%10))+'0';
     
	UART_Send_Bytes(UART1,print_buf,4);
}

//*********************************************************************
// name			: UART_PutCrlf()
// brief		: transmit line break characters
// note			: 
//*********************************************************************
void UART_PutCrlf(void){

/*	
    UART_TX_OneByte(UART0,'\r');
    UART_TX_OneByte(UART0,'\n');
*/	

    UART_TX_OneByte(UART1,'\r');
    UART_TX_OneByte(UART1,'\n');

/*	
    UART_TX_OneByte(UART2,'\r');
    UART_TX_OneByte(UART2,'\n');
*/	
	
}


//*********************************************************************
// name			: UART_SendCString()
// brief		: TX transmits string
// param		: one string
// retval		: none
// note			: 
// example		: send "Hello World!"
//            UART_SendCString("Hello World!")
//*********************************************************************
void UART_SendCString(const char *pstr){

    while(*pstr != 0){	
        if(*pstr == '\n'){
            UART_PutCrlf();
        }
        else{		
            //UART_TX_OneByte(UART0,*pstr);
            UART_TX_OneByte(UART1,*pstr);
            //UART_TX_OneByte(UART2,*pstr);
        }

		pstr++;
	}
}


//*********************************************************************
// name			: Check_Recive_Data_LED()
// brief		: compare the received data, if it is correct then light on the corresponding LED
// param		: none
// retval		: none
// note			:
//*********************************************************************
void Check_Recive_Data_LED(void)
{
    uchar i,check_result;
    check_result = 0;
 
    if(reciveBuf[0] == 0x55){					
        LED1_to_LED3_turn_off;
        LED1_out = 0;                   // if received data is 0x55, light on LED1(P2.0)
    }
    else if(reciveBuf[0] == 0xAA){
        LED1_to_LED3_turn_off;				
        LED2_out = 0;                   // if received data is 0xAA,  light onLED2(P2.1)
    }
    else if(reciveBuf[11] == 0xA5){	
        LED1_to_LED3_turn_off;

        // one by one data compare
        for (i=0; i<12; i++){           // compare 12 bytes of data
            if(reciveBuf[i] == sentBuf[i]){ 
                check_result++;
            }

            if(check_result==12){
                LED3_out = 0;           // if all data is correct, then light on LED3(P2.2)      
            }
		}
    }


}



//*********************************************************************
// name			: __interrupt IntRX0()
// brief		: RX0 interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntRX0(void){
    __asm("NOP");                       // no operation
}


//*********************************************************************
// name			: __interrupt IntRX1()
// brief		: RX1 interrupt service routine
// note			: if the received data matches the transmitted data, then light on the corresponding LED
//*********************************************************************
void __interrupt IntRX1(void){
    __asm("NOP");                       // no operation
	
    RX1_Int_Flag = 1;                   // set RX1_Int_Flag to 1
   
    reciveBuf[rx1_int_count] = RD1BUF;  // read out the received data

    rx1_int_count++ ;

}


//*********************************************************************
// name			: __interrupt IntRX2()
// brief		: RX2 interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntRX2(void){
    __asm("NOP");                       // no operation
}


//*********************************************************************
// name			: __interrupt IntTX0()
// brief		: TX0 interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntTX0(void){
    __asm("NOP");                       // no operation
}

//*********************************************************************
// name			: __interrupt IntTX1()
// brief		: TX1 interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntTX1(void){
    __asm("NOP");                       // no operation
}

//*********************************************************************
// name			: __interrupt IntTX2()
// brief		: TX2 interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntTX2(void){
    __asm("NOP");                       // no operation
}

/* ------------------------- end of file ---------------------------- */
