//*************************************************************************************
// file			: SQ7515_UART.c
// version		: V3.1
// brief		: UART related functions
// note			: UART related functions are gathered in this subroutine
//**************************************************************************************
#include "main.h"
unsigned char rbuf;                     	  // buffer for RX1 reveived value

// RT clocks
const unsigned char RTTable[]={
//1Mhz
	0x00,//4800
	0x02,//9600
	0x04,//19200
	0x04,//57600
//2MHz
	0x00,//4800
	0x00,//9600
	0x02,//19200
	0x04,//57600
	0x04,//115200
	0x03,//128000
//4MHz
	0x04,//4800
	0x00,//9600
	0x00,//19200
	0x04,//57600
	0x04,//115200
	0x03,//128000
//8MHz
	0x01,//4800
	0x04,//9600
	0x00,//19200
	0x03,//57600
	0x04,//115200
	0x03,//128000
//12MHz	
	0x04,//4800
	0x00,//9600
	0x00,//19200
	0x00,//57600
	0x02,//115200
	0x03,//128000	
//16MHz
	0x01,//4800
	0x01,//9600
	0x04,//19200
	0x03,//57600
	0x03,//115200
	0x03,//128000
	0x03,//256000
//24MHz
	0x04,//9600
	0x00,//19200
	0x00,//57600
	0x00,//115200
	0x04,//128000
	0x03,//256000
};
//------------------------------------------------------------------------------
// Baud rate	
const unsigned char DRTable[]={
//1Mhz
	0x0C,//4800
	0x06,//9600
	0x02,//19200
	0x00,//57600
//2MHz
	0x19,//4800
	0x0C,//9600
	0x06,//19200
	0x01,//57600
	0x00,//115200
	0x00,//128000
//4MHz
	0x30,//4800
	0x19,//9600
	0x0C,//19200
	0x03,//57600
	0x01,//115200
	0x01,//128000
//8MHz
	0x64,//4800
	0x30,//9600
	0x19,//19200
	0x08,//57600
	0x03,//115200
	0x03,//128000
//12MHz
	0x92,//4800
	0x4D,//9600
	0x26,//19200
	0x0C,//57600
	0x06,//115200
	0x05,//128000
//16MHz
	0xC9,//4800
	0x64,//9600
	0x30,//19200
	0x11,//57600
	0x08,//115200
	0x07,//128000
	0x03,//256000
//24M
	0x92,//9600   
	0x4D,//19200
	0x19,//57600
	0x0C,//115200
	0x0A,//128000
	0x05,//256000
};



//*********************************************************************
// name			: UART0_Init()
// brief		: UART0 initialization
// retval		: 
// note			: set serial data output in pins and UART related parameters
//*********************************************************************
void UART0_Init(){

//====== UART0 pin configuration initialization ======
	
/*	
	//--TX0 (P0.1) ; RX0 (P0.0)-------
	P0FC1_P1 = 1;                       // set P0.1 to TX0
	P0FC2_P1 = 0;                       
	P0FC1_P0 = 1;                       // set P0.0 to RX0
	P0FC2_P0 = 0;	
	FSELR0 |= 0x00;                     // set function selection register 0, function group selection b00
	PCSELR0_UART0 = 0x00;               // select peripheral function channel
*/	
	
/*	
	//--TX0 (P0.0) ; RX0 (P0.1)-------
	P0FC1_P0 = 1;                       // set P0.0 to TX0
	P0FC2_P0 = 0;	
	P0FC1_P1 = 1;                       // set P0.1 to RX0
	P0FC2_P1 = 0;                       
	FSELR0 |= 0x01;                     // set function selection register 0, function group selection b01
	PCSELR0_UART0 = 0x00;               // select peripheral function channel	
*/


/*
	//--TX0 (P3.7) ; RX0 (P3.6)-------
	P3FC1_P7 = 1;                       // set P3.7 to TX0
	P3FC2_P7 = 0;	
	P3FC1_P6 = 1;                       // set P3.6 to RX0
	P3FC2_P6 = 0;                       
	FSELR3 |= 0x00;                     // set function selection register 3, function group selection b00
	PCSELR0_UART0 = 0x01;               // select peripheral function channel		
*/	



/*
	//--TX0 (P3.6) ; RX0 (P3.7)-------
	P3FC1_P6 = 1;                       // set P3.6 to TX0
	P3FC2_P6 = 0;  
	P3FC1_P7 = 1;                       // set P3.7 to RX0
	P3FC2_P7 = 0;	
	FSELR3 |= 0x10;                     // set function selection register 3, function group selection b10
	PCSELR0_UART0 = 0x01;               // select peripheral function channel		
*/



//====== UART0 parameter setting ======
/*
	PCKEN1_UART0 	= 1;                // this must be enabled first, the relevant staging settings later to be useful,UART0 Enable
	UART0CR1_STOPBT = 0;                // set the bit length of the TX0 stop bit to 1 bit
	UART0CR2_STOPBR = 0;                // set the bit length of the RX0 stop bit to 1 bit	
	UART0CR1_PE 	= 0;                // set no parity

	//UART0CR1_PE 	= 1;                // parity added
	//UART0CR1_EVEN = 1;                // 0:odd-numbered parity 1:even-numbered parity

*/


}


//*********************************************************************
// name			: UART1_Init()
// brief		: UART1 initialization
// param		: 
// retval		: 
// note			: set serial data output in pins and UART related parameters
//*********************************************************************
void UART1_Init(){

//====== UART1 pin configuration initialization ======
	
/*	
	//--TX1 (P2.6) ; RX1 (P2.5)-------
	P2FC1_P6 = 1;                       // set P2.6 to TX1
	P2FC2_P6 = 0;                       
	P2FC1_P5 = 1;                       // set P2.5 to RX1
	P2FC2_P5 = 0;	
	FSELR2 |= 0x00;                     // set function selection register 2, function group selection b00
	PCSELR0_UART1 = 0x00;               // select peripheral function channel
*/	
	
	
	//--TX1 (P2.5) ; RX1 (P2.6)-------  [for this example program]
	P2FC1_P5 = 1;                       // set P2.5 to TX1
	P2FC2_P5 = 0;	
	P2FC1_P6 = 1;                       // set P2.6 to RX1
	P2FC2_P6 = 0;                       
	FSELR2 |= 0x10;                     // set function selection register 2, function group selection b10
	PCSELR0_UART1 = 0x00;               // select peripheral function channel	



/*
	//--TX1 (P5.2) ; RX1 (P5.1)-------
	P5FC1_P2 = 1;                       // set P5.2 to TX1
	P5FC2_P2 = 0;	
	P5FC1_P1 = 1;                       // set P5.1 to RX1
	P5FC2_P1 = 0;                       
	FSELR5 |= 0x00;                     // set function selection register 5, function group selection b00
	PCSELR0_UART1 = 0x01;               // select peripheral function channel		
*/	



/*
	//--TX1 (P5.1) ; RX1 (P5.2)-------
	P5FC1_P1 = 1;                       // set P5.1 to TX1
	P5FC2_P1 = 0;  
	P5FC1_P2 = 1;                       // set P5.2 to RX1
	P5FC2_P2 = 0;	
	FSELR5 |= 0x01;                     // set function selection register 5, function group selection b01
	PCSELR0_UART1 = 0x01;               // select peripheral function channel		
*/



//====== UART1parameter setting ======
	PCKEN1_UART1 	= 1;                // this must be enabled first, the relevant staging settings later to be useful,UART1 Enable
	UART1CR1_STOPBT = 0;                // set the bit length of the TX0 stop bit to 1 bit
	UART1CR2_STOPBR = 0;                // set the bit length of the RX0 stop bit to 1 bit	
	UART1CR1_PE 	= 0;                // set no parity
	
	
	//UART1CR1_PE 	= 1;                // parity added
	//UART1CR1_EVEN = 1;                // 0:odd-numbered parity 1:even-numbered parity

}


//*********************************************************************
// name			: UART2_Init()
// brief		: UART2 initialization
// param		: 
// retval		: 
// note			: set serial data output in pins and UART related parameters
//*********************************************************************
void UART2_Init(){

//====== UART2 pin configuration initialization ======
	
/*	
	//--TX2 (P0.5) ; RX2 (P0.4)-------
	P0FC1_P5 = 1;                       // set P0.5 to TX2
	P0FC2_P5 = 0;                       
	P0FC1_P4 = 1;                       // set P0.4 to RX2
	P0FC2_P4 = 0;	
	FSELR0 |= 0x00;                      // set function selection register 0, function group selection b00
	PCSELR0_UART2 = 0x00;               // select peripheral function channel
*/	
	
/*	
	//--TX2 (P0.4) ; RX2 (P0.5)-------
	P0FC1_P4 = 1;                       // set P0.4 to TX2
	P0FC2_P4 = 0;	
	P0FC1_P5 = 1;                       // set P0.5 to RX2
	P0FC2_P5 = 0;                       
	FSELR0 |= 0x10;                     // set function selection register 0, function group selection b10
	PCSELR0_UART2 = 0x00;               // select peripheral function channel	
*/


/*
	//--TX2 (P3.1) ; RX2 (P3.0)-------
	P3FC1_P1 = 1;                       // set P3.1 to TX2
	P3FC2_P1 = 0;	
	P3FC1_P0 = 1;                       // set P3.0 to RX2
	P3FC2_P0 = 0;                       
	FSELR3 |= 0x00;                     // set function selection register 3, function group selection b00
	PCSELR0_UART2 = 0x01;               // select peripheral function channel		
*/	



/*
	//--TX2 (P3.0) ; RX2 (P3.1)-------
	P3FC1_P0 = 1;                       // set P3.0 to TX2
	P3FC2_P0 = 0;
	P3FC1_P1 = 1;                       // set P3.1 to RX2
	P3FC2_P1 = 0;	
	FSELR3 |= 0x01;                     // set function selection register 3, function group selection b01
	PCSELR0_UART2 = 0x01;               // select peripheral function channel		
*/



//====== UART2parameter setting ======
/*
	PCKEN1_UART2 	= 1;                // this must be enabled first, the relevant staging settings later to be useful,UART2 Enable
	UART2CR1_STOPBT = 0;                // set the bit length of the TX0 stop bit to 1 bit
	UART2CR2_STOPBR = 0;                // set the bit length of the RX0 stop bit to 1 bit	
	UART2CR1_PE 	= 0;                // set no parity
	
	
	//UART2CR1_PE 	= 1;                // parity added
	//UART2CR1_EVEN = 1;                // 0:odd-numbered parity 1:even-numbered parity
*/


}

//*********************************************************************
// name			: UART0_IntEnable()
// brief		: enable UART0 interrupt
// param		: 
// retval		: 
// note			: 
//*********************************************************************
void UART0_IntEnable(){
	__ASM("DI");						// enable all maskable interrupts
	IER_UART0_TX = 1;                   // enable TX0 interrupt
	IER_UART0_RX = 1;	                // enable RX0 interrupt
	__ASM("EI");						// enable all maskable interrupts
}


//*********************************************************************
// name			: UART1_IntEnable()
// brief		: enable UART1 interrupt
// param		: 
// retval		: 
// note			: 
//*********************************************************************
void UART1_IntEnable(){
	__ASM("DI");						// enable all maskable interrupts
	IER_UART1_TX1 = 1;                  // enable TX1 interrupt
	IER_UART1_RX1 = 1;	                // enable RX1 interrupt
	__ASM("EI");						// enable all maskable interrupts
}

//*********************************************************************
// name			: UART2_IntEnable()
// brief		: enable UART2 interrupt
// param		: 
// retval		: 
// note			: 
//*********************************************************************
void UART2_IntEnable(){
	__ASM("DI");						// enable all maskable interrupts
	IER_UART2_TX2 = 1;                  // enable TX2 interrupt
	IER_UART2_RX2 = 1;	                // enable RX2 interrupt
	__ASM("EI");						// enable all maskable interrupts
}


//*********************************************************************
// name			: UART0_SetBaud rate()  / UART1_SetBaud rate() / UART2_SetBaud rate()
// brief		: set Baud rate
// param		: baudSel: 
// 				SYSCLK@1MHZ
// 					SYSCLK_1MHZ_BAUD_4800		0
// 					SYSCLK_1MHZ_BAUD_9600		1
// 					SYSCLK_1MHZ_BAUD_19200		2
// 					SYSCLK_1MHZ_BAUD_57600		3
//
// 				SYSCLK@2MHZ
// 					SYSCLK_2MHZ_BAUD_4800		4
// 					SYSCLK_2MHZ_BAUD_9600		5
// 					SYSCLK_2MHZ_BAUD_19200		6
// 					SYSCLK_2MHZ_BAUD_57600		7
// 					SYSCLK_2MHZ_BAUD_115200		8
// 					SYSCLK_2MHZ_BAUD_128000		9
//
// 				SYSCLK@4MHZ
// 					SYSCLK_4MHZ_BAUD_4800		10
// 					SYSCLK_4MHZ_BAUD_9600		11
// 					SYSCLK_4MHZ_BAUD_19200		12
// 					SYSCLK_4MHZ_BAUD_57600		13
// 					SYSCLK_4MHZ_BAUD_115200		14
// 					SYSCLK_4MHZ_BAUD_128000		15
//
// 				SYSCLK@8MHZ
// 					SYSCLK_8MHZ_BAUD_4800		16
// 					SYSCLK_8MHZ_BAUD_9600		17
// 					SYSCLK_8MHZ_BAUD_19200		18
// 					SYSCLK_8MHZ_BAUD_57600		19
// 					SYSCLK_8MHZ_BAUD_115200		20
// 					SYSCLK_8MHZ_BAUD_128000		21
//
// 				SYSCLK@12MHZ
// 					SYSCLK_12MHZ_BAUD_4800		22
// 					SYSCLK_12MHZ_BAUD_9600		23
// 					SYSCLK_12MHZ_BAUD_19200		24
// 					SYSCLK_12MHZ_BAUD_57600		25
// 					SYSCLK_12MHZ_BAUD_115200	26
// 					SYSCLK_12MHZ_BAUD_128000	27
//
// 				SYSCLK@16MHZ
// 					SYSCLK_16MHZ_BAUD_4800		28
// 					SYSCLK_16MHZ_BAUD_9600		29
// 					SYSCLK_16MHZ_BAUD_19200		30
// 					SYSCLK_16MHZ_BAUD_57600		31
// 					SYSCLK_16MHZ_BAUD_115200	32
// 					SYSCLK_16MHZ_BAUD_128000	33
// 					SYSCLK_16MHZ_BAUD_256000	34
//
// 				SYSCLK@24MHZ
// 					SYSCLK_24MHZ_BAUD_9600		35
// 					SYSCLK_24MHZ_BAUD_19200		36
// 					SYSCLK_24MHZ_BAUD_57600		37
// 					SYSCLK_24MHZ_BAUD_115200	38
// 					SYSCLK_24MHZ_BAUD_128000	39
// 					SYSCLK_24MHZ_BAUD_256000	40             
// retval		: 
// note			:
//*********************************************************************
void UART0_SetBaud rate(unsigned char baudSel){

//====== UART0 ======
// select the number of RT clocks
	UART0CR2_RTSEL = RTTable[baudSel];
	UART0_Stop();						// before changing UART0DR, set UART0CR1_RXE and UART0CR1_TXE to "0"
// select Baud rate	
	UART0DR  =  DRTable[baudSel];

}

//--------------------------------------------------------------
void UART1_SetBaud rate(unsigned char baudSel){
	
//====== UART1 ======  /*[for this example program]*/

//UART1CR2 = (UART1CR2 & 0x07) | ( RTTable[baudSel] <<3);
//UART1DR  =  DRTable[baudSel];

// select the number of RT clocks
	UART1CR2_RTSEL = RTTable[baudSel];
	UART1_Stop();						// before changing UART1DR, set UART1CR1_RXE and UART1CR1_TXE to "0"
// select Baud rate	
	UART1DR  =  DRTable[baudSel];
	
}

//--------------------------------------------------------------
void UART2_SetBaud rate(unsigned char baudSel){

//====== UART2 ======
// select the number of RT clocks
	UART2CR2_RTSEL = RTTable[baudSel];
	UART2_Stop();						// before changing UART2DR, set UART2CR1_RXE and UART2CR1_TXE to "0"
// select Baud rate	
	UART2DR  =  DRTable[baudSel];
	
}



//*********************************************************************
// name			: UART0_Start()
// brief		: start UART0
// param		: none
// retval		: none
// note			: enable TX0 and RX0
//*********************************************************************
void UART0_Start(){
    UART0CR1_TXE = ENABLE;                   // enable TX0 
	UART0CR1_RXE = ENABLE;                   // enable RX0
}


//*********************************************************************
// name			: UART0_Stop()
// brief		: stop UART0
// param		: none
// retval		: none
// note			: disable TX0 and RX0
//*********************************************************************
void UART0_Stop(){
	UART0CR1_TXE = DISABLE;                   // disable TX0
	UART0CR1_RXE = DISABLE;                   // disable RX0
}

//*********************************************************************
// name			: UART1_Start()
// brief		: start UART1
// param		: none
// retval		: none
// note			: enable TX1 and RX1
//*********************************************************************
void UART1_Start(){
    UART1CR1_TXE = ENABLE;                   // enable TX1 
	UART1CR1_RXE = ENABLE;                   // enable RX1
}


//*********************************************************************
// name			: UART1_Stop()
// brief		: stop UART1
// param		: none
// retval		: none
// note			: disable TX1 and RX1
//*********************************************************************
void UART1_Stop(){
	UART1CR1_TXE = DISABLE;                   // disable TX1
	UART1CR1_RXE = DISABLE;                   // disable RX1
}


//*********************************************************************
// name			: UART2_Start()
// brief		: start UART2
// param		: none
// retval		: none
// note			: enable TX2 and RX2
//*********************************************************************
void UART2_Start(){
    UART2CR1_TXE = ENABLE;                   // enable TX2 
	UART2CR1_RXE = ENABLE;                   // enable RX2
}

//*********************************************************************
// name			: UART2_Stop()
// brief		: stop UART2
// param		: none
// retval		: none
// note			: disable TX2 and RX2
//*********************************************************************
void UART2_Stop(){
	UART2CR1_TXE = DISABLE;                   // disable TX2
	UART2CR1_RXE = DISABLE;                   // disable RX2
}

//*********************************************************************
// name			: UART0_TX_OneByte()
// brief		: UART0 TX transmits a single byte
// param		: data: data to be transmitted by TX
// retval		: none
// note			: only for UART0
//*********************************************************************
void UART0_TX_OneByte(unsigned char data){
	
	while(UART0SR_TBFL == 1){			// wait for the send buffer to be empty before writing the data to be sent again
		CLR_WDT;						// clear the watchdog timer 
    }

	TD0BUF = data;						// write the data to be sent
    
	while(UART0SR_TBSY == 1){			// 1:transmission in progress , 0:before transmission or waiting for the end of transmission work
        CLR_WDT;						// clear the watchdog timer 
    }
}


//*********************************************************************
// name			: UART0_RX_OneByte()
// brief		: UART0 RX receives a single byte
// param		: none
// retval		: data to be received by RX
// note			: only for UART0        
//*********************************************************************
uchar UART0_RX_OneByte(){
	
    while(!UART0SR_RBFL);               // wait for UART0 RD0BUF to receive data
	return RD0BUF;                      // read out the received RD0BUF

}



//*********************************************************************
// name			: UART1_TX_OneByte()
// brief		: UART1 TX transmits a single byte
// param		: data: data to be transmitted by TX
// retval		: none
// note			: only for UART1
//*********************************************************************
void UART1_TX_OneByte(unsigned char data){

	while(UART1SR_TBFL == 1){			 // wait for the send buffer to be empty before writing the data to be sent again
		CLR_WDT;						 // clear the watchdog timer 
    }
	TD1BUF = data;						 // write the data to be sent
    
	while(UART1SR_TBSY ==1){			 // 1:transmission in progress , 0:before transmission or waiting for the end of transmission work
        CLR_WDT;						 // clear the watchdog timer 
    }
}


//*********************************************************************
// name			: UART1_RX_OneByte()
// brief		: UART1 RX receives a single byte
// param		: none
// retval		: data to be received by RX
// note			: only for UART1
//*********************************************************************
uchar UART1_RX_OneByte(){
    
    while(!UART1SR_RBFL);               // wait for UART1 RD1BUF to receive data
	return RD1BUF;                      // read out the received RD1BUF

}


//*********************************************************************
// name			: UART2_TX_OneByte()
// brief		: UART2 TX transmits a single byte
// param		: data: data to be transmitted by TX
// retval		: none
// note			: UART2
//*********************************************************************
void UART2_TX_OneByte(unsigned char data){

	while(UART2SR_TBFL == 1){			 // wait for the send buffer to be empty before writing the data to be sent again      
		CLR_WDT;                     	 // clear the watchdog timer 
    }
	TD2BUF = data;						 // write the data to be sent
    
	while(UART2SR_TBSY ==1){			 // 1:transmission in progress , 0:before transmission or waiting for the end of transmission work
        CLR_WDT;                         // clear the watchdog timer 
    }
}


//*********************************************************************
// name			: UART2_RX_OneByte()
// brief		: UART2 RX receives a single byte
// param		: none
// retval		: data to be received by RX
// note			: only for UART2
//*********************************************************************
uchar UART2_RX_OneByte(){

    while(!UART2SR_RBFL);               // wait for UART2 RD2BUF to receive data
	rbuf = RD2BUF;                       // read out the received RD2BUF
	
    while(UART1SR_RBSY ==1){		     // 1:transmission in progress , 0:before transmission or waiting for the end of transmission work
            CLR_WDT;                     // clear the watchdog timer 
    }
	return rbuf;
	
}



//**********************************************************************************
// name			: UART_Init()
// brief		: UART initialization
// param		: uart: UART_0, UART_1, UART_2	 // UART selection
//            parity_en: (parity selection)
//                  UART_PE_NONE    0x00	 // no parity
//                  UART_PE_EN      0x01	 // parity added		
//            parity:    (numbered parity selection)
//					UART_EVEN    	0x00	 // odd-numbered parity
// 					UART_ODD		0x01	 // even-numbered parity
//            stopBit:   (stop bit length)
//                  STOPbit_1bit    0x00  	 // set the bit length of the TX0 and RX stop bit to 1 bit
//                  STOPbit_2bit    0x01	 // set the bit length of the TX0 and RX stop bit to 2 bit
// retval		: 
// note			: 
//**********************************************************************************
void UART_Init(unsigned char uart,unsigned char parity_en,unsigned char parity,unsigned char stopBit){	
    if(uart==UART_0){
		PCKEN1_UART0 = 1;					  			    // this must be enabled first, the relevant staging settings later to be useful,UART0 Enable
		UART0CR1 = (stopBit<<5)|(parity_en<<3)|(parity<<4); // set TX parity and stop bit length
		UART0CR2 = (UART0CR2&0xFE)|stopBit;   			    // set RX stop bit length
	}
	else if(uart==UART_1){	
		PCKEN1_UART1 = 1;					  				// this must be enabled first, the relevant staging settings later to be useful,UART1 Enable
		UART1CR1 = (stopBit<<5)|(parity_en<<3)|(parity<<4); // set TX parity and stop bit length
		UART1CR2 = (UART1CR2&0xFE)|stopBit;	  				// set RX stop bit length
	}
	else if(uart==UART_2){
		PCKEN1_UART2 = 1;					  				// this must be enabled first, the relevant staging settings later to be useful,UART2 Enable
		UART2CR1 = (stopBit<<5)|(parity_en<<3)|(parity<<4); // set TX parity and stop bit length
		UART2CR2 = (UART2CR2&0xFE)|stopBit;					// set RX stop bit length
	}
	
}

//*********************************************************************
// name			: UART_IntEnable()
// brief		: UART interrupt selection and enable
// param		: uart: UART_0, UART_1, UART_2	 // UART interrupt selection
//            TRx_en: 
//                  UART_TXEn	 0x80
//                  UART_RXEn    0x40 
// retval		:
// note			:
//*********************************************************************
void UART_IntEnable(unsigned char uart,unsigned char TRx_en){

	__ASM("DI");						// enable all maskable interrupts
	if(uart==UART_0){
		if(TRx_en & UART_TXEn){
			IER_UART0_TX = 1;			// enable TX0 interrupt
		}
		else{
			IER_UART0_TX = 0;			// enable TX0 interrupt
		}
		//-----------------------
		if(TRx_en & UART_RXEn){
			IER_UART0_RX = 1;			// enable RX0 interrupt
		}
		else{
			IER_UART0_RX = 0;			// enable RX0 interrupt
		}		
	}
	//------------------------------------
	else if(uart==UART_1){
		if(TRx_en & UART_TXEn){
			IER_UART1_TX1 = 1;			// enable TX1 interrupt
		}
		else{
			IER_UART1_TX1 = 0;			// enable TX1 interrupt
		}
		if(TRx_en & UART_RXEn){
			IER_UART1_RX1 = 1;			// enable RX1 interrupt
		}
		else{
			IER_UART1_RX1 = 0;			// enable RX1 interrupt
		}	
	}
	//------------------------------------
	else if(uart==UART_2){
		if(TRx_en & UART_TXEn){
			IER_UART2_TX2 = 1;			// enable TX2 interrupt
		}
		else{
			IER_UART2_TX2 = 0;			// enable TX2 interrupt
		}
		if(TRx_en & UART_RXEn){
			IER_UART2_RX2 = 1;			// enable RX2 interrupt
		}
		else{
			IER_UART2_RX2 = 0;			// enable RX2 interrupt
		}	
	}
	__ASM("EI");						// enable all maskable interrupts
}


//*********************************************************************
// name			: UART_SetBaud rate()
// brief		: set Baud rate
// param		: 
//           uart: UART_0, UART_1, UART_2	 // UART selection
//           baudSel: 
// 				SYSCLK@1MHZ
// 					SYSCLK_1MHZ_BAUD_4800		0
// 					SYSCLK_1MHZ_BAUD_9600		1
// 					SYSCLK_1MHZ_BAUD_19200		2
// 					SYSCLK_1MHZ_BAUD_57600		3
//
// 				SYSCLK@2MHZ
// 					SYSCLK_2MHZ_BAUD_4800		4
// 					SYSCLK_2MHZ_BAUD_9600		5
// 					SYSCLK_2MHZ_BAUD_19200		6
// 					SYSCLK_2MHZ_BAUD_57600		7
// 					SYSCLK_2MHZ_BAUD_115200		8
// 					SYSCLK_2MHZ_BAUD_128000		9
//
// 				SYSCLK@4MHZ
// 					SYSCLK_4MHZ_BAUD_4800		10
// 					SYSCLK_4MHZ_BAUD_9600		11
// 					SYSCLK_4MHZ_BAUD_19200		12
// 					SYSCLK_4MHZ_BAUD_57600		13
// 					SYSCLK_4MHZ_BAUD_115200		14
// 					SYSCLK_4MHZ_BAUD_128000		15
//
// 				SYSCLK@8MHZ
// 					SYSCLK_8MHZ_BAUD_4800		16
// 					SYSCLK_8MHZ_BAUD_9600		17
// 					SYSCLK_8MHZ_BAUD_19200		18
// 					SYSCLK_8MHZ_BAUD_57600		19
// 					SYSCLK_8MHZ_BAUD_115200		20
// 					SYSCLK_8MHZ_BAUD_128000		21
//
// 				SYSCLK@12MHZ
// 					SYSCLK_12MHZ_BAUD_4800		22
// 					SYSCLK_12MHZ_BAUD_9600		23
// 					SYSCLK_12MHZ_BAUD_19200		24
// 					SYSCLK_12MHZ_BAUD_57600		25
// 					SYSCLK_12MHZ_BAUD_115200	26
// 					SYSCLK_12MHZ_BAUD_128000	27
//
// 				SYSCLK@16MHZ
// 					SYSCLK_16MHZ_BAUD_4800		28
// 					SYSCLK_16MHZ_BAUD_9600		29
// 					SYSCLK_16MHZ_BAUD_19200		30
// 					SYSCLK_16MHZ_BAUD_57600		31
// 					SYSCLK_16MHZ_BAUD_115200	32
// 					SYSCLK_16MHZ_BAUD_128000	33
// 					SYSCLK_16MHZ_BAUD_256000	34
//
// 				SYSCLK@24MHZ
// 					SYSCLK_24MHZ_BAUD_9600		35
// 					SYSCLK_24MHZ_BAUD_19200		36
// 					SYSCLK_24MHZ_BAUD_57600		37
// 					SYSCLK_24MHZ_BAUD_115200	38
// 					SYSCLK_24MHZ_BAUD_128000	39
// 					SYSCLK_24MHZ_BAUD_256000	40
// retval		: 
// note			:
//*********************************************************************
void UART_SetBaud rate(unsigned char uart,unsigned char baudSel){
	if(uart==UART_0){
		UART0CR2 = (UART0CR2 & 0xC7) | ( RTTable[baudSel] <<3);
 		UART0DR  = DRTable[baudSel];
	}
	else if(uart==UART_1){
		UART1CR2 = (UART1CR2 & 0xC7) | ( RTTable[baudSel] <<3);
		UART1DR  =  DRTable[baudSel];
	}
	else if(uart==UART_2){
		UART2CR2 = (UART2CR2 & 0xC7) | ( RTTable[baudSel] <<3);
		UART2DR  =  DRTable[baudSel];
	}
}



//*********************************************************************
// name			: UART_Start()
// brief		: start UART
// param		: 
//           uart: UART_0, UART_1, UART_2	 // UART selection
//           uartEn: 
//                  UART_TXEn	 0x80
//                  UART_RXEn    0x40
// retval		: 
// note			:
//*********************************************************************
void UART_Start(unsigned char uart, unsigned char uartEn){
	
	if(uart == UART_0){
		UART0CR1 = UART0CR1 | (uartEn&0xC0);
	}	
	else if(uart == UART_1){
		UART1CR1 = UART1CR1 | (uartEn&0xC0);
	}
	else if(uart == UART_2){
		UART2CR1 = UART2CR1 | (uartEn&0xC0);
	}

}

//*********************************************************************
// name			: UART_Stop()
// brief		: stop UART
// param		: 
//           uart: UART_0, UART_1, UART_2	 // UART selection
//           uartEn: 
//                  UART_TXEn	 0x80
//                  UART_RXEn    0x40
// retval		: 
// note			:
//*********************************************************************
void UART_Stop(unsigned char uart,unsigned char uartEn){
	
	
	if(uart == UART_0){
		UART0CR1 = UART0CR1 & (~(uartEn&0xC0));
	}
	else if(uart == UART_1){
		UART1CR1 = UART1CR1 & (~(uartEn&0xC0));
	}
	else if(uart == UART_2){
		UART2CR1 = UART2CR1 & (~(uartEn&0xC0));
	}

}


//*********************************************************************
// name			: UART_ReadState()
// brief		: get UART status
// param		: 
//           uart: UART_0, UART_1, UART_2	 // UART selection
// retval		: UART status
// note			:
//*********************************************************************

unsigned char UART_ReadState(unsigned char uart){
	if(uart==UART_0){
		return UART0SR;
	}
	else if(uart==UART_1){
		return UART1SR;
	}
	else if(uart==UART_2){
		return UART2SR;
	}
	else {
		return 0;
    }
}


//*********************************************************************
// name			: UART_TX_OneByte()
// brief		: UARTx TX transmits a single byte
// param		: 
//           uart: UART_0, UART_1, UART_2	 // UART selection
//           data: data to be trasmitted
// retval		: 
// note			:
//*********************************************************************
void UART_TX_OneByte(unsigned char uart,unsigned char data){
	
	if(uart==UART_0){
		while(UART0SR_TBFL);			// wait for the send buffer to be empty before writing the data to be sent again      
		TD0BUF = data;					// write the data to be sent
		while(UART0SR_TBFL);
	}
	else if(uart==UART_1){
		while(UART1SR_TBFL);			// wait for the send buffer to be empty before writing the data to be sent again      
		TD1BUF = data;					// write the data to be sent
		while(UART1SR_TBFL);
	}
	else if(uart==UART_2){
		while(UART2SR_TBFL);			// wait for the send buffer to be empty before writing the data to be sent again      
		TD2BUF = data;					// write the data to be sent
		while(UART2SR_TBFL);
	}	
	
}


//*********************************************************************
// name			: UART_RX_OneByte()
// brief		: UARTx RX receives a single byte
// param		: uart: UART_0, UART_1, UART_2	 // UART selection
// retval		: data to be received by RX
// note			:
//*********************************************************************
unsigned char UART_RX_OneByte(unsigned char uart){
	
	if(uart==UART_0){
        while(!UART0SR_RBFL);			// wait for UART0 RD0BUF to receive data
		return RD0BUF;					// wait for UART0 RD0BUF to receive data
	}
	else if(uart==UART_1){
        while(!UART1SR_RBFL);			// wait for UART0 RD1BUF to receive data
		return RD1BUF;					// wait for UART1 RD1BUF to receive data
	}
	else if(uart==UART_2){
        while(!UART2SR_RBFL);			// wait for UART2 RD2BUF to receive data
		return RD2BUF;					// read out the received RD2BUF
	}
	else{
        return 0;
    }
}

//*********************************************************************
// name			: UART_Send()
// brief		: UARTx TX transmits data
// param		: 
//			  uart: UART_0, UART_1, UART_2	 // UART selection
//            buf: data buffer pointer
//            len: data buffer size
// retval		: 
// note			: when the data to be transferred is more than 1 Byte, user set the datar length size
//*********************************************************************
void UART_Send(unsigned char uart,unsigned char* buf, unsigned char len){
	
	unsigned char i;
	for(i=0;i<len;i++){
		if(uart==UART_0){
			while(UART0SR_TBFL);        // wait for the send buffer of UART0 to be empty before writing the data to be sent again      
			TD0BUF = buf[i];            // put in the data to be sent
		}
		else if(uart==UART_1){
			while(UART1SR_TBFL);        // wait for the send buffer of UART1 to be empty before writing the data to be sent again      
			TD1BUF = buf[i];            // put in the data to be sent
		}
		else if(uart==UART_2){
			while(UART2SR_TBFL);        // wait for the send buffer of UART2 to be empty before writing the data to be sent again      
			TD2BUF = buf[i];            // put in the data to be sent
		}
	}
}


//*********************************************************************
// name			: UART_Recive()
// brief		: UARTx RX receives data
// param		: uart: UART_0, UART_1, UART_2	 // UART selection
// retval		: data to be received by RX
// note			: 
//*********************************************************************
unsigned char UART_Recive(unsigned char uart){
    unsigned char RET;

	switch(uart){	
		case UART_0:
		    RET = RD0BUF; 			// read out the received RD0BUF	
		break;   
		case UART_1:
		    RET = RD1BUF; 			// read out the received RD1BUF
		break;   
		case UART_2:
		    RET = RD2BUF; 			// read out the received RD2BUF
		break;  
        default:
		break;
	}
	
    return RET; 	
}


//*********************************************************************
// name			: UART_Nibble2Ascii()
// brief		: nibble to ASCII
// param		: anibblea nibble
// retval		: data in ASCII format
// note			:           
//*********************************************************************
unsigned char UART_Nibble2Ascii(unsigned char anibble){

	unsigned char	ascii_out = anibble;

	if(anibble > 9){
		ascii_out = ascii_out + 0x07;
	}

	ascii_out = ascii_out + 0x30;

	return(ascii_out);
}


//*********************************************************************
// name			: UART_PutByte()
// brief		: get the high nibble converted to ASCII, then TX transfer
//            get the low nibble converted to ASCII, then TX transfer
// param		: uart: UART_0, UART_1, UART_2	 // UART selection
//            abyte: data to be transferred
// retval		:
// note			: 
//*********************************************************************
void UART_PutByte(unsigned char uart,unsigned char abyte){

	unsigned char  temp1 = 0, temp2 = 0;	

    temp1 = (abyte >> 4) & 0x0F;        // get the high nibble
    temp2 = UART_Nibble2Ascii(temp1);   // high nibble to ASCII code
    UART_TX_OneByte(uartSel,temp2);     // TX transfer

    temp1 = abyte & 0x0F;               // get the low nibble
    temp2 = UART_Nibble2Ascii(temp1);   // low nibble to ASCII code
    UART_TX_OneByte(uartSel,temp2);     // TX transfer
}



//*********************************************************************
// name			: UART_SendHex()
// brief		: output data in hexadecimal format
// param		: pHex: data buffer pointer 
//            Count: data buffer size
// retval		:
// note			: 
//*********************************************************************
void UART_SendHex(uint8_t *pHex, uint8_t Count){
    uint8_t i;
      
    for(i=0 ;i<Count ;i++) {
        UART_PutByte(UART_1,*pHex++);   // output using UART1
    }

    UART_PutCrlf(); 
}


//*********************************************************************
// name			: UART_PutDecNumber()
// brief		: output an integer in decimal format
// param		: value: 0~9999
// retval		:
// note			:
//*********************************************************************
void UART_PutDecNumber(unsigned int value){

	unsigned char print_buf[4];  
	 
    if(value > 9999){
       value = 9999;
    }
	
	print_buf[0] = ((value%10000)/1000)+'0';
	print_buf[1] = ((value%1000)/100)+'0';
	print_buf[2] = ((value%100)/10)+'0';
	print_buf[3] = ((value%10))+'0';
	 
	UART_Send(UART_1,print_buf,4);	   // output using UART1
}

//*********************************************************************
// name			: UART_PutCrlf()
// brief		: transmit line break characters
// param		:
// retval		:
// note			: 
//*********************************************************************
void UART_PutCrlf(void)
{

/*	
    UART_TX_OneByte(UART_0,'\r');
	UART_TX_OneByte(UART_0,'\n');
*/	

	UART_TX_OneByte(UART_1,'\r');
	UART_TX_OneByte(UART_1,'\n');

/*	
    UART_TX_OneByte(UART_2,'\r');
	UART_TX_OneByte(UART_2,'\n');
*/	
	
}

//*********************************************************************
// name			: UART_SendCString()
// brief		: send string
// param		: pstr: string buffer pointer
// retval		:
// note			:  
//*********************************************************************
void UART_SendCString(const char *pstr)
{

	while(*pstr != 0)
	{	
	    if(*pstr == '\n'){
		   	UART_PutCrlf();
		}
        else{		
			//UART_TX_OneByte(UART_0,*pstr);
		    UART_TX_OneByte(UART_1,*pstr);
		    //UART_TX_OneByte(UART_2,*pstr);
        }

		pstr++;
	}

}

//*********************************************************************
// name			: __interrupt IntRX0()
// brief		: RX0 interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntRX0(void)
{
	__asm("NOP");						// no operation
	

}


//*********************************************************************
// name			: __interrupt IntRX1()
// brief		: RX1 interrupt service routine
// note			: if received data matches transmitted data, light on corresponding LED
//*********************************************************************
void __interrupt IntRX1(void)
{
	__asm("NOP");						// no operation

	rbuf = UART1_RX_OneByte();          // readout received data
	if(rbuf == 0x55){					
		LED1_out = 0;					// if received data is 0x55, light on LED1(P2.0)
	}
	else if(rbuf == 0xAA){				
		LED2_out = 0;					// if received data is 0xAA,  light onLED2(P2.1)
	}
	else if(rbuf == 0x76){				
		LED3_out = 0;					// if received data is 0x76, light on LED3(P2.2)
	}		

}


//*********************************************************************
// name			: __interrupt IntRX2()
// brief		: RX2 interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntRX2(void)
{
	__asm("NOP");						// no operation

}



//*********************************************************************
// name			: __interrupt IntTX0()
// brief		: TX0 interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntTX0(void)
{
	__asm("NOP");						// no operation

	
}



//*********************************************************************
// name			: __interrupt IntTX1()
// brief		: TX1 interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntTX1(void)
{
	__asm("NOP");						// no operation

}



//*********************************************************************
// name			: __interrupt IntTX2()
// brief		: TX2 interrupt service routine
// note			:
//*********************************************************************
void __interrupt IntTX2(void)
{
	__asm("NOP");						// no operation

}


/* ---------------------- end of file -------------------------------- */
