//*************************************************************************************
// file			: main.c
// version		: V1.0 2022/05/30
// brief		: main program
// description	:
//    SecureUnit EncryptWrite / DecryptRead  example
//    
//    KEY(16 bytes): 000102030405060708090A0B0C0D0E0F
//    NonceSeed(12 bytes): 111111111111111111111111
//    Plaintext(16 bytes): 00000000000000000000000000000001
//
//    Example code flow:
//     1. execute the block write command to set the key
//     2. execute the Nonce command
//     3. execute the WriteCompute command to get the MAC and cipher text
//     4. execute the EncWrite command to write the cipher text to address 0x0000
//     5. execute the EncRead command to get the MAC and cipher text
//     6. execute the DecRead command to get the decrypted plaintext
//     7. compare the plaintext with the ciphertext
//
// note			: 
//            1. set I2C1 to Master mode
//		             set P2.1 as SDA1
//		             set P2.2 as SCL1
//	             *remember to add pull-high resistors
//
//            2. set P2.5 as UART1 TX
//                          set P2.6 as UART1 RX
//                          set Baud rate to 115200
// 
//			     [Button Configuration] button 1 :  BTN1(P3.0), (EINT0)
//
//				 [LED Configuration]  LED 1  (P2.0)
//**************************************************************************************

#ifndef	IO_MEM
#define	IO_MEM
#endif

#include "main.h"

I2C_MasterPacket i2c_master;
I2C_SlavePacket  i2c_slave;

EXAMPLE Example;

uint8_t key1[16];
uint8_t Buf_PT[16];
uint8_t Buf_CT[16];

SecureUnit_Command_Packet SecureUnit_CMD;
SecureUnit_Response_Packet SecureUnit_RESP;

const uint8_t Key[16] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};
const uint8_t PT[16] =  {0xf3,0x44,0x81,0xec,0x3c,0xc6,0x27,0xba,
                        0xcd,0x5d,0xc3,0xfb,0x08,0xf2,0x73,0xe6};
const uint8_t CT[16] =  {0x03,0x36,0x76,0x3e,0x96,0x6d,0x92,0x59,
                        0x5a,0x56,0x7c,0xc9,0xce,0x53,0x7f,0x5e};

uint8_t state[16];

const uint8_t PlainText[16]={0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01};
const uint8_t KEY[16]={0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
                       0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f};

const uint8_t NonceSeed[16]={0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,
                            0x11,0x11,0x11,0x11};

uint8_t FirstBlock[6]={0x00,0x00,0x00,0x00,0x10,0x03};
uint8_t SecondBlock[16]={0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};
uint8_t Mac[16];
uint8_t Ciphertext[16];

//*********************************************************************
// name			: Delay_us()
// brief		:
// param		: 
// retval		: 
// note			: 
//*********************************************************************
void Delay_us(uint16_t val){

    TCA0_SetPeriod(val*6);
    TCA0_Int_Flag = 0;
    TCA0_Start();
    while(!TCA0_Int_Flag);
    TCA0_Stop();
  
}

//*********************************************************************
// name			: Nonce()
// brief		:
// param		: 
// retval		: 
// note			: 
//*********************************************************************
uint8_t Nonce(uint8_t Mode, uint8_t *pInSeed, SecureUnit_Response_Packet *pResp){

    SecureUnit_Command_Packet cmd;	 
    //SecureUnit_Response_Packet resp;
    uint8_t i;

    cmd.Length = 0x15;
    cmd.Opcode = OpCode_Nonce;
    cmd.Mode = Mode;
    cmd.Param1 = 0x00;
    cmd.Param2 = 0x00;
    cmd.pDIN = pInSeed;
    if(SecureUnit_Command_Execute(&cmd,pResp)!=CMD_RET_OK){
        return false;
    }

	return true;        
}


//*********************************************************************
// name			: SetKey()
// brief		:
// param		: 
// retval		: 
// note			: 
//*********************************************************************
uint8_t SetKey(uint8_t KeyID, uint8_t *pKey, SecureUnit_Response_Packet *pResp){

    SecureUnit_Command_Packet cmd;

    cmd.Length = 0x19; 
    cmd.Opcode = OpCode_BlockWrite;	      // Block write command						  
    cmd.Mode = 0x00;
    cmd.Param1 = 0xF200+(KeyID * 0x10);   // Write Key 0
    cmd.Param2 = 0x0010; 
    cmd.pDIN = pKey;
    if(SecureUnit_Command_Execute(&cmd,pResp)!=CMD_RET_OK){
        return false;
    }

    return true;
}

//*********************************************************************
// name			: Auth_Outbound()
// brief		:
// param		: 
// retval		: 
// note			: 
//*********************************************************************
uint8_t Auth_Outbound(SecureUnit_Response_Packet *pResp){

    SecureUnit_Command_Packet cmd;

    cmd.Length = 0x09; 
    cmd.Opcode = OpCode_Auth;	      // Auth command						  
    cmd.Mode = 0x02;
    cmd.Param1 = 0x0000; 			  // Write Key 0
    cmd.Param2 = 0x0000; 
    //cmd.pDIN = pKey;
    
    if(SecureUnit_Command_Execute(&cmd,pResp)!=CMD_RET_OK){
        return false;
    }

	return true;
}


//*********************************************************************
// name			: WriteCompute()
// brief		:
// param		: 
// retval		: 
// note			: 
//*********************************************************************
uint8_t WriteCompute(uint8_t KeyID, uint8_t Count,
                    uint8_t *pInData, SecureUnit_Response_Packet *pResp){

    SecureUnit_Command_Packet cmd;
	uint8_t buf[6+16+16];
	//uint8_t i;

	cmd.Length = 0x2F; 
	cmd.Opcode = OpCode_WriteCompute;	      // WriteCompute command						  
	cmd.Mode = 0x00;
	cmd.Param1 = (uint16_t)KeyID; 			  // Write Key 0
    cmd.Param2 = (uint16_t)Count; 
	
	/// First Block 
    memcpy(&buf[0],FirstBlock,6);
	/// Second Block (All zero)
    memcpy(&buf[6],SecondBlock,16);
    /// InData
    memcpy(&buf[6+16],pInData,16);	
    cmd.pDIN = buf;
	
    if(SecureUnit_Command_Execute(&cmd,pResp)!=CMD_RET_OK){
        return false;
    }

	return true;
}

//*********************************************************************
// name			: EncWrite()
// brief		:
// param		: 
// retval		: 
// note			: 
//*********************************************************************
uint8_t EncWrite(uint8_t Mode, uint16_t Addr, uint8_t Count, uint8_t *pInMac, 
                uint8_t *pInData, SecureUnit_Response_Packet *pResp){

    SecureUnit_Command_Packet cmd;
    uint8_t buf[16+16];

	cmd.Length = 41; 
	cmd.Opcode = OpCode_EncWrite;	          // EncWrite command						  
	cmd.Mode = Mode;
	cmd.Param1 = Addr; 			     
	cmd.Param2 = (uint16_t)Count; 

    /// InMac
    memcpy(&buf[0],pInMac,16);
    /// InData
    memcpy(&buf[16],pInData,16);

	cmd.pDIN = buf;

    if(SecureUnit_Command_Execute(&cmd,pResp)!=CMD_RET_OK){
        return false;
    }

	return true;

}

//*********************************************************************
// name			: EncRead()
// brief		:
// param		: 
// retval		: 
// note			: 
//*********************************************************************
uint8_t EncRead(uint8_t Mode, uint16_t Addr, uint8_t Count,
                SecureUnit_Response_Packet *pResp){

    SecureUnit_Command_Packet cmd;

	cmd.Length = 0x09; 
	cmd.Opcode = OpCode_EncRead;	       					  
	cmd.Mode = Mode;
	cmd.Param1 = Addr; 			  
	cmd.Param2 = (uint16_t)Count; 

    if(SecureUnit_Command_Execute(&cmd,pResp)!=CMD_RET_OK){
        return false;
    }

	return true;

}

//*********************************************************************
// name			: DecRead()
// brief		:
// param		: 
// retval		: 
// note			: 
//*********************************************************************
uint8_t DecRead(uint8_t KeyID, uint8_t Count, uint8_t *pInMac, 
                uint8_t *pInData, SecureUnit_Response_Packet *pResp){

    SecureUnit_Command_Packet cmd;
	uint8_t buf[6+16+16+16];

	cmd.Length = 0x3F; 
	cmd.Opcode = OpCode_DecRead;	          // Block write command						  
	cmd.Mode = 0x00;
	cmd.Param1 = (uint16_t)KeyID; 			  // Write Key 0
	cmd.Param2 = (uint16_t)Count; 

	/// First Block 
    memcpy(&buf[0],FirstBlock,6);
	/// Second Block (All zero)
    memcpy(&buf[6],SecondBlock,16);
	/// InMac
	memcpy(&buf[6+16],pInMac,16);
    /// InData
    memcpy(&buf[6+16+16],pInData,16);	
    cmd.pDIN = buf;
	
    if(SecureUnit_Command_Execute(&cmd,pResp)!=CMD_RET_OK){
        return false;
    }

	return true;

}

//*********************************************************************
// name			: main()
// brief		: main program
// param		: 
// retval		: 
// note			: 
//*********************************************************************
void main(){
/* ---------------------- system initialization ------------------------ */
	PLLCR0_PLLEN  = 1;                  // enable PLL
	while(PLLCR0_PLLRDY ==0);			// wait for PLL to be ready
	CLKCR0_SCKPSC = 1;                  // system frequency divided by 2   (48M/2=24MHz) 
    CLKCR0_SCKSRC = 1;                  // switch clock source to PLL
	while(CLKCR0_SCKRDY ==0);			// wait for system clock to be ready
/* ---------------------- pin configuration initialization ------------------------ */
    P0PU = 0xFF;                        // enable all built-in pull-up resistors
    P1PU = 0xFF;
    P2PU = 0xFF;
    P3PU = 0xFF;
    P4PU = 0xFF;
    P5PU = 0xFF;

    P2OE_P0 = 1;                        // set P2.0 as output pin
    P2PU_P0 = 0;                        // disable P2.0 built-in pull-up resistor

    P3OE_P7 = 1;                        // set P3.7 as output pin
    P3PU_P7 = 0;                        // disable P3.7 built-in pull-up resistor

    P3OE_P6 = 1;                        // set P3.6 as output pin
    P3PU_P6 = 0;                        // disable P3.6 built-in pull-up resistor
	
    LED1_out = 1;                       // set LED1 to the light off
    LED2_out = 1;                       // set LED2 to the light off
    LED3_out = 1;                       // set LED3 to the light off

/* ---------------------- EINT initialization ------------------------ */
	EINT_Init();                        // initialize EINT
	EINT_IntEnable();                   // enable EINT interrupt
/* ---------------------- I2C1 initialization ------------------------ */
    I2C1_Init();                        // set I2C1 as Master and disable I2C1 interrupt 
/* ---------------------- UART1 initialization ------------------------ */
    UART1_Init();
    UART1_SetBaud rate(SYSCLK_24MHZ_BAUD_115200);
	UART1_Start();
/* ---------------------- TCA0 initialization ------------------------ */    
    TCA0_Timer_Init();
    TCA0_IntEnable();	

#ifdef HW_CRC
/* ---------------------- CRC initialization ------------------------ */
    CRC_Init();
#endif // HW_CRC
///-------------------------------------------------------------    

    UART_SendCString("SecureUnit API Example: EncWrite/DecRead \n");


    //// Step 1:Set Key @ 0xF200 (Key 0)
    if(SetKey(0x00,(uint8_t*)KEY,&SecureUnit_RESP))
    {
        if(SecureUnit_RESP.CMDST==0){
            UART_SendCString("\nSet key success!\n");
        }
        else{
            UART_SendCString("\nSet key fail!\n");
        }
    }
    else{
       UART_SendCString("Set key fail!\n");
    }

    //// Step2: Execute Nonce command
	if(Nonce(0x01,(uint8_t*)NonceSeed,&SecureUnit_RESP)){
        if(SecureUnit_RESP.CMDST==0){
            UART_SendCString("Nonce command execute success!\n");
		}
		else{
            UART_SendCString("Nonce command execute fail!\n");
        }
	}
	else{
		UART_SendCString("Nonce command execute fail!\n");
    }
	
    //// Step 3: Execute WriteCompute command to get MAC and ciphertext 
	if(WriteCompute(0x00, 0x10, (uint8_t*)PlainText, &SecureUnit_RESP)){
        // SecureUnit serial command excute OK
		if(SecureUnit_RESP.CMDST == 0){
            UART_SendCString("\nWriteCompute command execute success!\n");
            /// Out MAC
            memcpy(&Mac[0],&SecureUnit_RESP.DOUT[0],16);
            UART_SendCString("MAC:"); UART_SendHex(Mac,16);
            /// Ciphertext
            memcpy(&Ciphertext[0],&SecureUnit_RESP.DOUT[16],16);
            UART_SendCString("CipherText:"); UART_SendHex(Ciphertext,16);
		}
		else{
            UART_SendCString("\nWriteCompute command execute fail!\n");
        }
    }
	else{
        UART_SendCString("\nWriteCompute command execute fail!\n");
    }

      
    /// Execute nonce command to reset mac count
	if(Nonce(0x01, (uint8_t*)NonceSeed, &SecureUnit_RESP)){
        UART_SendCString("Nonce command execute success!\n");
    }
	else{
        UART_SendCString("Nonce command execute fail!\n");
    }

    ///Step 4: Execute EncWrite command to write chipertext into address 0x0000
    if(EncWrite(0x00,0x0000,0x10,(uint8_t*)Mac,(uint8_t*)Ciphertext,&SecureUnit_RESP)){
        if(SecureUnit_RESP.CMDST==0){
            UART_SendCString("\nEncWrite command success!\n"); 
        }
		else{
            UART_SendCString("\nEncWrite command fail!\n");
        }
    }
	else{
        UART_SendCString("\nEncWrite command fail!\n");
    }


    /// Execute nonce command to reset mac count
	if(Nonce(0x01,(uint8_t*)NonceSeed,&SecureUnit_RESP)){
        __NOP();
	}
    else{
        __NOP();
    }


    ///Step 5: Execute EncRead command to read out MAC and ciphertext
    if(EncRead(0x00,0x00,0x10,&SecureUnit_RESP)){
	    __NOP();
	    if(SecureUnit_RESP.CMDST == 0){
            UART_SendCString("\nEncRead command success!\n");
            /// Out MAC
            memcpy(&Mac[0],&SecureUnit_RESP.DOUT[0],16);
            UART_SendCString("MAC: "); UART_SendHex(Mac,16);
            /// Ciphertext
            memcpy(&Ciphertext[0],&SecureUnit_RESP.DOUT[16],16);
            UART_SendCString("Ciphertext: "); UART_SendHex(Ciphertext,16);
        }
        else{
            UART_SendCString("\nEncRead command fail!\n");
        }	
    }
	else{
        UART_SendCString("\nEncRead command fail!\n");
    }


    /// Execute nonce command to reset mac count
	if(Nonce(0x01,(uint8_t*)NonceSeed,&SecureUnit_RESP)){
        __NOP();
	}
    else{
        __NOP();
    }

    /// Step 6: Execute DecRead command to get decrypted plaintext
	if(DecRead(0x00,0x10,(uint8_t*)Mac,(uint8_t*)Ciphertext,&SecureUnit_RESP)){
        if(SecureUnit_RESP.CMDST==0){
            UART_SendCString("\nDecRead command success!\n");
			
			////Step 7: Compare the decrypted plaintext and original plaintext 
            if(memcmp(PlainText,SecureUnit_RESP.DOUT,16)==0){
                UART_SendCString("Plaintext decrypt success!\n");  
                UART_SendCString("Plaintext: "); UART_SendHex(SecureUnit_RESP.DOUT,16);
            }
			else{
                UART_SendCString("Plaintext decrypt fail!!\n");
            }
        }
		else{
            UART_SendCString("\nDecRead command success!\n");
        }
	}
	else{
        UART_SendCString("\nDecRead command fail!\n");
    }

	while(1);

}
/* ---------------------- interrupt service routine ------------------------ */
void __interrupt OnlyReti(void){
    __asm("NOP");                       // no operation
}

void __interrupt_n OnlyRetn(void){
    __asm("NOP");                       // no operation
}

void __interrupt_n IntWDT(void){
    __asm("NOP");                       // no operation
}

void __interrupt_n IntSWI(void){        // software reset interrupt service routine
    PRSTR7 = 0x5a;
    PRSTR7 = 0xa5;
    PRSTR7 = 0xc3;
    PRSTR7 = 0x3c;	
}
/* ------------------- end of interrupt service routine ---------------------- */


/* -------------------- SQ7515 interrupt vector ------------------ */
#pragma section const INT_VECTOR1 0xff40
void * const IntTbl1[] = {											
	OnlyReti,				/* 0xff40 : Reserved		*/									
	OnlyReti,				/* 0xff42 : Reserved		*/									
	OnlyReti,				/* 0xff44 : Reserved		*/									
	OnlyReti,				/* 0xff46 : Reserved		*/								
	OnlyReti,				/* 0xff48 : Reserved		*/									
	OnlyReti,				/* 0xff4a : Reserved		*/									
	OnlyReti,				/* 0xff4c : Reserved		*/									
	OnlyReti,				/* 0xff4e : Reserved		*/									
	OnlyReti,				/* 0xff50 : Reserved		*/									
	OnlyReti,				/* 0xff52 : Reserved		*/									
	OnlyReti,				/* 0xff54 : Reserved		*/									
	OnlyReti,				/* 0xff56 : Reserved		*/									
	OnlyReti,				/* 0xff58 : IntTCA7			*/	//interrupt source:TCA7 16-bit timer								
	OnlyReti,				/* 0xff5a : IntTCA6			*/	//interrupt source:TCA6 16-bit timer							
	OnlyReti,				/* 0xff5c : Reserved		*/									
	OnlyReti,				/* 0xff5e : Reserved		*/									
	OnlyReti,				/* 0xff60 : Reserved		*/									
	OnlyReti,				/* 0xff62 : Reserved		*/									
	OnlyReti,				/* 0xff64 : Reserved		*/									
	OnlyReti,				/* 0xff66 : Reserved		*/									
	OnlyReti,				/* 0xff68 : Reserved		*/									
	OnlyReti,				/* 0xff6a : Reserved		*/									
	OnlyReti,				/* 0xff6c : IntTX2			*/	//interrupt source:UART2 TX2								
	OnlyReti,				/* 0xff6e : IntRX2			*/	//interrupt source:UART2 RX2							
	OnlyReti,				/* 0xff70 : Reserved		*/									
	OnlyReti,				/* 0xff72 : Reserved		*/									
	OnlyReti,				/* 0xff74 : Reserved		*/									
	OnlyReti,				/* 0xff76 : IntTCA5			*/	//interrupt source:TCA5 16-bit timer								
	OnlyReti,				/* 0xff78 : IntTCA4			*/	//interrupt source:TCA4 16-bit timer								
	OnlyReti,				/* 0xff7a : Reserved		*/									
	OnlyReti,				/* 0xff7c : Reserved		*/									
	OnlyReti,				/* 0xff7e : Reserved		*/									
	OnlyReti,				/* 0xff80 : Reserved		*/									
	OnlyReti,		    	/* 0xff82 : Reserved		*/									
	OnlyReti,	    		/* 0xff84 : Reserved		*/									
	OnlyReti,				/* 0xff86 : Reserved		*/									
	OnlyReti,				/* 0xff88 : Reserved		*/									
	OnlyReti,				/* 0xff8a : Reserved		*/									
	OnlyReti,				/* 0xff8c : IntSIO1			*/	//interrupt source:SIO 1								
	OnlyReti,				/* 0xff8e : IntI2C1			*/	//interrupt source:I2C1						
	OnlyReti,				/* 0xff90 : IntTX1			*/	//interrupt source:UART1 TX1							
	OnlyReti,				/* 0xff92 : IntRX1			*/	//interrupt source:UART1 RX1								
	OnlyReti,				/* 0xff94 : Reserved		*/									
	OnlyReti,				/* 0xff96 : Reserved		*/									
	OnlyReti,				/* 0xff98 : Reserved		*/									
	OnlyReti,				/* 0xff9a : IntTCA3			*/	//interrupt source:TCA3 16-bit timer							
	OnlyReti,				/* 0xff9c : IntTCA2			*/	//interrupt source:TCA2 16-bit timer								
	OnlyReti,				/* 0xff9e : Reserved		*/									
	OnlyReti,				/* 0xffa0 : IntMAC			*/	//interrupt source:multiplier								
	OnlyReti,				/* 0xffa2 : IntEXT7			*/	//external interrupt7								
	OnlyReti,				/* 0xffa4 : IntEXT6			*/	//external interrupt6								
	OnlyReti,				/* 0xffa6 : IntEXT5			*/	//external interrupt5								
	OnlyReti,				/* 0xffa8 : IntEXT4			*/	//external interrupt4								
	OnlyReti,				/* 0xffaa : Reserved		*/									
	OnlyReti,				/* 0xffac : Reserved		*/									
	OnlyReti,				/* 0xffae : Reserved		*/							
	OnlyReti,				/* 0xffb0 : IntDIC			*/	//interrupt source:DIC								
	OnlyReti,				/* 0xffb2 : Reserved		*/							
	OnlyReti,				/* 0xffb4 : Reserved		*/									
	OnlyReti,				/* 0xffb6 : Reserved		*/									
	OnlyReti,				/* 0xffb8 : Reserved		*/									
	OnlyReti,				/* 0xffba : Reserved		*/									
	OnlyReti,				/* 0xffbc : Reserved		*/									
	OnlyReti,				/* 0xffbe : Reserved		*/									
	OnlyReti,				/* 0xffc0 : Reserved		*/									
	OnlyReti,				/* 0xffc2 : Reserved		*/									
	OnlyReti,				/* 0xffc4 : IntFMC			*/	//interrupt source:flash memory controller								
	OnlyReti,				/* 0xffc6 : Reserved		*/					
	OnlyReti,				/* 0xffc8 : Reserved		*/									
	OnlyReti,				/* 0xffca : Reserved		*/									
	OnlyReti,				/* 0xffcc : Reserved		*/									
	OnlyReti,               /* 0xffce : IntADC			*/	//interrupt source:ADC								
	OnlyReti,				/* 0xffd0 : IntEXT3			*/	//external interrupt3								
	IntEXT2,				/* 0xffd2 : IntEXT2			*/	//external interrupt2								
	IntEXT1,				/* 0xffd4 : IntEXT1			*/	//external interrupt1								
	IntEXT0,				/* 0xffd6 : IntEXT0			*/	//external interrupt0								
	OnlyReti,				/* 0xffd8 : IntSIO0			*/	//interrupt source:SIO 0								
	OnlyReti,				/* 0xffda : IntI2C0			*/	//interrupt source:I2C0								
	OnlyReti,				/* 0xffdc : IntTX0			*/	//interrupt source:UART0 TX0								
	OnlyReti,				/* 0xffde : IntRX0			*/	//interrupt source:UART0 RX0								
	OnlyReti,				/* 0xffe0 : Reserved		*/									
	OnlyReti,				/* 0xffe2 : Reserved		*/									
	OnlyReti,				/* 0xffe4 : Reserved		*/									
	OnlyReti,				/* 0xffe6 : IntTCA1			*/	//interrupt source:TCA1 16-bit timer								
	IntTCA0,				/* 0xffe8 : IntTCA0			*/	//interrupt source:TCA0 16-bit timer								
	OnlyReti,				/* 0xffea : Reserved		*/									
	OnlyReti,				/* 0xffec : Reserved		*/									
	OnlyReti,			    /* 0xffee : IntRTC			*/	//interrupt source:real-time clock timer								
	OnlyReti,				/* 0xfff0 : IntTBT			*/  //interrupt source:time base timer				
	OnlyReti,				/* 0xfff2 : Reserved		*/									
	OnlyReti,				/* 0xfff4 : IntCFD			*/	//interrupt source:clock fail detection								
	OnlyReti,				/* 0xfff6 : IntLVD			*/	//interrupt source:low voltage detection							
	IntWDT,					/* 0xfff8 : IntWDT			*/	//non-maskable interrupt source:watchdog timer		
	(void *)0xffff,			/* 0xfffa : Reserved		*/									
	IntSWI,					/* 0xfffc : IntSWI/INTUNDEF */	//non-maskable		
	STARTUP					/* 0xfffe : RESET			*/	//non-maskable			
};
#pragma section const
/* ------------------- end of SQ7515 interrupt vector ---------------------- */

